import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { S3Client, PutObjectCommand, GetObjectCommand } from '@aws-sdk/client-s3';
import { getSignedUrl } from '@aws-sdk/s3-request-presigner';
import { isProd } from '../config/config';
import { AppLoggerService } from './logger.service';
import { handleKnownErrors } from '../utils/handle-error.util';
import { ERROR_CODES } from '../constants/error-string-constants';

@Injectable()
export class AwsS3Service {
  private s3Client: S3Client;

  constructor(
    private configService: ConfigService,
    private readonly logger: AppLoggerService,
  ) {
    const isProdvar = isProd(this.configService);
    const s3Config: any = {
      region: this.configService.get<string>('AWS_REGION') || '',
    };
    if (!isProdvar) {
      s3Config.credentials = {
        accessKeyId: this.configService.get<string>('AWS_ACCESS_KEY_ID') || '',
        secretAccessKey: this.configService.get<string>('AWS_SECRET_ACCESS_KEY') || '',
      };
    }
    this.s3Client = new S3Client(s3Config);
  }

  /**
   * Upload data to S3
   * @param key - S3 key
   * @param data - data to upload
   * @param contentType - content type
   * @returns S3 upload result
   */
  async uploadToS3(key: string, data: Buffer | string, contentType: string) {
    const params = {
      Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
      Key: key,
      Body: Buffer.isBuffer(data) ? data : Buffer.from(data, 'base64'),
      ContentType: contentType,
    };
    
    try {
      const command = new PutObjectCommand(params);
      const result = await this.s3Client.send(command);
      return this.getS3Url(key); // Return the public URL of the uploaded file
    } catch (error) {
      this.logger.error('Error uploading to S3', '', { key, error });
      handleKnownErrors(ERROR_CODES.FILE_UPLOAD_FAILED, error);
    }
  }

  /**
   * Generate pre-signed URL for file upload to S3
   * @param fileName - Name of the file to be uploaded
   * @param contentType - Content type of the file
   * @returns Object containing upload URL and access URL
   */
  async generateUploadUrl(
    fileName: string,
    contentType: string,
    prefix = 'assets',
    userLevelDetails = false,
    defaultContent = false
  ): Promise<{ putUrl: string; accessUrl: string }> {
    try {
      const timestamp = Date.now();
      let key = `${prefix}`;
      if (!userLevelDetails) {
        if (defaultContent) {
         key += `/${timestamp}_${fileName}`;
        } else {
        key += `_${timestamp}_${fileName}`;
        }
      } else {
        if (defaultContent) {
          key += `/${timestamp}_${fileName}`;
        }
      }
      
      const params = {
        Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
        Key: key,
        ContentType: contentType,
      };

      const command = new PutObjectCommand(params);
      const putUrl = await getSignedUrl(this.s3Client, command, { expiresIn: 600 });
      const accessUrl = `https://${this.configService.get('AWS_S3_BUCKET_NAME')}.s3.${this.configService.get('AWS_REGION')}.amazonaws.com/${key}`;

      return { putUrl, accessUrl };
    } catch (error) {
      this.logger.error('Error generating presigned URL for S3', '', { prefix, fileName, error });
      handleKnownErrors(ERROR_CODES.FILE_UPLOAD_FAILED, error);
    }
  }

  /**
   * Generate a signed URL for accessing an S3 object
   * @param key - S3 object key
   * @param expiresIn - URL expiration time in seconds (default: 3600)
   * @returns Promise<string> Signed URL for accessing the S3 object
   */
  async getSignedUrl(key: string, expiresIn: number = 3600): Promise<string> {
    const command = new GetObjectCommand({
      Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
      Key: key,
    });

    try {
      return await getSignedUrl(this.s3Client, command, { expiresIn });
    } catch (error) {
      this.logger.error('Error generating signed URL for S3', '', { key, error });
      handleKnownErrors(ERROR_CODES.FAILED_TO_GENERATE_FILE_URLS, error);
    }
  }

  /**
   * Generate a long-lived signed URL for accessing an S3 object (for reports and permanent access)
   * Uses maximum AWS-allowed expiration of 7 days (604,800 seconds)
   * Note: AWS S3 signed URLs cannot be truly permanent; maximum expiration is 7 days
   * @param url - Full S3 URL or just the key
   * @returns Promise<string> Signed URL for accessing the S3 object (valid for 7 days)
   */
  async generateLongLivedSignedUrl(url: string): Promise<string> {
    try {
      // Extract key from full URL if needed
      const key = this.extractS3KeyFromUrl(url) || '';

      if (!key) {
        this.logger.error('Invalid S3 URL provided for long-lived signed URL generation', '', { url });
        return url; // Return original URL if key extraction fails
      }

      const command = new GetObjectCommand({
        Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
        Key: key,
      });

      // AWS S3 maximum expiration: 7 days (604,800 seconds)
      // Note: Attempting to use longer expiration will result in an error
      return await getSignedUrl(this.s3Client, command, { expiresIn: 604800 });
    } catch (error) {
      this.logger.error('Error generating long-lived signed URL for S3', '', { url, error });
      // Return original URL if signed URL generation fails
      return url;
    }
  }

  /**
   * Generate a public URL for accessing an S3 object
   * @param key - S3 object key
   * @returns string Public URL for accessing the S3 object
   */
  getS3Url(key: string): string {
    const bucketName =  process.env.AWS_S3_BUCKET_NAME
    const region = process.env.AWS_REGION
    return `https://${bucketName}.s3.${region}.amazonaws.com/${key}`;
  }

  private extractS3KeyFromUrl(s3Url: string): string | null {
    if (!s3Url) return null;
    
    try {
      const url = new URL(s3Url);
      const hostname = url.hostname;
      const pathname = url.pathname;
      
      // Check if it's the bucket.s3.region.amazonaws.com format
      if (hostname.includes('.s3.') && hostname.includes('.amazonaws.com')) {
        // Remove leading slash and decode URI component to handle special characters properly
        // This prevents double-encoding issues (e.g., %20 becoming %2520)
        return decodeURIComponent(pathname.substring(1));
      }
      
      // Check if it's the s3.region.amazonaws.com/bucket format
      if (hostname.includes('s3.') && hostname.includes('.amazonaws.com') && !hostname.startsWith('s3.')) {
        const pathParts = pathname.split('/');
        if (pathParts.length > 2) {
          // Remove empty first element and bucket name, join the rest
          // Decode URI component to handle special characters properly
          return decodeURIComponent(pathParts.slice(2).join('/'));
        }
      }

      this.logger.log(`Unknown S3 URL format: ${s3Url}`);
      return null;
    } catch (error) {
      this.logger.error(`Error parsing S3 URL: ${error}`);
      return null;
    }
  }
}