import { Response } from 'express';
import { HttpStatus } from '@nestjs/common';


export class ErrorHandler {
  /**
   * Sends a server error response.
   * @param res - Express Response object.
   * @param error - Optional error object to include the error message.
   */
  async serverError(res: Response, error?: Error): Promise<void> {
    const response = {
      statusCode: HttpStatus.INTERNAL_SERVER_ERROR,
      error: 'Internal Server Error',
      message: error?.message || 'An unexpected error occurred',
    };
    res.status(HttpStatus.INTERNAL_SERVER_ERROR).json(response);
  }

  /**
   * Sends a forbidden response.
   * @param res - Express Response object.
   * @param message - Message to include in the response.
   */
  async forbidden(res: Response, message: string): Promise<void> {
    res.status(HttpStatus.FORBIDDEN).json({
      statusCode: HttpStatus.FORBIDDEN,
      error: 'Forbidden',
      message,
    });
  }

  /**
   * Sends a bad request response.
   * @param res - Express Response object.
   * @param message - Message to include in the response.
   */
  async badRequest(res: Response, message: string): Promise<void> {
    res.status(HttpStatus.BAD_REQUEST).json({
      statusCode: HttpStatus.BAD_REQUEST,
      error: 'Bad Request',
      message,
    });
  }

  /**
   * Sends a not found response.
   * @param res - Express Response object.
   * @param message - Optional message to include in the response.
   */
  async notFound(res: Response, message?: string): Promise<void> {
    res.status(HttpStatus.NOT_FOUND).json({
      statusCode: HttpStatus.NOT_FOUND,
      error: 'Not Found',
      message: message || 'Resource not found',
    });
  }

  /**
   * Sends an unauthorized response.
   * @param res - Express Response object.
   */
  async unauthorized(res: Response): Promise<void> {
    res.status(HttpStatus.UNAUTHORIZED).json({
      statusCode: HttpStatus.UNAUTHORIZED,
      error: 'Unauthorized',
      message: 'Unauthorized access',
    });
  }

  /**
   * Sends a file download response.
   * @param res - Express Response object.
   * @param data - File data to send.
   * @param fileName - Name of the file to be downloaded.
   * @param contentType - MIME type of the file.
   * @param length - Length of the file in bytes.
   */
  async download(
    res: Response,
    data: any,
    fileName: string,
    contentType: string,
    length: number,
  ): Promise<void> {
    res.setHeader('Content-Disposition', `attachment; filename="${fileName}"`);
    res.setHeader('Content-Length', length.toString());
    res.setHeader('Content-Type', contentType);
    res.send(data);
  }
}

export default ErrorHandler;
