import {
  Injectable,
  CanActivate,
  ExecutionContext,
  ForbiddenException,
} from "@nestjs/common";
import { Reflector } from "@nestjs/core";
import { ROLE_GUARD_STRINGS } from "../constants/strings-constants";

@Injectable()
export class RolesGuard implements CanActivate {
  constructor(private readonly reflector: Reflector) {}

  /**
   * Determines if the user has the required roles to access the route.
   * @param context - The execution context of the request.
   * @returns {boolean} - True if the user has the required roles, false otherwise.
   */
  canActivate(context: ExecutionContext): boolean {
    // Check for roles at the method (handler) level first
    const methodRoles = this.reflector.get<string[]>(
      "roles",
      context.getHandler()
    );
    if (methodRoles) {
      return this.checkRoles(context, methodRoles);
    }

    // If no roles are defined at the method level, check at the controller level
    const controllerRoles = this.reflector.get<string[]>(
      "roles",
      context.getClass()
    );
    if (controllerRoles) {
      return this.checkRoles(context, controllerRoles);
    }

    // If no roles are defined at either level, allow access
    return true;
  }
  /**
   * Checks if the user has at least one of the required roles.
   * @param context - The execution context of the request.
   * @param requiredRoles - The roles required to access the route.
   * @returns {boolean} - True if the user has the required roles, false otherwise.
   */
  private checkRoles(
    context: ExecutionContext,
    requiredRoles: string[]
  ): boolean {
    const request = context.switchToHttp().getRequest();
    const user = request.user;

    if (!user || !user.roles) {
      throw new ForbiddenException(ROLE_GUARD_STRINGS.MISSING_ROLES);
    }

    // Check if the user has at least one of the required roles
    const hasRole = requiredRoles.some((role) => user.roles.includes(role));
    if (!hasRole) {
      throw new ForbiddenException(ROLE_GUARD_STRINGS.UNAUTHORIZED);
    }

    return true;
  }
}
