import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
  Index,
} from 'typeorm';
import { User } from './user.entity';
import { UserSessionStatusEnum } from '../enum/user-session-status.enum';

@Entity('user_session')
@Index('idx_user_session_user_id', ['userId'], { where: '"user_id" IS NOT NULL' })
@Index('idx_user_session_phone_number', ['countryCode', 'phoneNumber'], {
  where: '"phone_number" IS NOT NULL AND "country_code" IS NOT NULL',
})
@Index('idx_user_session_email', ['email'], { where: '"email" IS NOT NULL' })
@Index('idx_user_session_jwt_token', ['jwtToken'], {
  unique: true,
  where: '"jwt_token" IS NOT NULL',
})
export class UserSession {
  @PrimaryGeneratedColumn({ type: 'bigint' })
  id: number;

  @Column({ name: 'user_id', type: 'bigint', nullable: true })
  userId: number;

  @ManyToOne(() => User, { onDelete: 'CASCADE', nullable: true })
  @JoinColumn({ name: 'user_id' })
  user: User;

  @Column({ name: 'phone_number', type: 'varchar', length: 255, nullable: true })
  phoneNumber: string;

  @Column({ name: 'country_code', type: 'varchar', length: 5, nullable: true })
  countryCode: string;

  @Column({ type: 'varchar', length: 320, nullable: true })
  email: string;

  @CreateDateColumn({ name: 'login_at', type: 'timestamptz' })
  loginAt: Date;

  @Column({ type: 'varchar', length: 6, nullable: true })
  otp: string;

  @Column({ name: 'attempt_count', type: 'smallint', default: 0 })
  attemptCount: number;

  @Column({ name: 'jwt_token', type: 'text', nullable: true })
  jwtToken: string | null;

  @Column({
    name: 'session_status',
    type: 'enum',
    enum: UserSessionStatusEnum,
    default: UserSessionStatusEnum.INITIATED,
  })
  sessionStatus: UserSessionStatusEnum;

  @Column({ name: 'user_agent', type: 'varchar', length: 1024, nullable: true })
  userAgent: string;

  @Column({ name: 'ip_address', type: 'varchar', length: 45, nullable: true })
  ipAddress: string;

  @Column({ name: 'expires_at', type: 'timestamptz', nullable: true })
  expiresAt: Date | null;

  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz' })
  updatedAt: Date;

  constructor(partial: Partial<UserSession>) {
    Object.assign(this, partial);
  }
}
