import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
} from 'typeorm';
import { ApprovalStatusEnum } from '../enum/approval-status.enum';
import { ApprovalTrackTypeEnum } from '../enum/approval-track-type.enum';
import {
  ProgramRegistration,
  RegistrationApproval,
  User,
  ProgramRegistrationSwap,
  Program,
  ProgramSession,
} from './index';

@Entity('hdb_registration_approval_track')
export class RegistrationApprovalTrack {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ name: 'approval_id', type: 'bigint' })
  approvalId: number;

  @Column({ name: 'registration_id', type: 'bigint' })
  registrationId: number;

  @Column({
    name: 'approval_status',
    type: 'enum',
    enum: ApprovalStatusEnum,
    default: ApprovalStatusEnum.PENDING,
  })
  approvalStatus: ApprovalStatusEnum;

  @Column({ name: 'approval_date', type: 'timestamptz', nullable: true })
  approvalDate: Date | null;

  @Column({ name: 'approved_by', type: 'bigint', nullable: true })
  approvedBy: number | null;

  @Column({ name: 'rejection_reason', type: 'text', nullable: true })
  rejectionReason: string | null;

  @Column({ name: 'reviewer_comments', type: 'text', nullable: true })
  reviewerComments: string | null;

  @Column({ name: 'auto_approved', type: 'boolean', default: false })
  autoApproved: boolean;

  @Column({
    name: 'type',
    type: 'enum',
    enum: ApprovalTrackTypeEnum,
    default: ApprovalTrackTypeEnum.REGISTRATION,
  })
  type: ApprovalTrackTypeEnum;

  @Column({ name: 'swap_request_id', type: 'bigint', nullable: true })
  swapRequestId: number | null;


  @Column({ name: 'allocated_program_id', type: 'bigint', nullable: true })
  allocatedProgramId?: number | null;

  @Column({ name: 'allocated_session_id', type: 'bigint', nullable: true })
  allocatedSessionId?: number | null;

  @CreateDateColumn({
    name: 'created_at',
    type: 'timestamptz',
    default: () => 'CURRENT_TIMESTAMP',
  })
  createdAt: Date;

  @UpdateDateColumn({
    name: 'updated_at',
    type: 'timestamptz',
    default: () => 'CURRENT_TIMESTAMP',
    onUpdate: 'CURRENT_TIMESTAMP',
  })
  updatedAt: Date;

  @Column({ name: 'created_by', type: 'bigint', nullable: true })
  createdBy: number | null;

  @Column({ name: 'updated_by', type: 'bigint', nullable: true })
  updatedBy: number | null;

  @ManyToOne(() => RegistrationApproval, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'approval_id' })
  approval: RegistrationApproval;

  @ManyToOne(() => ProgramRegistration, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'registration_id' })
  registration: ProgramRegistration;

  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'approved_by' })
  approvedByUser: User | null;

  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'created_by' })
  createdByUser: User | null;

  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'updated_by' })
  updatedByUser: User | null;

  @ManyToOne(() => ProgramRegistrationSwap, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'swap_request_id' })
  swapRequest?: ProgramRegistrationSwap | null;

  @ManyToOne(() => Program, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'allocated_program_id' })
  allocatedProgram?: Program | null;

  @ManyToOne(() => ProgramSession, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'allocated_session_id' })
  allocatedSession?: ProgramSession | null;

  constructor(partial: Partial<RegistrationApprovalTrack>) {
    Object.assign(this, partial);
  }
}