import { Entity, PrimaryGeneratedColumn, Column, CreateDateColumn, UpdateDateColumn, OneToMany, ManyToOne, JoinColumn, DeleteDateColumn } from 'typeorm';
import { Program } from './program.entity';
import { User } from './user.entity';
import { ModeOfOperationEnum, OnlineTypeEnum, FrequencyEnum, RegistrationLevelEnum } from '../enum/program.enums';
import { TdsApplicabilityEnum } from '../enum/tds-applicability.enum';

/**
 * Entity representing a program type in version 2
 * Contains all the configuration and settings for different types of programs
 */
@Entity('program_type_v1')
export class ProgramType {
  @PrimaryGeneratedColumn()
  id: number;

  @Column({ name: 'name', type: 'varchar', length: 255, nullable: false })
  name: string;

  @Column({ name: 'description', type: 'text', nullable: true })
  description: string;

  @Column({ name: 'mode_of_operation', type: 'enum', enum: ModeOfOperationEnum, nullable: false })
  modeOfOperation: ModeOfOperationEnum;

  @Column({ name: 'online_type', type: 'enum', enum: OnlineTypeEnum, default: OnlineTypeEnum.NA })
  onlineType: OnlineTypeEnum;

  @Column({ name: 'max_session_duration_days', type: 'int', default: 8 })
  maxSessionDurationDays: number;

  @Column({ name: 'has_multiple_sessions', type: 'boolean', default: true })
  hasMultipleSessions: boolean;

  @Column({ name: 'frequency', type: 'enum', enum: FrequencyEnum, default: FrequencyEnum.YEARLY })
  frequency: FrequencyEnum;

  @Column({ name: 'default_start_time', type: 'time', nullable: true })
  defaultStartTime: string;

  @Column({ name: 'default_end_time', type: 'time', nullable: true })
  defaultEndTime: string;

  @Column({ name: 'default_duration', type: 'varchar', length: 50, nullable: true })
  defaultDuration: string;

  @Column({ name: 'requires_residence', type: 'boolean', default: false })
  requiresResidence: boolean;

  @Column({ name: 'involves_travel', type: 'boolean', default: false })
  involvesTravel: boolean;

  @Column({ name: 'has_checkin_checkout', type: 'boolean', default: false })
  hasCheckinCheckout: boolean;

  @Column({ name: 'requires_payment', type: 'boolean', default: true })
  requiresPayment: boolean;

  @Column({ name: 'requires_attendance_all_sessions', type: 'boolean', default: true })
  requiresAttendanceAllSessions: boolean;

  @Column({ name: 'allows_minors', type: 'boolean', default: false })
  allowsMinors: boolean;

  @Column({ name: 'allows_proxy_registration', type: 'boolean', default: false })
  allowsProxyRegistration: boolean;

  @Column({ name: 'requires_approval', type: 'boolean', default: false })
  requiresApproval: boolean;

  @Column({ name: 'registration_level', type: 'enum', enum: RegistrationLevelEnum, default: RegistrationLevelEnum.SESSION })
  registrationLevel: RegistrationLevelEnum;

  @Column({ name: 'is_active', type: 'boolean', default: true })
  isActive: boolean;

  @Column({ name: 'no_of_session', type: 'int', default: 0, nullable: false })
  noOfSession: number;

  @Column({ name: 'waitlist_applicable', type: 'boolean', default: false })
  waitlistApplicable: boolean;

  @Column({ name: 'max_capacity', type: 'int', default: 0 })
  maxCapacity: number;

  @Column({ name: 'meta', type: 'json', nullable: true, default: () => "'{}'" })
  meta: Record<string, any>;

  @Column({ name: 'venue', type: 'text', nullable: true })
  venue: string;

  @Column({ name: 'limited_seats', type: 'boolean', default: false })
  limitedSeats: boolean;

  @Column({ name: 'is_grouped_program', type: 'boolean', default: false })
  isGroupedProgram: boolean;


  @Column({ name: 'logo_url', type:'varchar', length: 1024, nullable: true })
  logoUrl: string;

  @Column({ name: 'banner_url', type:'varchar', length: 1024, nullable: true })
  bannerUrl: string;

  @Column({name : "key", type: "varchar", length: 255, nullable: true, unique: true})
  key: string;

  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true })
  deletedAt: Date;

  @ManyToOne(() => User, { nullable: true })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @ManyToOne(() => User, { nullable: true })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  @OneToMany(() => Program, program => program.type)
  programs: Program[];

  @Column({ name: 'igst', type: 'decimal', precision: 5, scale: 2, default: 0 })
  igst: number;

  @Column({ name: 'cgst', type: 'decimal', precision: 5, scale: 2, default: 0 })
  cgst: number;

  @Column({ name: 'sgst', type: 'decimal', precision: 5, scale: 2, default: 0 })
  sgst: number;

  @Column({ name: 'gst_percentage', type: 'decimal', precision: 5, scale: 2, default: 18.0 })
  gstPercentage: number;

  @Column({ name: 'tds_percent', type: 'decimal', precision: 5, scale: 2, default: 0 })
  tdsPercent: number;

  @Column({name: "gst_number", type: "varchar", length: 15, nullable: true})
  gstNumber: string;

 @Column({
    name: 'tds_applicability',
    type: 'enum',
    enum: TdsApplicabilityEnum,
    nullable: true,
  })
  tdsApplicability: TdsApplicabilityEnum;

  @Column({ name: 'invoice_sender_name', type: 'varchar', length: 255, nullable: true })
  invoiceSenderName: string;

  @Column({ name: 'invoice_sender_pan', type: 'varchar', length: 50, nullable: true })
  invoiceSenderPan: string;

  @Column({ name: 'invoice_sender_cin', type: 'varchar', length: 50, nullable: true })
  invoiceSenderCin: string;

  @Column({ name: 'invoice_sender_address', type: 'text', nullable: true })
  invoiceSenderAddress: string;

  @Column({ name: 'helpline_number', type: 'text', nullable: true })
  helplineNumber: string;

  @Column({ name: 'email_sender_name', type: 'text', nullable: true })
  emailSenderName: string;

  @Column({ name: 'venue_name_in_emails', type: 'text', nullable: true })
  venueNameInEmails: string;

  @Column({ name: 'seeker_can_share_experience', type: 'boolean', default: false })
  seekerCanShareExperience: boolean;

  constructor(partial: Partial<ProgramType>) {
    Object.assign(this, partial);
  }
}