import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  DeleteDateColumn,
  ManyToOne,
  OneToMany,
  JoinColumn,
} from 'typeorm';
import { User } from './user.entity';
import { Room } from './room.entity';
import { Program } from './program.entity';
import { RoomAllocation } from './room-allocation.entity';
import { RoomStatus } from '../enums/room-status.enum';
import { RoomCategoryEnum } from '../enum/room-category.enum';

/**
 * Entity representing the mapping of program inventory to rooms
 * Contains room availability and reservation details for programs
 */
@Entity('program_room_inventory_map')
export class ProgramRoomInventoryMap {
  @PrimaryGeneratedColumn()
  id: number;

  @Column({ name: 'room_id', type: 'bigint', nullable: false })
  roomId: number;

  @Column({ name: 'program_id', type: 'bigint', nullable: false })
  programId: number;

  @Column({ name: 'sub_program_id', type: 'bigint', nullable: true })
  subProgramId: number;

  @Column({ name: 'remaining_occupancy', type: 'int', nullable: false })
  remainingOccupancy: number;

  @Column({ name: 'is_reserved', type: 'boolean', nullable: false })
  isReserved: boolean;

  @Column({ name: 'reserved_for', type: 'varchar', length: 255, nullable: true })
  reservedFor: string;

  @Column({ 
    name: 'room_category', 
    type: 'enum',
    enum: RoomCategoryEnum,
    default: RoomCategoryEnum.GENERAL,
    nullable: false 
  })
  roomCategory: RoomCategoryEnum;

  @Column({ 
    name: 'room_status', 
    type: 'enum',
    enum: RoomStatus,
    default: RoomStatus.AVAILABLE,
    nullable: false 
  })
  roomStatus: RoomStatus;

  @Column({ name: 'occupied_starts_at', type: 'timestamp', nullable: true })
  occupiedStartsAt: Date;

  @Column({ name: 'occupied_ends_at', type: 'timestamp', nullable: true })
  occupiedEndsAt: Date;

  @Column({ name: 'remarks', type: 'text', nullable: true })
  remarks: string;

  @CreateDateColumn({ name: 'created_at', type: 'timestamp', nullable: false })
  createdAt: Date;

  @UpdateDateColumn({ name: 'updated_at', type: 'timestamp', nullable: true })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamp', nullable: true })
  deletedAt: Date;

  @Column({ name: 'created_by', type: 'bigint', nullable: false })
  createdById: number;

  @Column({ name: 'updated_by', type: 'bigint', nullable: true })
  updatedById: number;

  // Relationships
  @ManyToOne(() => Room, (room) => room.programRoomInventoryMaps, { nullable: false })
  @JoinColumn({ name: 'room_id' })
  room: Room;

  @ManyToOne(() => Program, { nullable: false })
  @JoinColumn({ name: 'program_id' })
  program: Program;

  @ManyToOne(() => Program, { nullable: true })
  @JoinColumn({ name: 'sub_program_id' })
  subProgram: Program;

  @ManyToOne(() => User, { nullable: false })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @ManyToOne(() => User, { nullable: true })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  @OneToMany(() => RoomAllocation, (allocation) => allocation.programRoomInventoryMap)
  roomAllocations: RoomAllocation[];

  constructor(partial: Partial<ProgramRoomInventoryMap>) {
    Object.assign(this, partial);
  }
}