import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  JoinColumn,
  OneToOne,
  ManyToOne,
  ManyToMany,
  JoinTable,
  OneToMany,
} from 'typeorm';
import { ProgramRegistration } from './program-registration.entity';
import { Program } from './program.entity';
import { SwapRequestedProgram } from './swap-requested-program.entity';
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';

@Entity('hdb_program_registration_swap')
@Auditable()
export class ProgramRegistrationSwap {
  @PrimaryGeneratedColumn('increment', { name: 'id', type: 'bigint' })
  id: number;

  @Column({ name: 'swap_requirement', type: 'varchar', length: 50, default: 'SWAP_REQUEST' })
  swapRequirement: 'SWAP_REQUEST' | 'SWAP_DEMAND';

  @Column({ name: 'program_registration_id', type: 'bigint', })
  programRegistrationId: number;

  @SkipAudit()
  @OneToOne(() => ProgramRegistration, )
  @JoinColumn({ name: 'program_registration_id', referencedColumnName: 'id' })
  programRegistration?: ProgramRegistration;

  @Column({ name: 'type', type: 'varchar', length: 255 })
  type: string;

  @Column({ name: 'status', type: 'varchar', length: 255 })
  status: string;

  @Column({ name: 'comment', type: 'text' })
  comment: string;

  @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
  auditRefId: number;

  @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
  parentRefId: number;

  @SkipAudit()
  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @SkipAudit()
  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz' , default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @SkipAudit()
  @Column({ name: 'created_by', type: 'bigint', })
  createdBy: number;

  @SkipAudit()
  @Column({ name: 'updated_by', type: 'bigint', })
  updatedBy: number;

  @Column({ name: 'current_program_id', type: 'bigint' })
  currentProgramId: number;

  @SkipAudit()
  @ManyToOne(() => Program, {
    nullable: true,
  })
  @JoinColumn({ name: 'current_program_id', referencedColumnName: 'id' })
  currentProgram: Program;
  
  @Column({ name: 'allocated_program_id', type: 'bigint' })
  allocatedProgramId: number;
  
  @SkipAudit()
  @ManyToOne(() => Program, {
    nullable: true,
  })  
  @JoinColumn({ name: 'allocated_program_id', referencedColumnName: 'id' })
  allocatedProgram: Program;

  @SkipAudit()
  @OneToMany(() => SwapRequestedProgram, (swapRequestedProgram) => swapRequestedProgram.swapRequest)
  swapRequestedProgram?: SwapRequestedProgram[];

  @SkipAudit()
  @ManyToMany(() => Program, { nullable: true })
  @JoinTable({
    name: 'hdb_swap_requested_program',
    joinColumn: {
      name: 'swap_request_id',
      referencedColumnName: 'id',
    },
    inverseJoinColumn: {
      name: 'program_id',
      referencedColumnName: 'id',
    },
  })
  requestedPrograms?: Program[];
}
