import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  DeleteDateColumn,
  ManyToOne,
  JoinColumn,
} from 'typeorm';
import { Program, ProgramSession, User } from './index';
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';

@Entity('program_registration_failures')
@Auditable()  
export class ProgramRegistrationFailure {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ name: 'program_id', type: 'bigint' })
  programId: number;

  @Column({ name: 'program_session_id', type: 'bigint', nullable: true })
  programSessionId: number | null;

  @Column({ name: 'payload', type: 'jsonb' })
  payload: Record<string, any>;

  @Column({ name: 'failure_reason', type: 'varchar', length: 255 })
  failureReason: string;

  @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
  auditRefId: number;

  @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
  parentRefId: number;

  @SkipAudit()
  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @SkipAudit()
  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true, default: null })
  deletedAt: Date;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  @SkipAudit()
  @ManyToOne(() => Program, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'program_id' })
  program: Program;

  @SkipAudit()
  @ManyToOne(() => ProgramSession, { nullable: true, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'program_session_id' })
  programSession: ProgramSession | null;

  constructor(partial: Partial<ProgramRegistrationFailure>) {
    Object.assign(this, partial);
  }
}
