import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
} from 'typeorm';
import { CommunicationTypeEnum } from '../enum/communication-type.enum';
import { CommunicationStatusEnum } from '../enum/communication-status.enum';
import { ProgramRegistration, User } from '../entities/index';
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';

@Entity('hdb_communication_track')
@Auditable()
export class CommunicationTrack {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ name: 'registration_id', type: 'bigint' })
  registrationId: number;

  @Column({ name: 'typ', type: 'enum', enum: CommunicationTypeEnum })
  typ: CommunicationTypeEnum;

  @Column({
    name: 'stats',
    type: 'enum',
    enum: CommunicationStatusEnum,
    default: CommunicationStatusEnum.INITIATED,
  })
  stats: CommunicationStatusEnum;

  @Column({ name: 'failed_reason', type: 'text', nullable: true })
  failedReason: string;

  @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
  auditRefId: number;

  @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
  parentRefId: number;

  @SkipAudit()
  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @SkipAudit()
  @UpdateDateColumn({
    name: 'updated_at',
    type: 'timestamptz',
    default: () => 'CURRENT_TIMESTAMP',
    onUpdate: 'CURRENT_TIMESTAMP',
  })
  updatedAt: Date;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  @SkipAudit()
  @ManyToOne(() => ProgramRegistration, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'registration_id' })
  registration: ProgramRegistration;

  constructor(partial: Partial<CommunicationTrack>) {
    Object.assign(this, partial);
  }
}
