/**
 * Filter Hierarchy Configuration
 * Centralized configuration for three-level filter system with new naming conventions
 * parentFilter (level1) → kpiFilter (level2) → sideFilters (level3)
 */

import { citiesList } from "../constants/constants";

export interface FilterOption {
  key: string;
  label: string;
  value: string;
  kpiFilter?: string;
  kpiCategory?: string;
  programType?: string;
  programSequence?: number;
  isDefault?: boolean;
}

export interface ParentFilterOption {
  key: string;
  label: string;
  value: string;
  isDefault?: boolean;
  kpiOptions: FilterOption[];
}

export interface SideFilterOption {
  key: string;
  label: string;
  type: 'checkbox' | 'dropdown' | 'date' | 'multi';
  options: Array<{
    label: string;
    value: string;
  }>;
}

export interface ReportDefinition {
  code: string;
  label: string;
  description: string;
  headers: string[];
}

export interface FilterHierarchyConfig {
  filterHierarchy: {
    parentOptions: ParentFilterOption[];
    sideFilterSets: {
      baseSets: {
        [key: string]: SideFilterOption[];
      };
      contextualFilters: {
        [parentKey: string]: {
          [kpiKey: string]: string[];
        };
      };
    };
  };
}

// Report definitions with headers and column configurations for data exports
export const REPORT_DEFINITIONS: { [key: string]: ReportDefinition } = {
  all_seekers: {
    code: 'all_seekers',
    label: 'All Seekers',
    description: 'High Level Data of registered seekers',
    headers: [
      'Name', 'Email', 'Mobile', 'Gender', 'Country', 
      'User created on', 'City', 'Registration Status'
    ]
  },
  registration_form_report: {
    code: 'registration_form_report',
    label: 'Registration Form Report',
    description: 'Detailed Data of registered seekers',
    headers: [
      'Name', 'HDB Allocation', 'Infinitheism Contact', 
      'Payment Status', 'Travel Status', 'Offered', 'Status', 'Email', 'DOB', 
      'Mobile', 'Gender', 'Country', 'City', 'Room-mate Preference', 
      'HDB/MSD done?', 'Last HDB/MSD', 'Choices', 'Experiences', 'Payment method', 
      'Jacket Size', 'Ratria Pillar in Leonia', 'Registered on', 
      'Swap Status', 'Wants Swap/Can Shift', 'Following up', 'Rating', 
      'Recommendation', 'Recommendation Comments', 'Payment From ID'
    ]
  },
  swap_can_shift_report: {
    code: 'swap_can_shift_report',
    label: 'Swap Can Shift Report',
    description: 'Swap and Can Shift requests data of registered seekers',
    headers: [
      'Name', 'HDB Allocation', 'Infinitheism Contact', 'Payment Status', 
      'Travel Status', 'Goodies Status', 'Offered', 'Swap Status', 
      'Wants Swap/Can Shift', 'Following up', 'Swap / Can shift request to', 
      'Swap / Can Shift RM Comments', 'Status', 'Date of blessing', 
      'Email', 'DOB', 'Mobile', 'Gender', 'City', 'Country', 
      'Room-mate Preference', 'HDB/MSD\'s done', 'Last HDB/MSD', 'Choices', 
      'Experiences', 'Payment method', 'Jacket Size', 'Ratria Pillar in Leonia', 
      'Registered on', 'Rating', 'Recommendation', 'Recommendation Comments', 
      'Payment From ID'
    ]
  },
  drafts_report: {
    code: 'drafts_report',
    label: 'Drafts Report',
    description: '',
    headers: [
      'Name', 'Infinitheism Contact', 'Status', 'Email', 'DOB', 
      'Mobile', 'Gender', 'Country', 'City', 'Room-mate Preference', 
      'HDB/MSD done?', 'Last HDB/MSD', 'Choices', 'Experiences', 'Drafted on'
    ]
  },
  finance_report: {
    code: 'finance_report',
    label: 'Finance Report',
    description: '',
    headers: [
      'S.No.', 'Seeker Email', 'Registration ID', 'Seeker Name', 'HDB / MSD Offered', 'Blessed', 
      'Mobile', 'City', 'Country', 'Contact Person (RM)', 'Date of blessing', 
      'Fee Amount', 'GST 18%', 'Total amount', 'TDS', 'Amount Received', 
      'GSTIN', 'TAN', 'Payment Status', 'Mode of payment', 'Order ID (Razor pay)', 
      'Payment ID (Razor pay)', 'Cheque/DD #', 'Cheque/DD Date', 'NEFT/Bank Transfer Ref. #', 
      'NEFT / Bank Transfer Date', 'Payment link completion date', 'Invoice Number', 
      'Invoice Date', 'Invoice Name', 'Invoice Email', 'Invoice Address', 'Pincode', 
      'Place of Supply', 'GSTIN Status', 'e-Invoice Ack #', 'e-Invoice Date', 'Programe Name with period', 'Modified User', 'Modified Time'
    ]
  },
  travel_report: {
    code: 'travel_report',
    label: 'Travel Report',
    description: '',
    headers: [
      'Name', 'Travel Status', 'HDB', 'Email', 'HDB\'s Done',
      'Mobile', 'Date of Birth', 'Gender', 'Age', 'City', 'Note Book',
      'Ratria Pillar in Leonia', 'Flask', 'Jacket Size', 'T-Shirt Size', 'Upload ID card (Front / Front+Back)',
      'Upload ID card (only Back)', 'Pick-Up from', 'Coming From', 'Onward Airlines',
      'Onward Flight Number', 'Onward Arrival Time', 'Onward Additional Information',
      'Drop to', 'Going to', 'Return Airlines', 'Return Flight Number',
      'Return Departure Time', 'Return Additional Information', 'Uploaded International ID card',
      'Upload VISA Copy', 'Upload Passport Copy', 'Modified User', 'Comments'
    ]
  },
  // master_list: {
  //   code: 'master_list',
  //   label: 'Master List',
  //   description: '',
  //   headers: [
  //     'Name', 'Date of Birth', 'Age', 'City', 'Country',
  //     'Infinitheism Contact', 'Other', 'Email', 'Mobile', 'Registration Date',
  //     'HDB/MSD done', 'Last HDB/MSD', 'Offered', 'All Preferences',
  //     'Registration Status', 'Blessed Date', 'Payment Status', 'Modified User',
  //     'Last Modified Time', 'Registration ID', 'Loggedin-Seeker', 'Auto Number',
  //     'Gender', 'Song 1', 'Song 2'
  //   ]
  // },
  master_travel_and_goodies_report: {
    code: 'master_travel_and_goodies_report',
    label: 'Master - Travel and Goodies Report',
    description: '',
    headers: [
      'Name', 'Date Of Birth', 'Gender', 'Travel Status', 'Goodies Status', 'Age', 'Contact Person',
      'Mobile', 'Alternate Number', 'HDB/MSD', 'HDB/MSD With Date', 'HDB/MSD\'s Done',
      'Last HDB/MSD', 'Date of Blessing', 'Pick-Up From', 'Coming From', 'Onward Terminal', 
      'Onward Airlines', 'Onward Flight Number', 'Onward Arrival Date and Time', 'Onward Additional Information',
      'Drop To', 'Going To', 'Return Terminal', 'Return Airlines', 'Return Flight Number', 
      'Return Departure Date and Time', 'Return Additional Information', 'Upload ID Card (Front / Front+Back)',
      'Upload ID Card (Only Back)', 'Upload International ID Card', 'Upload Passport Copy', 'Upload VISA Copy',
      'Ratria Pillar', 'Where do you want to give the Ratria Pillars', 'Flask', 'Note Book', 'Jacket Size', 'T-Shirt Size',
      'Room Number', 'Travel Last Modified Date/Time', 'Registration ID'
    ]
  },
  travel_onward_and_return_report: {
    code: 'travel_onward_and_return_report',
    label: 'Travel - Onward and Return Report',
    description: '',
    headers: [
      '#', 'Registration number/ID', 'Name', 'Travel Status', 'HDB/MSD', 'Last HDB/MSD', 'HDB Allocation',
      'Email', 'Registered Mobile', 'Gender', 'Date of Birth', 'Age', 'RM', 'City', 'Alternate Number',
      'Pick-Up From', 'Coming From', 'Onward Terminal', 'Onward Airlines', 'Onward Flight Number',
      'Onward Arrival Time', 'Onward Additional Information', 'Drop To', 'Going To', 'Return Terminal',
      'Return Airlines', 'Return Flight Number', 'Return Departure Time', 'Return Additional Information',
      'Added Time', 'Modified Time'
    ]
  },
  goodies_and_ratria_pillar_report: {
    code: 'goodies_and_ratria_pillar_report',
    label: 'Goodies and Ratria Pillar Report',
    description: '',
    headers: [
      '#', 'Registration number/ID', 'Name', 'Goodies Status', 'HDB/MSD done', 'Last HDB/MSD', 'HDB Allocation',
      'Gender', 'RM', 'City', 'Ratria Pillar', 'Flask', 'Note Book',
      'T-Shirt Size', 'Jacket Size', 'Added Time', 'Modified Time'
    ]
  },
  travel_onward_report: {
    code: 'travel_onward_report',
    label: 'Travel - Onward Report',
    description: '',
    headers: [
      '#','Room #','Registration number/ID', 'Name', 'Travel Status', 'HDB/MSD done', 'Last HDB/MSD',
      'HDB Allocation', 'Email', 'Registered Mobile', 'Gender', 'Date of Birth', 'Age', 'RM', 'City',
      'Alternate Number', 'Pick-Up From', 'Coming From', 'Onward Terminal', 'Onward Airlines',
      'Onward Flight Number', 'Onward Arrival Time', 'Onward Additional Information',
      'Added Time', 'Modified Time'
    ]
  },
  travel_return_report: {
    code: 'travel_return_report',
    label: 'Travel - Return Report',
    description: '',
    headers: [
      '#','Room #','Registration number/ID', 'Name', 'Travel Status', 'Last HDB/MSD',
      'HDB Allocation', 'Email', 'Registered Mobile', 'Gender', 'Date of Birth', 'Age', 'RM', 'City',
      'Alternate Number', 'Drop To', 'Going To', 'Return Terminal', 'Return Airlines',
      'Return Flight Number', 'Return Departure Time', 'Return Additional Information',
      'Added Time', 'Modified Time'
    ]
  },
  payment_list: {
    code: 'payment_list',
    label: 'Payment List',
    description: '',
    headers: [
      'Name', 'Email', 'Mobile', 'Contact Person', 'Pay method', 'HDB Offered',
      'Payment Status', 'RazorPay Status', 'Investment', 'GST @ 18%', 'Batch Code',
      'Modified Time', 'Modified User', 'Registration ID', 'RazorPay Order ID',
      'Razor pay Payment ID'
    ]
  },
  birthday_list: {
    code: 'birthday_list',
    label: 'Birthday List',
    description: '',
    headers: [
      'HDB', 'Name', 'Date of Birth', 'Email', 'Gender',
      'Mobile', 'Contact', 'HDB\'s Done', 'City', 'Country'
    ]
  },
  song_preferences_list: {
    code: 'song_preferences_list',
    label: 'Song Preferences List',
    description: '',
    headers: [
      'Name', 'Email', 'Mobile', 'Gender', 'Date of Birth', 'City',
      'Song Preference 1', 'Song Preference 2', 'Song Preference 3', 'Contact Person'
    ]
  },
  hold_list: {
    code: 'hold_list',
    label: 'Hold List',
    description: '',
    headers: [
      'Name', 'HDB/MSD\'s done', 'Offered', 'City', 'Country',
      'Infinitheism Contact', 'Rating', 'Mobile', 'Room-mate Name',
      'Last Modified Time', 'Modified User', 'Travel Status'
    ]
  },
  confirmed_list: {
    code: 'confirmed_list',
    label: 'Confirmed List',
    description: '',
    headers: [
      'Name', 'HDB Offered', 'HDB/MSD\'s done', 'Last HDB/MSD',
      'Choices', 'City', 'Country', 'Infinitheism Contact', 'Rating', 'Mobile',
      'Gender', 'Room-mate Name', 'Blessed Date and Time', 'Date of Birth',
      'Registration ID', 'Email'
    ]
  }
};

export const FILTER_HIERARCHY_CONFIG: FilterHierarchyConfig = {
  filterHierarchy: {
    parentOptions: [
      {
        key: "all",
        label: "All",
        value: "all",
        isDefault: true,
        kpiOptions: [
          {
            key: "all",
            label: "All",
            value: "all",
            kpiFilter: "all",
            kpiCategory: "all",
            isDefault: true
          },
          {
            key: "unassigned",
            label: "Unassigned",
            value: "unassigned",
            kpiFilter: "newSeekersPending",
            kpiCategory: "registrations"
          },
          {
            key: "pending",
            label: "Pending",
            value: "registrationPendingCount",
            kpiFilter: "regPending",
            kpiCategory: "registrations"
          },
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "approvedSeekers",
            kpiCategory: "registrations"
          },
          {
            key: "msd1",
            label: "MSD 1",
            value: "msd1",
            kpiFilter: "program_{MSD1_ID}",
            kpiCategory: "programs",
            programType: "MSD",
            programSequence: 1
          },
          {
            key: "hdb1",
            label: "HDB 1",
            value: "hdb1",
            kpiFilter: "program_{HDB1_ID}",
            kpiCategory: "programs",
            programType: "HDB",
            programSequence: 1
          },
          {
            key: "msd2",
            label: "MSD 2",
            value: "msd2",
            kpiFilter: "program_{MSD2_ID}",
            kpiCategory: "programs",
            programType: "MSD",
            programSequence: 2
          },
          {
            key: "hdb2",
            label: "HDB 2",
            value: "hdb2",
            kpiFilter: "program_{HDB2_ID}",
            kpiCategory: "programs",
            programType: "HDB",
            programSequence: 2
          },
          {
            key: "hdb3",
            label: "HDB 3",
            value: "hdb3",
            kpiFilter: "program_{HDB3_ID}",
            kpiCategory: "programs",
            programType: "HDB",
            programSequence: 3
          },
          {
            key: "onHold",
            label: "Hold",
            value: "onHold",
            kpiFilter: "rejected",
            kpiCategory: "registrations"
          },
          {
            key: "ytd",
            label: "Swap demand",
            value: "ytd",
            kpiFilter: "onHold",
            kpiCategory: "registrations"
          },
          {
            key: "cancelled",
            label: "Cancelled",
            value: "cancelled",
            kpiFilter: "cancelled",
            kpiCategory: "registrations"
          }
        ]
      },
      {
        key: "blessed",
        label: "Blessed",
        value: "blessed",
        kpiOptions: [
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "approvedSeekers",
            kpiCategory: "blessed",
            isDefault: true
          },
          {
            key: "msd1",
            label: "MSD 1",
            value: "msd1",
            kpiFilter: "program_{MSD1_ID}",
            kpiCategory: "blessed",
            programType: "MSD",
            programSequence: 1
          },
          {
            key: "hdb1",
            label: "HDB 1",
            value: "hdb1",
            kpiFilter: "program_{HDB1_ID}",
            kpiCategory: "blessed",
            programType: "HDB",
            programSequence: 1
          },
          {
            key: "msd2",
            label: "MSD 2",
            value: "msd2",
            kpiFilter: "program_{MSD2_ID}",
            kpiCategory: "blessed",
            programType: "MSD",
            programSequence: 2
          },
          {
            key: "hdb2",
            label: "HDB 2",
            value: "hdb2",
            kpiFilter: "program_{HDB2_ID}",
            kpiCategory: "blessed",
            programType: "HDB",
            programSequence: 2
          },
          {
            key: "hdb3",
            label: "HDB 3",
            value: "hdb3",
            kpiFilter: "program_{HDB3_ID}",
            kpiCategory: "blessed",
            programType: "HDB",
            programSequence: 3
          },
          {
            key: "swapRequests",
            label: "Swap request",
            value: "swapRequests",
            kpiFilter: "swapRequests",
            kpiCategory: "blessed"
          }
        ]
      },
      {
        key: "msd1Blessed",
        label: "MSD 1",
        value: "msd1Blessed",
        kpiOptions: [
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "program_{MSD1_ID}",
            kpiCategory: "program_{MSD1_ID}",
            isDefault: true
          },
          {
            key: "paymentPending",
            label: "Payment pending",
            value: "paymentPending",
            kpiFilter: "paymentPending",
            kpiCategory: "program_{MSD1_ID}"
          },
          {
            key: "paymentComplete",
            label: "Payment complete",
            value: "paymentComplete",
            kpiFilter: "paymentComplete",
            kpiCategory: "program_{MSD1_ID}"
          },
          {
            key: "invoicePending",
            label: "Invoice pending",
            value: "invoicePending",
            kpiFilter: "invoicePending",
            kpiCategory: "program_{MSD1_ID}"
          },
          {
            key: "invoiceComplete",
            label: "Invoice complete",
            value: "invoiceComplete",
            kpiFilter: "invoiceComplete",
            kpiCategory: "program_{MSD1_ID}"
          },
          {
            key: "travelPending",
            label: "Travel pending",
            value: "travelPending",
            kpiFilter: "travelPending",
            kpiCategory: "program_{MSD1_ID}"
          },
          {
            key: "travelComplete",
            label: "Travel complete",
            value: "travelComplete",
            kpiFilter: "travelComplete",
            kpiCategory: "program_{MSD1_ID}"
          }
        ]
      },
      {
        key: "hdb1Blessed",
        label: "HDB 1",
        value: "hdb1Blessed",
        kpiOptions: [
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "program_{HDB1_ID}",
            kpiCategory: "program_{HDB1_ID}",
            isDefault: true
          },
          {
            key: "paymentPending",
            label: "Payment pending",
            value: "paymentPending",
            kpiFilter: "paymentPending",
            kpiCategory: "program_{HDB1_ID}"
          },
          {
            key: "paymentComplete",
            label: "Payment complete",
            value: "paymentComplete",
            kpiFilter: "paymentComplete",
            kpiCategory: "program_{HDB1_ID}"
          },
          {
            key: "invoicePending",
            label: "Invoice pending",
            value: "invoicePending",
            kpiFilter: "invoicePending",
            kpiCategory: "program_{HDB1_ID}"
          },
          {
            key: "invoiceComplete",
            label: "Invoice complete",
            value: "invoiceComplete",
            kpiFilter: "invoiceComplete",
            kpiCategory: "program_{HDB1_ID}"
          },
          {
            key: "travelPending",
            label: "Travel pending",
            value: "travelPending",
            kpiFilter: "travelPending",
            kpiCategory: "program_{HDB1_ID}"
          },
          {
            key: "travelComplete",
            label: "Travel complete",
            value: "travelComplete",
            kpiFilter: "travelComplete",
            kpiCategory: "program_{HDB1_ID}"
          }
        ]
      },
      {
        key: "msd2Blessed",
        label: "MSD 2",
        value: "msd2Blessed",
        kpiOptions: [
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "program_{MSD2_ID}",
            kpiCategory: "program_{MSD2_ID}",
            isDefault: true
          },
          {
            key: "paymentPending",
            label: "Payment pending",
            value: "paymentPending",
            kpiFilter: "paymentPending",
            kpiCategory: "program_{MSD2_ID}"
          },
          {
            key: "paymentComplete",
            label: "Payment complete",
            value: "paymentComplete",
            kpiFilter: "paymentComplete",
            kpiCategory: "program_{MSD2_ID}"
          },
          {
            key: "invoicePending",
            label: "Invoice pending",
            value: "invoicePending",
            kpiFilter: "invoicePending",
            kpiCategory: "program_{MSD2_ID}"
          },
          {
            key: "invoiceComplete",
            label: "Invoice complete",
            value: "invoiceComplete",
            kpiFilter: "invoiceComplete",
            kpiCategory: "program_{MSD2_ID}"
          },
          {
            key: "travelPending",
            label: "Travel pending",
            value: "travelPending",
            kpiFilter: "travelPending",
            kpiCategory: "program_{MSD2_ID}"
          },
          {
            key: "travelComplete",
            label: "Travel complete",
            value: "travelComplete",
            kpiFilter: "travelComplete",
            kpiCategory: "program_{MSD2_ID}"
          }
        ]
      },
      {
        key: "hdb2Blessed",
        label: "HDB 2",
        value: "hdb2Blessed",
        kpiOptions: [
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "program_{HDB2_ID}",
            kpiCategory: "program_{HDB2_ID}",
            isDefault: true
          },
          {
            key: "paymentPending",
            label: "Payment pending",
            value: "paymentPending",
            kpiFilter: "paymentPending",
            kpiCategory: "program_{HDB2_ID}"
          },
          {
            key: "paymentComplete",
            label: "Payment complete",
            value: "paymentComplete",
            kpiFilter: "paymentComplete",
            kpiCategory: "program_{HDB2_ID}"
          },
          {
            key: "invoicePending",
            label: "Invoice pending",
            value: "invoicePending",
            kpiFilter: "invoicePending",
            kpiCategory: "program_{HDB2_ID}"
          },
          {
            key: "invoiceComplete",
            label: "Invoice complete",
            value: "invoiceComplete",
            kpiFilter: "invoiceComplete",
            kpiCategory: "program_{HDB2_ID}"
          },
          {
            key: "travelPending",
            label: "Travel pending",
            value: "travelPending",
            kpiFilter: "travelPending",
            kpiCategory: "program_{HDB2_ID}"
          },
          {
            key: "travelComplete",
            label: "Travel complete",
            value: "travelComplete",
            kpiFilter: "travelComplete",
            kpiCategory: "program_{HDB2_ID}"
          }
        ]
      },
      {
        key: "hdb3Blessed",
        label: "HDB 3",
        value: "hdb3Blessed",
        kpiOptions: [
          {
            key: "blessed",
            label: "Blessed",
            value: "blessed",
            kpiFilter: "program_{HDB3_ID}",
            kpiCategory: "program_{HDB3_ID}",
            isDefault: true
          },
          {
            key: "paymentPending",
            label: "Payment pending",
            value: "paymentPending",
            kpiFilter: "paymentPending",
            kpiCategory: "program_{HDB3_ID}"
          },
          {
            key: "paymentComplete",
            label: "Payment complete",
            value: "paymentComplete",
            kpiFilter: "paymentComplete",
            kpiCategory: "program_{HDB3_ID}"
          },
          {
            key: "invoicePending",
            label: "Invoice pending",
            value: "invoicePending",
            kpiFilter: "invoicePending",
            kpiCategory: "program_{HDB3_ID}"
          },
          {
            key: "invoiceComplete",
            label: "Invoice complete",
            value: "invoiceComplete",
            kpiFilter: "invoiceComplete",
            kpiCategory: "program_{HDB3_ID}"
          },
          {
            key: "travelPending",
            label: "Travel pending",
            value: "travelPending",
            kpiFilter: "travelPending",
            kpiCategory: "program_{HDB3_ID}"
          },
          {
            key: "travelComplete",
            label: "Travel complete",
            value: "travelComplete",
            kpiFilter: "travelComplete",
            kpiCategory: "program_{HDB3_ID}"
          }
        ]
      },
      {
        key: "regPending",
        label: "Pending",
        value: "regPending",
        kpiOptions: [
          {
            key: "regPending",
            label: "Pending",
            value: "regPending",
            kpiFilter: "regPending",
            kpiCategory: "registrations",
            isDefault: true
          },
          {
            key: "msd1Pref",
            label: "MSD 1 pref",
            value: "msd1Pref",
            kpiFilter: "regPending_program_{MSD1_ID}",
            kpiCategory: "regPending"
          },
          {
            key: "hdb1Pref",
            label: "HDB 1 pref",
            value: "hdb1Pref",
            kpiFilter: "regPending_program_{HDB1_ID}",
            kpiCategory: "regPending"
          },
          {
            key: "msd2Pref",
            label: "MSD 2 pref",
            value: "msd2Pref",
            kpiFilter: "regPending_program_{MSD2_ID}",
            kpiCategory: "regPending"
          },
          {
            key: "hdb2Pref",
            label: "HDB 2 pref",
            value: "hdb2Pref",
            kpiFilter: "regPending_program_{HDB2_ID}",
            kpiCategory: "regPending"
          },
          {
            key: "hdb3Pref",
            label: "HDB 3 pref",
            value: "hdb3Pref",
            kpiFilter: "regPending_program_{HDB3_ID}",
            kpiCategory: "regPending"
          }
        ]
      },
      {
        key: "swapDemand",
        label: "Swap demand",
        value: "swapDemand",
        kpiOptions: [
          {
            key: "swapDemand",
            label: "Swap demand",
            value: "swapDemand",
            kpiFilter: "onHold",
            kpiCategory: "swapDemand",
            isDefault: true
          },
          {
            key: "msd1Pref",
            label: "MSD 1 pref",
            value: "msd1Pref",
            kpiFilter: "onHold_program_{MSD1_ID}",
            kpiCategory: "swapDemand"
          },
          {
            key: "hdb1Pref",
            label: "HDB 1 pref",
            value: "hdb1Pref",
            kpiFilter: "onHold_program_{HDB1_ID}",
            kpiCategory: "swapDemand"
          },
          {
            key: "msd2Pref",
            label: "MSD 2 pref",
            value: "msd2Pref",
            kpiFilter: "onHold_program_{MSD2_ID}",
            kpiCategory: "swapDemand"
          },
          {
            key: "hdb2Pref",
            label: "HDB 2 pref",
            value: "hdb2Pref",
            kpiFilter: "onHold_program_{HDB2_ID}",
            kpiCategory: "swapDemand"
          },
          {
            key: "hdb3Pref",
            label: "HDB 3 pref",
            value: "hdb3Pref",
            kpiFilter: "onHold_program_{HDB3_ID}",
            kpiCategory: "swapDemand"
          }
        ]
      },
      {
        key: "swapRequests",
        label: "Swap request",
        value: "swapRequests",
        kpiOptions: [
          {
            key: "swapRequests",
            label: "Swap request",
            value: "swapRequests",
            kpiFilter: "swapRequests",
            kpiCategory: "swapRequests",
            isDefault: true
          },
          {
            key: "msd1Pref",
            label: "MSD 1 pref",
            value: "msd1Pref",
            kpiFilter: "swapRequests_program_{MSD1_ID}",
            kpiCategory: "swapRequests"
          },
          {
            key: "hdb1Pref",
            label: "HDB 1 pref",
            value: "hdb1Pref",
            kpiFilter: "swapRequests_program_{HDB1_ID}",
            kpiCategory: "swapRequests"
          },
          {
            key: "msd2Pref",
            label: "MSD 2 pref",
            value: "msd2Pref",
            kpiFilter: "swapRequests_program_{MSD2_ID}",
            kpiCategory: "swapRequests"
          },
          {
            key: "hdb2Pref",
            label: "HDB 2 pref",
            value: "hdb2Pref",
            kpiFilter: "swapRequests_program_{HDB2_ID}",
            kpiCategory: "swapRequests"
          },
          {
            key: "hdb3Pref",
            label: "HDB 3 pref",
            value: "hdb3Pref",
            kpiFilter: "swapRequests_program_{HDB3_ID}",
            kpiCategory: "swapRequests"
          }
        ]
      }
    ],
    sideFilterSets: {
      baseSets: {
        common: [
          {
            key: "gender",
            label: "Gender",
            type: "checkbox",
            options: [
              {
                label: "Male",
                value: "Male"
              },
              {
                label: "Female",
                value: "Female"
              }
            ]
          },
          {
            key: "age",
            label: "Age",
            type: "checkbox",
            options: [
              {
                label: "<=20 years",
                value: "0-20"
              },
              {
                label: "21 - 30 years",
                value: "21-30"
              },
              {
                label: "31 - 50 years",
                value: "31-50"
              },
              {
                label: "51 - 65 years",
                value: "51-65"
              },
              {
                label: "> 65 years",
                value: ">65"
              }
            ]
          },
          {
            key: "rmContact",
            label: "RM Contact",
            type: "dropdown",
            options: []
          },
          {
            key: 'numberOfHdbs',
            label: 'Number of HDBs',
            type: 'multi',
            options: [
              {
                label: 'Equals',
                value: '=',
              },
              {
                label: 'Greater than',
                value: '>',
              },
              {
                label: 'Less than',
                value: '<',
              },
              {
                label: 'Within a range',
                value: '-',
              },
            ]
          },
          {
            key: 'experienceTags',
            label: "Experiences",
            type: "dropdown",
            options: []
          },
          {
            key: "freeSeat",
            label: "Seat Type",
            type: "checkbox",
            options: [
              {
                label: "Free Seat",
                value: "yes"
              },
              {
                label: "Paid Seat",
                value: "no"
              }
            ]
          },
          {
            key: "dateRange",
            label: "Seeker registration date range",
            type: "date",
            options: [
              {
                label: "Start Date",
                value: "startDate"
              },
              {
                label: "End Date",
                value: "endDate"
              }
            ]
          },
          {
            key: "preferredRoomMate",
            label: "Preferred Roommate",
            type: "checkbox",
            options: [
              {
                label: "Yes",
                value: "yes"
              },
              {
                label: "No",
                value: "no"
              }
            ]
          },
          {
            key: "organisation",
            label: "From Organization",
            type: "checkbox",
            options: [
              {
                label: "Yes",
                value: "yes"
              },
              {
                label: "No",
                value: "no"
              }
            ]
          },
          {
            key: "rmRating",
            label: "RM rating",
            type: "checkbox",
            options: [
              {
                label: "1 & up",
                value: "1up"
              },
              {
                label: "2 & up",
                value: "2up"
              },
              {
                label: "3 & up",
                value: "3up"
              },
              {
                label: "4 & up",
                value: "4up"
              },
              {
                label: "Only 5",
                value: "only5"
              },
              {
                label: "Yet to review",
                value: "unrated"
              }
            ]
          },
          {
            key: "recommendation",
            label: "RM Recommendation",
            type: "checkbox",
            options: [
              {
                label: "Wholeheartedly",
                value: "Wholeheartedly"
              },
              {
                label: "Affirmatively",
                value: "Affirmatively"
              },
              {
                label: "Supportively",
                value: "Supportively"
              },
              {
                label: "None",
                value: "none"
              }
            ]
          },
          {
            key: "birthdayListRange",
            label: "Birthday List Range",
            type: "checkbox",
            options: [
              {
                label: "MSD 1",
                value: "{MSD1_ID}"
              },
              {
                label: "HDB 1",
                value: "{HDB1_ID}"
              },
              {
                label: "MSD 2",
                value: "{MSD2_ID}"
              },
              {
                label: "HDB 2",
                value: "{HDB2_ID}"
              },
              {
                label: "HDB 3",
                value: "{HDB3_ID}"
              }
            ]
          },
          {
            key: "location",
            label: "Location",
            type: "dropdown",
            options: citiesList
          }
        ],
        approvalStatus: [
          {
            key: "approvalStatus",
            label: "Seeker status",
            type: "checkbox",
            options: [
              {
                label: "Unassigned",
                value: "pending"
              },
              {
                label: "Blessed",
                value: "approved"
              },
              {
                label: "Swap demand",
                value: "on_hold"
              },
              {
                label: "Hold",
                value: "rejected"
              }
            ]
          }
        ],
        payment: [
          {
            key: "paymentMode",
            label: "Payment mode",
            type: "checkbox",
            options: [
              {
                label: "Online",
                value: "online"
              },
              {
                label: "Offline",
                value: "offline"
              }
            ]
          },
          {
            key: "paymentStatus",
            label: "Payment status",
            type: "checkbox",
            options: [
              {
                label: "Completed",
                value: "payment_completed"
              },
              {
                label: "Pending",
                value: "payment_pending"
              },
              {
                label: "No payment",
                value: "no_payment"
              }
            ]
          }
        ],
        invoice: [
          {
            key: "invoiceStatus",
            label: "Invoice status",
            type: "checkbox",
            options: [
              {
                label: "Completed",
                value: "invoice_completed"
              },
              {
                label: "Pending",
                value: "invoice_pending"
              }
            ]
          }
        ],
        travel: [
          {
            key: "travelStatus",
            label: "Travel status",
            type: "checkbox",
            options: [
              {
                label: "Completed",
                value: "travel_completed"
              },
              {
                label: "Pending",
                value: "travel_pending"
              }
            ]
          }
        ],
        swap: [
          {
            key: "swapRequests",
            label: "Swap request",
            type: "checkbox",
            options: [
              {
                label: "Can shift",
                value: "can_shift"
              },
              {
                label: "Wants swap",
                value: "wants_swap"
              }
            ]
          }
        ],
        goodies: [
          {
            key: "goodiesStatus",
            label: "Goodies status",
            type: "checkbox",
            options: [
              {
                label: "Completed",
                value: "goodies_completed"
              },
              {
                label: "Pending",
                value: "goodies_pending"
              }
            ]
          }
        ]
      },
      contextualFilters: {
        all: {
          all: ["common", "approvalStatus", "payment", "invoice", "travel", "goodies", "swap"],
          unassigned: ["common", "payment", "invoice", "travel", "goodies"],
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          hdb1: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          hdb2: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          hdb3: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          msd1: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          msd2: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          onHold: ["common", "payment", "invoice", "travel", "goodies"],
          registrationPendingCount: ["common", "payment", "invoice", "travel", "goodies"],
          ytd: ["common", "payment", "invoice", "travel", "goodies"],
          cancelled: ["common", "payment", "invoice", "travel", "goodies"]
        },
        blessed: {
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          hdb1: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          hdb2: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          hdb3: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          msd1: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          msd2: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          swapRequests: ["common", "payment", "invoice", "travel", "goodies"],
          cancelled: ["common", "payment", "invoice", "travel", "goodies"]
        },
        hdb1Blessed: {
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          paymentPending: ["common", "invoice", "travel", "goodies", "swap"],
          paymentComplete: ["common", "invoice", "travel", "goodies", "swap"],
          invoicePending: ["common", "payment", "travel", "goodies", "swap"],
          invoiceComplete: ["common", "payment", "travel", "goodies", "swap"],
          travelPending: ["common", "payment", "invoice", "goodies", "swap"],
          travelComplete: ["common", "payment", "invoice", "goodies", "swap"]
        },
        hdb2Blessed: {
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          paymentPending: ["common", "invoice", "travel", "goodies", "swap"],
          paymentComplete: ["common", "invoice", "travel", "goodies", "swap"],
          invoicePending: ["common", "payment", "travel", "goodies", "swap"],
          invoiceComplete: ["common", "payment", "travel", "goodies", "swap"],
          travelPending: ["common", "payment", "invoice", "goodies", "swap"],
          travelComplete: ["common", "payment", "invoice", "goodies", "swap"]
        },
        hdb3Blessed: {
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          paymentPending: ["common", "invoice", "travel", "goodies", "swap"],
          paymentComplete: ["common", "invoice", "travel", "goodies", "swap"],
          invoicePending: ["common", "payment", "travel", "goodies", "swap"],
          invoiceComplete: ["common", "payment", "travel", "goodies", "swap"],
          travelPending: ["common", "payment", "invoice", "goodies", "swap"],
          travelComplete: ["common", "payment", "invoice", "goodies", "swap"]
        },
        msd1Blessed: {
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          paymentPending: ["common", "invoice", "travel", "goodies", "swap"],
          paymentComplete: ["common", "invoice", "travel", "goodies", "swap"],
          invoicePending: ["common", "payment", "travel", "goodies", "swap"],
          invoiceComplete: ["common", "payment", "travel", "goodies", "swap"],
          travelPending: ["common", "payment", "invoice", "goodies", "swap"],
          travelComplete: ["common", "payment", "invoice", "goodies", "swap"]
        },
        msd2Blessed: {
          blessed: ["common", "payment", "invoice", "travel", "goodies", "swap"],
          paymentPending: ["common", "invoice", "travel", "goodies", "swap"],
          paymentComplete: ["common", "invoice", "travel", "goodies", "swap"],
          invoicePending: ["common", "payment", "travel", "goodies", "swap"],
          invoiceComplete: ["common", "payment", "travel", "goodies", "swap"],
          travelPending: ["common", "payment", "invoice", "goodies", "swap"],
          travelComplete: ["common", "payment", "invoice", "goodies", "swap"]
        },
        swapRequests: {
          swapRequests: ["common", "payment", "invoice", "travel", "goodies"],
          hdb1Pref: ["common", "payment", "invoice", "travel", "goodies"],
          hdb2Pref: ["common", "payment", "invoice", "travel", "goodies"],
          hdb3Pref: ["common", "payment", "invoice", "travel", "goodies"],
          msd1Pref: ["common", "payment", "invoice", "travel", "goodies"],
          msd2Pref: ["common", "payment", "invoice", "travel", "goodies"]
        },
        swapDemand: {
          swapDemand: ["common", "payment", "invoice", "travel", "goodies"],
          hdb1Pref: ["common", "payment", "invoice", "travel", "goodies"],
          hdb2Pref: ["common", "payment", "invoice", "travel", "goodies"],
          hdb3Pref: ["common", "payment", "invoice", "travel", "goodies"],
          msd1Pref: ["common", "payment", "invoice", "travel", "goodies"],
          msd2Pref: ["common", "payment", "invoice", "travel", "goodies"]
        },
        regPending: {
          regPending: ["common", "payment", "invoice", "travel", "goodies"],
          hdb1Pref: ["common", "payment", "invoice", "travel", "goodies"],
          hdb2Pref: ["common", "payment", "invoice", "travel", "goodies"],
          hdb3Pref: ["common", "payment", "invoice", "travel", "goodies"],
          msd1Pref: ["common", "payment", "invoice", "travel", "goodies"],
          msd2Pref: ["common", "payment", "invoice", "travel", "goodies"]
        }
      }
    }
  }
};
