import { Controller, Get, HttpCode, HttpStatus, Res, UseGuards } from '@nestjs/common';
import { Response } from 'express';
import { CancellationReasonService } from './cancellation-reason.service';
import { ApiBearerAuth, ApiOperation, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import {
  cancellationReasonConstMessages,
  SWAGGER_API_RESPONSE,
} from 'src/common/constants/strings-constants';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('cancellation-reasons')
@Controller('cancellation-reasons')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles(
  'admin',
  'viewer',
  'finance_manager',
  'relational_manager',
  'shoba',
  'rm',
  'operational_manger',
  'rm_support',
  'mahatria',
)
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class CancellationReasonController {
  constructor(
    private readonly service: CancellationReasonService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: cancellationReasonConstMessages.GET_ALL })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async findAll(@Res() res: Response) {
    this.logger.log(cancellationReasonConstMessages.FINDING_ACTIVE);
    try {
      const data = await this.service.findActive();
      await this.responseService.success(res, cancellationReasonConstMessages.RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
