import { Controller, Post, Body, HttpStatus, UseGuards, Req } from '@nestjs/common';
import { UploadFileDto } from './dto/create-aws-handling.dto';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ApiBearerAuth, ApiOperation, ApiResponse, ApiSecurity } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { AppLoggerService } from 'src/common/services/logger.service';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
@Controller('aws')
@UseGuards(CombinedAuthGuard, RolesGuard)
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class AwsHandlingController {
  constructor(
    private readonly awsS3Service: AwsS3Service,
    private readonly logger: AppLoggerService,
  ) {}

  @Roles('admin', 'relational_manager', 'finance_manager', 'mahatria', 'shoba', 'viewer', 'operational_manger', 'rm_support', 'rm')
  @Post('/file-upload')
  @ApiOperation({ summary: 'Generate pre-signed URL for file upload' })
  @ApiResponse({ status: 200, description: 'Successfully generated file upload URL' })
  @ApiResponse({ status: 400, description: 'Invalid request parameters' })
  @ApiResponse({ status: 500, description: 'Internal server error' })
  async uploadFile(@Body() fileInfo: UploadFileDto, @Req() req: any) {
    try {
      let userId = req.user?.id;
      if (fileInfo.userId && fileInfo.userId != null && fileInfo.userId !== undefined) {
        userId = fileInfo.userId;
      }
      const imageType = fileInfo.imageType;
      let defaultContent = false;

      let prefix = 'assets';
      let userLevelDetails = false;

      if (fileInfo.programCode && userId) {
        prefix = `assets/program/${fileInfo.programCode}/users/${userId}`;
        if (imageType) {
          prefix += `/${imageType}`;
        } else {
          defaultContent = true; // Default content if no image type is provided
        }
      } else if (userId) {
        userLevelDetails = true;
        prefix = `assets/users/${userId}`;
        if (imageType) {
          prefix += `/${imageType}`;
        } else {
          defaultContent = true; // Default content if no image type is provided
        }
      }
      const data = await this.awsS3Service.generateUploadUrl(
        fileInfo.fileName,
        fileInfo.contentType,
        prefix,
        userLevelDetails,
        defaultContent  
      );

      return {
        statusCode: HttpStatus.OK,
        message: 'Successfully retrieved file upload URL',
        data,
      };
    } catch (err) {
      this.logger.error('Error in generating file upload URL', err);
      handleKnownErrors(ERROR_CODES.FAILED_TO_GENERATE_FILE_URLS, err);
    }
  }
}
