import {
    Injectable,
    CanActivate,
    ExecutionContext,
    UnauthorizedException,
  } from '@nestjs/common';
  import { FirebaseAuthService } from './firebase-auth.service';
import { UserService } from 'src/user/user.service';
import { ROLE_KEYS } from 'src/common/constants/strings-constants';
import { AppLoggerService } from 'src/common/services/logger.service';
  
  @Injectable()
  export class FirebaseAuthGuard implements CanActivate {
    constructor(
      private readonly firebaseAuthService: FirebaseAuthService,
      private readonly userService: UserService,
      private readonly logger: AppLoggerService,
    ) {}

    /**
     * Gets the highest priority role from user's role mappings
     * @param user - The user object with role mappings
     * @returns the user role map with the highest priority (lowest priority number)
     */
    private getHighestPriorityRole(user: any): any {
      if (!user.userRoleMaps || user.userRoleMaps.length === 0) {
        return null;
      }

      // Filter out roles without valid role objects
      const validRoleMaps = user.userRoleMaps.filter(
        (urm) => urm.role && typeof urm.role.name === 'string' && urm.role.name.length > 0
      );

      if (validRoleMaps.length === 0) {
        return null;
      }

      // Sort by priority (ascending - lower numbers = higher priority) and return the first one
      const highestPriorityRoleMap = validRoleMaps.sort((a, b) => {
        const priorityA = a.role.priority || 999;
        const priorityB = b.role.priority || 999;
        return priorityA - priorityB;
      })[0];

      this.logger.debug('Highest priority role selected', {
        userId: user.id,
        selectedRole: {
          name: highestPriorityRoleMap.role.name,
          priority: highestPriorityRoleMap.role.priority,
          roleKey: highestPriorityRoleMap.role.roleKey,
          isSystemRole: highestPriorityRoleMap.role.isSystemRole
        },
        allAvailableRoles: validRoleMaps.map(urm => ({
          name: urm.role.name,
          priority: urm.role.priority || 999,
          roleKey: urm.role.roleKey,
          isSystemRole: urm.role.isSystemRole
        }))
      });

      return highestPriorityRoleMap;
    }

    /**
     * Validates active role header and checks if user has the specified role
     * @param activeRole - The role key to validate
     * @param user - The user object with role mappings
     * @returns boolean indicating if the role key is valid and user has that role
     */
    private validateRoleKeyHeader(activeRole: string, user: any): boolean {
      if (!activeRole || !user.userRoleMaps) {
        return false;
      }

      // Get all valid role keys from the ROLE_KEYS enum
      const validRoleKeys = Object.values(ROLE_KEYS);
      
      // Check if the provided role key is valid
      if (!validRoleKeys.includes(activeRole)) {
        this.logger.warn('Invalid role key provided in activerole header', {
          userId: user.id,
          providedRoleKey: activeRole,
          validRoleKeys,
          userAvailableRoles: user.userRoleMaps.map(urm => ({
            roleName: urm.role?.name,
            roleKey: urm.role?.roleKey
          }))
        });
        return false;
      }

      // Check if user has this specific role
      const hasRole = user.userRoleMaps.some(
        (urm) => urm.role && urm.role.roleKey === activeRole
      );

      if (!hasRole) {
        this.logger.warn('User attempted to use role they do not have', {
          userId: user.id,
          userName: user.fullName || user.firstName || 'Unknown',
          attemptedRoleKey: activeRole,
          userAvailableRoles: user.userRoleMaps.map(urm => ({
            roleName: urm.role?.name,
            roleKey: urm.role?.roleKey
          }))
        });
      }

      return hasRole;
    }

    /**
     * Filters user roles based on the provided role key header
     * @param user - The user object with role mappings
     * @param roleKey - The specific role key to filter by
     * @returns filtered user object with only the specified role
     */
    private filterUserRolesByKey(user: any, roleKey: string): any {
      const filteredRoleMaps = user.userRoleMaps.filter(
        (urm) => urm.role && urm.role.roleKey === roleKey
      );

      return {
        ...user,
        userRoleMaps: filteredRoleMaps,
        roles: filteredRoleMaps.map((urm) => urm.role?.name).filter(Boolean),
      };
    }

    /**
     * Validates Firebase authentication token and authorizes user access
     * @param context - The execution context containing request information
     * @returns Promise<boolean> indicating if the user is authenticated and authorized
     */
    async canActivate(context: ExecutionContext): Promise<boolean> {
      const request = context.switchToHttp().getRequest();
      const firebaseToken =
        request.headers.authorization?.split('Bearer ')[1]?.trim();
      const userId = request.headers.userid;
      const activeRole = request.headers.activerole?.trim();
      
      if (!firebaseToken || !userId) {
        throw new UnauthorizedException('Missing authentication headers');
      }
  
      const decodedToken = await this.firebaseAuthService.verifyToken(
        firebaseToken,
        userId,
      );
      const user = await this.userService.getUserByCombinedPhoneNumber(decodedToken.phone_number);
      if (!user) {
        throw new UnauthorizedException('User not found');
      }

      // Ensure userRoleMaps is an array
      if (!Array.isArray(user.userRoleMaps)) {
        user.userRoleMaps = [];
      }

      // Log initial user authentication for audit purposes
      this.logger.log('Firebase authentication successful - initial role detection', {
        userId: user.id,
        userName: user.fullName || user.firstName || 'Unknown',
        userEmail: user.email || 'No email',
        userMobile: user.phoneNumber || 'No mobile',
        firebaseUid: decodedToken.uid,
        phoneNumber: decodedToken.phone_number,
        availableRoles: user.userRoleMaps.map(urm => ({
          roleName: urm.role?.name,
          roleKey: urm.role?.roleKey,
          roleId: urm.role?.id,
          priority: urm.role?.priority,
          isSystemRole: urm.role?.isSystemRole
        })),
        totalRolesCount: user.userRoleMaps.length,
        requestedActiveRole: activeRole || 'none',
        ipAddress: request.ip,
        userAgent: request.headers['user-agent'],
        endpoint: request.originalUrl,
        method: request.method
      });

      // If active role is provided, validate and filter roles
      if (activeRole) {
        const isValidRoleKey = this.validateRoleKeyHeader(activeRole, user);
        if (!isValidRoleKey) {
          throw new UnauthorizedException('Invalid active role or user does not have the specified role');
        }
        
        // Filter user object to only include the specified role
        const filteredUser = this.filterUserRolesByKey(user, activeRole);
        
        // Log successful role switching
        this.logger.log('Role switching successful - user switched to specific role', {
          userId: user.id,
          userName: user.fullName || user.firstName || 'Unknown',
          authMethod: 'FIREBASE',
          switchedToRole: {
            roleKey: activeRole,
            roleName: filteredUser.userRoleMaps[0]?.role?.name,
            roleId: filteredUser.userRoleMaps[0]?.role?.id,
            priority: filteredUser.userRoleMaps[0]?.role?.priority,
            isSystemRole: filteredUser.userRoleMaps[0]?.role?.isSystemRole
          },
          previousAvailableRolesCount: user.userRoleMaps.length,
          currentActiveRolesCount: filteredUser.userRoleMaps.length,
          sessionInfo: {
            ipAddress: request.ip,
            userAgent: request.headers['user-agent'],
            endpoint: request.originalUrl,
            method: request.method,
            timestamp: new Date().toISOString()
          },
          auditInfo: {
            action: 'ROLE_SWITCH',
            category: 'AUTHENTICATION',
            description: `User ${user.id} switched to role ${activeRole} via Firebase auth`
          }
        });

        request.user = filteredUser;
        return true;
      }

      // Get the highest priority role when no active role is specified
      const highestPriorityRoleMap = this.getHighestPriorityRole(user);

      if (!highestPriorityRoleMap) {
        // Log when default viewer role is assigned
        this.logger.log('No valid roles found - assigning default viewer role', {
          userId: user.id,
          userName: user.fullName || user.firstName || 'Unknown',
          authMethod: 'FIREBASE',
          originalUserRole: user.role,
          assignedDefaultRole: 'viewer',
          auditInfo: {
            action: 'DEFAULT_ROLE_ASSIGNMENT',
            category: 'AUTHENTICATION',
            description: `User ${user.id} assigned default viewer role due to no valid roles via Firebase auth`
          }
        });

        user.userRoleMaps.push({
          role: {
            name: user.role || 'viewer',
            id: 0,
            roleKey: '',
            priority: 999,
            isSystemRole: false,
            createdAt: new Date(),
            updatedAt: new Date(),
            userRoleMaps: [],
          },
          id: 0,
          user: user,
          createdAt: new Date(),
          updatedAt: new Date(),
        });

        request.user = {
          ...user,
          roles: [user.role || 'viewer'],
        };
      } else {
        // Return user with only the highest priority role
        const filteredUser = {
          ...user,
          userRoleMaps: [highestPriorityRoleMap],
          roles: [highestPriorityRoleMap.role.name],
        };

        // Log successful authentication with highest priority role
        this.logger.log('User authenticated with highest priority role', {
          userId: user.id,
          userName: user.fullName || user.firstName || 'Unknown',
          authMethod: 'FIREBASE',
          selectedHighestPriorityRole: {
            name: highestPriorityRoleMap.role.name,
            priority: highestPriorityRoleMap.role.priority,
            roleKey: highestPriorityRoleMap.role.roleKey,
            isSystemRole: highestPriorityRoleMap.role.isSystemRole,
            roleId: highestPriorityRoleMap.role.id
          },
          totalAvailableRolesCount: user.userRoleMaps.length,
          allAvailableRoles: user.userRoleMaps.map(urm => ({
            name: urm.role?.name,
            priority: urm.role?.priority,
            roleKey: urm.role?.roleKey,
            isSystemRole: urm.role?.isSystemRole
          })).filter(role => role.name),
          auditInfo: {
            action: 'HIGHEST_PRIORITY_ROLE_ACCESS',
            category: 'AUTHENTICATION',
            description: `User ${user.id} authenticated with highest priority role: ${highestPriorityRoleMap.role.name} via Firebase auth`
          }
        });

        request.user = filteredUser;
      }
      
      return true;
    }
  }