import { CanActivate, ExecutionContext, Injectable, UnauthorizedException } from '@nestjs/common';
import { AuthService } from './auth.service';
import { FirebaseAuthService } from './firebase-auth.service';
import { UserService } from 'src/user/user.service';
import { ROLE_KEYS } from 'src/common/constants/strings-constants';
import { AppLoggerService } from 'src/common/services/logger.service';

@Injectable()
export class CombinedAuthGuard implements CanActivate {
  constructor(
    private readonly authService: AuthService,
    private readonly firebaseAuthService: FirebaseAuthService,
    private readonly userService: UserService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Gets the highest priority role from user's role mappings
   * @param user - The user object with role mappings
   * @returns the user role map with the highest priority (lowest priority number)
   */
  private getHighestPriorityRole(user: any): any {
    if (!user.userRoleMaps || user.userRoleMaps.length === 0) {
      return null;
    }

    // Filter out roles without valid role objects
    const validRoleMaps = user.userRoleMaps.filter(
      (urm) => urm.role && typeof urm.role.name === 'string' && urm.role.name.length > 0
    );

    if (validRoleMaps.length === 0) {
      return null;
    }

    // Sort by priority (ascending - lower numbers = higher priority) and return the first one
    const highestPriorityRoleMap = validRoleMaps.sort((a, b) => {
      const priorityA = a.role.priority || 999;
      const priorityB = b.role.priority || 999;
      return priorityA - priorityB;
    })[0];

    this.logger.debug('Highest priority role selected', {
      userId: user.id,
      selectedRole: {
        name: highestPriorityRoleMap.role.name,
        priority: highestPriorityRoleMap.role.priority,
        roleKey: highestPriorityRoleMap.role.roleKey,
        isSystemRole: highestPriorityRoleMap.role.isSystemRole
      },
      allAvailableRoles: validRoleMaps.map(urm => ({
        name: urm.role.name,
        priority: urm.role.priority || 999,
        roleKey: urm.role.roleKey,
        isSystemRole: urm.role.isSystemRole
      }))
    });

    return highestPriorityRoleMap;
  }

  /**
   * Validates active role header and checks if user has the specified role
   * @param activeRole - The role key to validate
   * @param user - The user object with role mappings
   * @returns boolean indicating if the role key is valid and user has that role
   */
  private validateRoleKeyHeader(activeRole: string, user: any): boolean {
    if (!activeRole || !user.userRoleMaps) {
      return false;
    }

    // Get all valid role keys from the ROLE_KEYS enum
    const validRoleKeys = Object.values(ROLE_KEYS);
    
    // Check if the provided role key is valid
    if (!validRoleKeys.includes(activeRole)) {
      this.logger.warn('Invalid role key provided in activerole header', {
        userId: user.id,
        providedRoleKey: activeRole,
        validRoleKeys,
        userAvailableRoles: user.userRoleMaps.map(urm => ({
          roleName: urm.role?.name,
          roleKey: urm.role?.roleKey
        }))
      });
      return false;
    }

    // Check if user has this specific role
    const hasRole = user.userRoleMaps.some(
      (urm) => urm.role && urm.role.roleKey === activeRole
    );

    if (!hasRole) {
      this.logger.warn('User attempted to use role they do not have', {
        userId: user.id,
        userName: user.fullName || user.firstName || 'Unknown',
        attemptedRoleKey: activeRole,
        userAvailableRoles: user.userRoleMaps.map(urm => ({
          roleName: urm.role?.name,
          roleKey: urm.role?.roleKey
        }))
      });
    }

    return hasRole;
  }

  /**
   * Filters user roles based on the provided role key header
   * @param user - The user object with role mappings
   * @param roleKey - The specific role key to filter by
   * @returns filtered user object with only the specified role
   */
  private filterUserRolesByKey(user: any, roleKey: string): any {
    const filteredRoleMaps = user.userRoleMaps.filter(
      (urm) => urm.role && urm.role.roleKey === roleKey
    );

    return {
      ...user,
      userRoleMaps: filteredRoleMaps,
      roles: filteredRoleMaps.map((urm) => urm.role?.name).filter(Boolean),
    };
  }

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const request = context.switchToHttp().getRequest();
    const token = request.headers.authorization?.split('Bearer ')[1]?.trim();
    const activeRole = request.headers.activerole?.trim();

    if (!token) {
      throw new UnauthorizedException('Missing authentication token');
    }

    let user;

    // Check if token ends with 'custom'
    if (token.endsWith('custom')) {
      // OTP Authentication
      const clearedToken = token.replace(' custom', '');
      const session = await this.authService.validateJwt(clearedToken);
      if (!session || !session.user) {
        throw new UnauthorizedException('Invalid token');
      }
      user = session.user;
    } else {
      // Firebase Authentication
      const userId = request.headers.userid;
      if (!userId) {
        throw new UnauthorizedException('Missing userId header');
      }

      const decodedToken = await this.firebaseAuthService.verifyToken(token, userId);
      user = await this.userService.getUserByCombinedPhoneNumber(decodedToken.phone_number);
      
      if (!user) {
        throw new UnauthorizedException('User not found');
      }
    }

    // Common user role mapping logic
    if (!Array.isArray(user.userRoleMaps)) {
      user.userRoleMaps = [];
    }

    // Log initial user roles for audit purposes
    this.logger.log('User authentication successful - initial role detection', {
      userId: user.id,
      userName: user.fullName || user.firstName || 'Unknown',
      userEmail: user.emailAddress || 'No email',
      userMobile: user.mobileNumber || 'No mobile',
      availableRoles: user.userRoleMaps.map(urm => ({
        roleName: urm.role?.name,
        roleKey: urm.role?.roleKey,
        roleId: urm.role?.id,
        priority: urm.role?.priority,
        isSystemRole: urm.role?.isSystemRole
      })),
      totalRolesCount: user.userRoleMaps.length,
      requestedActiveRole: activeRole || 'none',
      ipAddress: request.ip,
      userAgent: request.headers['user-agent'],
      endpoint: request.originalUrl,
      method: request.method
    });

    // If active role is provided, validate and filter roles
    if (activeRole) {
      const isValidRoleKey = this.validateRoleKeyHeader(activeRole, user);
      if (!isValidRoleKey) {
        throw new UnauthorizedException('Invalid active role or user does not have the specified role');
      }
      
      // Filter user object to only include the specified role
      const filteredUser = this.filterUserRolesByKey(user, activeRole);
      
      // Log successful role switching
      this.logger.log('Role switching successful - user switched to specific role', {
        userId: user.id,
        userName: user.fullName || user.firstName || 'Unknown',
        switchedToRole: {
          roleKey: activeRole,
          roleName: filteredUser.userRoleMaps[0]?.role?.name,
          roleId: filteredUser.userRoleMaps[0]?.role?.id,
          priority: filteredUser.userRoleMaps[0]?.role?.priority,
          isSystemRole: filteredUser.userRoleMaps[0]?.role?.isSystemRole
        },
        previousAvailableRolesCount: user.userRoleMaps.length,
        currentActiveRolesCount: filteredUser.userRoleMaps.length,
        sessionInfo: {
          ipAddress: request.ip,
          userAgent: request.headers['user-agent'],
          endpoint: request.originalUrl,
          method: request.method,
          timestamp: new Date().toISOString()
        },
        auditInfo: {
          action: 'ROLE_SWITCH',
          category: 'AUTHENTICATION',
          description: `User ${user.id} switched to role ${activeRole}`
        }
      });

      request.user = filteredUser;
      return true;
    }

    // Get the highest priority role when no active role is specified
    const highestPriorityRoleMap = this.getHighestPriorityRole(user);

    if (!highestPriorityRoleMap) {
      // Log when default viewer role is assigned
      this.logger.log('No valid roles found - assigning default viewer role', {
        userId: user.id,
        userName: user.fullName || user.firstName || 'Unknown',
        originalUserRole: user.role,
        assignedDefaultRole: 'viewer',
        auditInfo: {
          action: 'DEFAULT_ROLE_ASSIGNMENT',
          category: 'AUTHENTICATION',
          description: `User ${user.id} assigned default viewer role due to no valid roles`
        }
      });

      user.userRoleMaps.push({
        role: {
          name: user.role || 'viewer',
          id: 0,
          roleKey: '',
          createdAt: new Date(),
          updatedAt: new Date(),
          userRoleMaps: [],
        },
        id: 0,
        user: user,
        createdAt: new Date(),
        updatedAt: new Date(),
      });

      request.user = {
        ...user,
        roles: [user.role || 'viewer'],
      };
    } else {
      // Return user with only the highest priority role
      const filteredUser = {
        ...user,
        userRoleMaps: [highestPriorityRoleMap],
        roles: [highestPriorityRoleMap.role.name],
      };

      // Log successful authentication with highest priority role
      this.logger.log('User authenticated with highest priority role', {
        userId: user.id,
        userName: user.fullName || user.firstName || 'Unknown',
        selectedHighestPriorityRole: {
          name: highestPriorityRoleMap.role.name,
          priority: highestPriorityRoleMap.role.priority,
          roleKey: highestPriorityRoleMap.role.roleKey,
          isSystemRole: highestPriorityRoleMap.role.isSystemRole,
          roleId: highestPriorityRoleMap.role.id
        },
        totalAvailableRolesCount: user.userRoleMaps.length,
        allAvailableRoles: user.userRoleMaps.map(urm => ({
          name: urm.role?.name,
          priority: urm.role?.priority,
          roleKey: urm.role?.roleKey,
          isSystemRole: urm.role?.isSystemRole
        })).filter(role => role.name),
        auditInfo: {
          action: 'HIGHEST_PRIORITY_ROLE_ACCESS',
          category: 'AUTHENTICATION',
          description: `User ${user.id} authenticated with highest priority role: ${highestPriorityRoleMap.role.name}`
        }
      });

      request.user = filteredUser;
    }

    return true;
  }
}