import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as jwt from 'jsonwebtoken';
import { AuthRepository } from './auth.repository';
import { InitiateLoginDto } from './dto/initiate-login.dto';
import { ValidateOtpDto } from './dto/validate-otp.dto';
import { LoginTypeEnum } from 'src/common/enum/login-type.enum';
import { UserService } from 'src/user/user.service';
import { UserSessionStatusEnum } from 'src/common/enum/user-session-status.enum';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { UserSession, User, UserProfileExtension } from 'src/common/entities';
import { CommunicationService } from 'src/communication/communication.service';
import { zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { Msg91Service } from 'src/common/services/msg91.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';

const DEV_PHONE_NUMBERS = ['6666666666', '7777777777', '8888888888', '9999999999', '8291673037', '1234567890'];
const DEV_EMAILS = [
  'mahatria@infinitheism.com',
  'admin@infinitheism.com',
  'finance_manager@infinitheism.com',
  'relational_manager@infinitheism.com',
  'shoba@infinitheism.com',
  'rm@infinitheism.com',
  'operational_manger@infinitheism.com',
  'prabhakar@divami.com',
];
const MAX_OTP_ATTEMPTS = 10;
const OTP_EXPIRY_MINUTES = 11;

@Injectable()
export class AuthService {
  constructor(
    private readonly repo: AuthRepository,
    private readonly userService: UserService,
    private readonly configService: ConfigService,
    private readonly communicationService: CommunicationService,
    private readonly msg91Service: Msg91Service,
    private readonly logger: AppLoggerService,
  ) {
    this.msg91OtpTemplateId = this.configService.get<string>('MSG91_OTP_TEMPLATE_ID') ?? '';
  }
  private msg91OtpTemplateId: string;

  private isOTPSimulated(dto: InitiateLoginDto): boolean {
    const env = this.configService.get<string>('NODE_ENV') || 'development';
    if (
      (env === 'development' || env === 'qa' || env === 'qa_auto' || env === 'qa_perf' || env === 'uat') &&
      (DEV_PHONE_NUMBERS.includes(dto.phoneNumber || '') ||
        DEV_EMAILS.includes(dto.email || '') ||
        dto.phoneNumber?.startsWith('12345') ||
        dto.email?.startsWith('infitest-'))
    ) {
      return true;
    }
    return false;
  }

  private isOTPSendSimulated(dto: InitiateLoginDto): boolean {
    const env = this.configService.get<string>('NODE_ENV') || 'development';
    if (
      (env === 'development' || env === 'qa' || env === 'qa_auto' || env === 'uat') &&
      (DEV_PHONE_NUMBERS.includes(dto.phoneNumber || '') ||
        DEV_EMAILS.includes(dto.email || '') ||
        dto.phoneNumber?.startsWith('12345') ||
        dto.email?.startsWith('infitest-'))
    ) {
      return true;
    }
    return false;
  }

  private generateOtp(dto: InitiateLoginDto): string {
    return this.isOTPSimulated(dto)
      ? '123456'
      : Math.floor(100000 + Math.random() * 900000).toString();
  }

  /**
   * Sends OTP via SMS or Email based on login type
   * @param dto - Login details (phone/email)
   * @param otp - Generated OTP
   * @param user - User object (if exists)
   */
  private async sendOtp(dto: InitiateLoginDto, otp: string, user?: User | null): Promise<void> {
    try {
      if (this.isOTPSendSimulated(dto)) {
        return; // Simulated, do not send
      }
      if (dto.loginType === LoginTypeEnum.EMAIL && dto.email) {
        await this.communicationService.sendSingleEmail({
          templateKey: process.env.ZEPTO_OTP_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_NOREPLY_EMAIL,
            name: zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: dto.email,
            name: user?.fullName || dto.email,
          },
          subject: 'Login OTP - infinitheism',
          mergeInfo: {
            otp: otp,
            userName: user?.fullName ? ` ${user?.fullName?.trim()}` : '...',
          },
          trackinfo: {
            registrationId: user?.id || 0,
          },
        });
      }

      if (dto.loginType === LoginTypeEnum.PHONE && dto.phoneNumber && dto.countryCode) {
        const formattedPhoneNumber = `${dto.countryCode.replace('+', '')}${dto.phoneNumber}`;
        await this.msg91Service.sendSms(
          formattedPhoneNumber,
          {
            var1: otp,
          },
          this.msg91OtpTemplateId,
        );
      }
    } catch (error) {
      if (this.isOTPSimulated(dto)) {
        // Do nothing, as this is a simulated OTP
        return;
      } else {
        this.logger.error('Error sending OTP', '', { dto, error });
        handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
      }
    }
  }

  async initiateLogin(dto: InitiateLoginDto, userAgent: string, ipAddress: string): Promise<void> {
    let user: User | null = null;
    if (dto.loginType === LoginTypeEnum.PHONE && dto.phoneNumber && dto.countryCode) {
      user = await this.userService.getUserByCountryCodeAndPhoneNumber(
        dto.countryCode,
        dto.phoneNumber,
      );
    } else if (dto.loginType === LoginTypeEnum.EMAIL && dto.email) {
      user = await this.userService.getUserByEmail(dto.email);
    } else {
      throw new InifniBadRequestException(ERROR_CODES.AUTH_INVALID_LOGIN_TYPE) as Error;
    }

    const otp = this.generateOtp(dto);
    await this.repo.createSession({
      userId: user ? user.id : undefined,
      phoneNumber: dto.phoneNumber,
      countryCode: dto.countryCode,
      email: dto.email,
      otp,
      loginAt: new Date(),
      expiresAt: new Date(Date.now() + OTP_EXPIRY_MINUTES * 60 * 1000),
      attemptCount: 0,
      sessionStatus: UserSessionStatusEnum.INITIATED,
      userAgent,
      ipAddress,
    });

    // Send OTP using the helper method
    await this.sendOtp(dto, otp, user);
  }

  async resendOtp(dto: InitiateLoginDto, userAgent: string, ipAddress: string): Promise<void> {
    const session = await this.repo.findLatestSession(dto.phoneNumber, dto.countryCode, dto.email);
    if (!session) {
      throw new InifniNotFoundException(ERROR_CODES.USER_SESSION_NOTFOUND, null, null, '') as Error;
    }

    // Get user info for personalized messaging
    let user: User | null = session.user;
    if (!user) {
      if (dto.loginType === LoginTypeEnum.PHONE && dto.phoneNumber && dto.countryCode) {
        user = await this.userService.getUserByCountryCodeAndPhoneNumber(
          dto.countryCode,
          dto.phoneNumber,
        );
      } else if (dto.loginType === LoginTypeEnum.EMAIL && dto.email) {
        user = await this.userService.getUserByEmail(dto.email);
      }
    }

    // Check if we need to regenerate OTP
    let regenerate = false;
    let otp = session.otp;

    if (session.attemptCount >= MAX_OTP_ATTEMPTS) {
      regenerate = true;
      await this.repo.updateSession(session.id, { sessionStatus: UserSessionStatusEnum.FAILED });
    } else if (!session.expiresAt || session.expiresAt.getTime() < Date.now()) {
      regenerate = true;
      await this.repo.updateSession(session.id, { sessionStatus: UserSessionStatusEnum.EXPIRED });
    }

    if (regenerate) {
      otp = this.generateOtp(dto);
      await this.repo.createSession({
        userId: session.userId,
        phoneNumber: dto.phoneNumber,
        countryCode: dto.countryCode,
        email: dto.email,
        otp,
        loginAt: new Date(),
        expiresAt: new Date(Date.now() + OTP_EXPIRY_MINUTES * 60 * 1000),
        attemptCount: 0,
        sessionStatus: UserSessionStatusEnum.INITIATED,
        userAgent,
        ipAddress,
      });
    }

    // Send OTP via SMS or Email
    await this.sendOtp(dto, otp, user);
  }

  async validateOtp(dto: ValidateOtpDto): Promise<{ token: string; user: User, roles: string[], userProfileExtension?: any }> {
    const session = await this.repo.findLatestSession(dto.phoneNumber, dto.countryCode, dto.email);
    if (!session) {
      throw new InifniNotFoundException(ERROR_CODES.USER_SESSION_NOTFOUND, null, null, '') as Error;
    }
    if (session.attemptCount >= MAX_OTP_ATTEMPTS) {
      await this.repo.updateSession(session.id, {
        sessionStatus: UserSessionStatusEnum.FAILED,
      });
      throw new InifniBadRequestException(ERROR_CODES.OTP_ATTEMPTS_EXCEEDED) as Error;
    }
    if (!session.expiresAt || session.expiresAt.getTime() < Date.now()) {
      await this.repo.updateSession(session.id, {
        sessionStatus: UserSessionStatusEnum.EXPIRED,
      });
      throw new InifniBadRequestException(ERROR_CODES.OTP_EXPIRED) as Error;
    }
    if (session.otp !== dto.otp) {
      await this.repo.updateSession(session.id, { attemptCount: session.attemptCount + 1 });
      throw new InifniBadRequestException(ERROR_CODES.AUTH_INVALID_OTP) as Error;
    }

    let user: User | null = session?.user ?? null;
    let userCreated = false;
    if (!user) {
      user = await this.userService.createMinimalUser({
        phoneNumber: dto.phoneNumber,
        countryCode: dto.countryCode,
        email: dto.email,
      });
      userCreated = true;
    }
    if (!user) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, '') as Error;
    }

    const roles = await this.userService.getUserRoles(user.id);
    const payload = {
      userId: user.id,
      user: user,
      roles: roles,
    };
    const secret =
      this.configService.get<string>('JWT_SECRET') || 'defaultSecretKey';
    const token = jwt.sign(payload, secret);

    await this.repo.updateSession(session.id, {
      attemptCount: session.attemptCount + 1,
      sessionStatus: UserSessionStatusEnum.VALIDATED,
      jwtToken: token,
      expiresAt: null,
      userId: userCreated ? user.id : session.userId,
    });

    // Fetch user_profile_extension details
    let userProfileExtension: UserProfileExtension | null = null;
    try {
      userProfileExtension = await this.userService.getUserProfileExtensionByUserId(user.id);
    } catch (e) {
      userProfileExtension = null;
    }
    return { token, user, roles, userProfileExtension };
  }

  async validateJwt(token: string): Promise<UserSession | null> {
    const secret =
      this.configService.get<string>('JWT_SECRET') || 'defaultSecretKey';
    try {
      const payload: any = jwt.verify(token, secret);
      return await this.repo.findByJwtToken(payload.userId, token);
    } catch (error) {
      this.logger.error('Error validating JWT token', '', { token, error });
      return null;
    }
  }

  async logout(userId: string, token: string): Promise<void> {
    const session = await this.repo.findByJwtToken(userId, token);
    if (!session) {
      return;
    }
    await this.repo.updateSession(session.id, {
      sessionStatus: UserSessionStatusEnum.LOGGED_OUT,
      jwtToken: null,
      updatedAt: new Date(),
    });
  }
}
