import {
  Body,
  Controller,
  HttpCode,
  HttpStatus,
  Post,
  Req,
  Res,
  UseGuards,
  UsePipes,
  ValidationPipe,
} from '@nestjs/common';
import { Request, Response } from 'express';
import { InitiateLoginDto } from './dto/initiate-login.dto';
import { ValidateOtpDto } from './dto/validate-otp.dto';
import { AuthService } from './auth.service';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { authConstMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import {
  ApiBearerAuth,
  ApiBody,
  ApiOperation,
  ApiResponse,
  ApiSecurity,
  ApiTags,
} from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { CombinedAuthGuard } from './combined-auth.guard';

@ApiTags('auth')
@Controller('auth')
export class AuthController {
  constructor(
    private readonly authService: AuthService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Post('login')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: authConstMessages.LOGIN_ATTEMPT })
  @ApiBody({ type: InitiateLoginDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.OK })
  async initiateLogin(@Body() dto: InitiateLoginDto, @Req() req: Request, @Res() res: Response) {
    this.logger.log(authConstMessages.LOGIN_ATTEMPT, dto);
    try {
      // Extract user agent and IP address
      const userAgent = req.headers['user-agent'] || '';
      const ipAddress = req.ip || req.socket.remoteAddress || '';

      // Pass them to the service (add to DTO or as extra params)
      await this.authService.initiateLogin(dto, userAgent, ipAddress);
      await this.responseService.success(res, authConstMessages.OTP_SENT, undefined, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post('resend-otp')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: authConstMessages.RESEND_OTP })
  @ApiBody({ type: InitiateLoginDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.SUCCESS })
  async resendOtp(@Body() dto: InitiateLoginDto, @Req() req: Request, @Res() res: Response) {
    this.logger.log(authConstMessages.RESEND_OTP, dto);
    try {
      // Extract user agent and IP address
      const userAgent = req.headers['user-agent'] || '';
      const ipAddress = req.ip || req.socket.remoteAddress || '';

      await this.authService.resendOtp(dto, userAgent, ipAddress);
      await this.responseService.success(res, authConstMessages.OTP_SENT, {}, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post('validate-otp')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: authConstMessages.VALIDATE_OTP })
  @ApiBody({ type: ValidateOtpDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async validateOtp(@Body() dto: ValidateOtpDto, @Res() res: Response) {
    this.logger.log(authConstMessages.VALIDATE_OTP, dto);
    try {
      const { token, user, roles, userProfileExtension } = await this.authService.validateOtp(dto);
      res.setHeader('Authorization', `Bearer ${token}`);
      await this.responseService.success(
        res,
        authConstMessages.OTP_VALIDATED,
        { userId: user.id, user, token, roles, userProfileExtension },
        HttpStatus.OK
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @UseGuards(CombinedAuthGuard, RolesGuard)
  @Roles('admin', 'viewer', 'finance_manager', 'relational_manager', 'shoba', 'rm', 'operational_manger', 'rm_support','mahatria')
  @ApiBearerAuth('Authorization')
  @ApiSecurity('userIdAuth')
  @Post('logout')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: authConstMessages.LOGOUT })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async logout(@Req() req: any, @Res() res: Response) {
    const userId = req.user?.id;
    this.logger.log(authConstMessages.LOGOUT);
    try {
      const token = req.headers.authorization?.split('Bearer ')[1]?.trim();
      await this.authService.logout(userId, token);
      await this.responseService.success(res, authConstMessages.LOGOUT_SUCCESS);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
