import {
  EventSubscriber,
  EntitySubscriberInterface,
  InsertEvent,
  UpdateEvent,
  RemoveEvent,
  DataSource,
  EntityMetadata,
  EntityManager,
} from 'typeorm';
import { AuditHistoryLogDetail } from 'src/common/entities/audit-history-log-detail.entity';
import { AuditHistoryLog } from 'src/common/entities/audit-history-log.entity';
import { isAuditable, getAuditEntityName } from '../decorators/auditable.decorator';
import { getSkipAuditFields } from '../decorators/skip-audit.decorator';
import { AuditHistoryContext } from '../interfaces/audit-history-context.interface';
import { Injectable } from '@nestjs/common';
import {
  AUDIT_CONTEXT_KEY,
  AuditHistoryAction,
  AuditHistoryLogType,
} from '../audit-history.constants';

@Injectable()
@EventSubscriber()
export class AuditHistorySubscriber implements EntitySubscriberInterface {
  constructor(dataSource: DataSource) {
    dataSource.subscribers.push(this);
  }

  private getAuditHistoryContext(event: any): AuditHistoryContext | undefined {
    return event.queryRunner?.data?.[AUDIT_CONTEXT_KEY];
  }

  private getEntityId(entity: any, metadata: EntityMetadata): string {
    if (entity.id) return String(entity.id);

    // Handle composite keys
    const primaryColumns = metadata.primaryColumns;
    if (primaryColumns.length === 1) {
      return String(entity[primaryColumns[0].propertyName]);
    }

    // For composite keys, create a concatenated string
    return primaryColumns.map((col) => `${col.propertyName}:${entity[col.propertyName]}`).join(',');
  }

  private async createAuditLog(
    event: any,
    action: AuditHistoryAction,
    entity: any,
    metadata: EntityMetadata,
    details: { fieldName: string; oldValue: any; newValue: any; fieldType: string }[] = [],
  ): Promise<void> {
    const context = this.getAuditHistoryContext(event);
    // Always use metadata.target for consistent entity class reference
    const entityClass = metadata.target;
    const entityName = getAuditEntityName(entityClass);
    const entityType = (entityClass as any).name || 'Unknown';
    const entityId = this.getEntityId(entity, metadata);
    const parentRefId = entity?.parentRefId || this.determineParentRefId(entity);

    // Debug logging to help identify parentRefId issues

    const auditLog = new AuditHistoryLog();
    auditLog.action = action;
    auditLog.entityType = entityType;
    auditLog.entityName = entityName;
    auditLog.entityId = entityId;
    auditLog.parentRefId = parentRefId;
    auditLog.type = AuditHistoryLogType.AUDIT;
    auditLog.userId = context?.userId || null;
    auditLog.ipAddress = context?.ipAddress || null;
    auditLog.userAgent = context?.userAgent || null;
    auditLog.apiEndpoint = context?.apiEndpoint || null;
    auditLog.httpMethod = context?.httpMethod || null;
    auditLog.requestId = context?.requestId || null;
    auditLog.metadata = context ? { ...context } : null;

    await event.queryRunner.manager.save(AuditHistoryLog, auditLog);

    if (details.length > 0) {
      const auditDetails = details.map((detail) => {
        const auditDetail = new AuditHistoryLogDetail();
        auditDetail.auditHistoryLog = auditLog;
        auditDetail.fieldName = detail.fieldName;
        auditDetail.oldValue = detail.oldValue;
        auditDetail.newValue = detail.newValue;
        auditDetail.fieldType = detail.fieldType;
        return auditDetail;
      });

      await event.queryRunner.manager.save(AuditHistoryLogDetail, auditDetails);
    }
  }

  async afterInsert(event: InsertEvent<any>): Promise<void> {
    if (!isAuditable(event.entity)) return;

    const skipFields = getSkipAuditFields(event.entity);
    const details: { fieldName: string; oldValue: any; newValue: any; fieldType: string }[] = [];

    for (const column of event.metadata.columns) {
      const fieldName = column.propertyName;
      if (skipFields.includes(fieldName)) continue;

      const value = event.entity[fieldName];
      if (value !== undefined && value !== null) {
        // For boolean fields, store normalized values (true/false)
        const shouldNormalizeBooleans = this.isBooleanField(null, value);
        const normalizedValue = shouldNormalizeBooleans ? this.normalizeValueForComparison(value) : value;
        
        details.push({
          fieldName,
          oldValue: null,
          newValue: normalizedValue, // Store normalized boolean or original value
          fieldType: column.type.toString(),
        });
      }
    }

    // Set audit fields after creation when ID is available
    const entityId = this.getEntityId(event.entity, event.metadata);
    if (entityId && event.entity.id) {
      const parentRefId = this.determineParentRefId(event.entity);

      // Check if entity has audit fields before trying to update them
      const hasAuditRefId = event.metadata.columns.some((col) => col.propertyName === 'auditRefId');
      const hasParentRefId = event.metadata.columns.some(
        (col) => col.propertyName === 'parentRefId',
      );

      if (hasAuditRefId && hasParentRefId) {
        await event.queryRunner.manager.update(event.metadata.target, event.entity.id, {
          auditRefId: event.entity.id,
          parentRefId: parentRefId,
        });

        // Update the entity instance with the new values for audit logging
        event.entity.auditRefId = event.entity.id;
        event.entity.parentRefId = parentRefId;
      }
    }

    await this.createAuditLog(
      event,
      AuditHistoryAction.INSERT,
      event.entity,
      event.metadata,
      details,
    );
  }

  async loadEntityWithAllRelations(
    manager: EntityManager,
    entityClass: any,
    id: any,
  ): Promise<any> {
    const metadata = manager.connection.getMetadata(entityClass);
    //eliminate createdby,updatedby user
    const skipFields = getSkipAuditFields(entityClass);
    // Filter out skipFields from relations
    const relations = metadata.relations
      .filter((relation) => !skipFields.includes(relation.propertyName))
      .map((relation) => relation.propertyName);

    return await manager.findOne(entityClass, {
      where: { id },
      relations,
    });
  }

  async beforeUpdate(event: UpdateEvent<any>): Promise<void> {
    const entityClass = event.metadata.target as any;
    if (!isAuditable(entityClass)) return;

    const updatedData: any = event.entity || {};

    // Check if this is an audit-only update (only auditRefId and parentRefId are being updated)
    // to prevent infinite loops
    const updatedKeys = Object.keys(updatedData);
    const isAuditOnlyUpdate =
      updatedKeys.length <= 2 &&
      updatedKeys.every((key) => ['auditRefId', 'parentRefId'].includes(key));

    if (isAuditOnlyUpdate) {
      return;
    }

    // Get the entity ID for loading the existing entity
    // Priority: entity.id first, then auditRefId, then try to get ID from databaseEntity
    let entityId = (event.entity as any)?.id;
    if (!entityId && (event.entity as any)?.auditRefId) {
      entityId = (event.entity as any)?.auditRefId;
    }
    if (!entityId && event.databaseEntity) {
      entityId = event.databaseEntity?.id;
    }

    const dbEntity = entityId
      ? (await this.loadEntityWithAllRelations(event.manager, entityClass, entityId)) || {}
      : {};

    // Debug logging for audit data issues

    const skipFields = getSkipAuditFields(entityClass);
    const details: { fieldName: string; oldValue: any; newValue: any; fieldType: string }[] = [];

    for (const column of event.metadata.columns) {
      const fieldName = column.propertyName;
      if (skipFields.includes(fieldName)) continue;

      const oldValue = dbEntity?.[fieldName];
      const newValue =
        updatedData[fieldName] !== undefined ? updatedData[fieldName] : dbEntity?.[fieldName];

      let oldValueUpdated = oldValue;
      let newValueUpdated = newValue;

      // Normalize values for comparison
      oldValueUpdated = this.normalizeValueForComparison(oldValueUpdated);
      newValueUpdated = this.normalizeValueForComparison(newValueUpdated);


      if (oldValueUpdated !== newValueUpdated) {
        // For boolean fields, store normalized values (true/false) instead of original strings
        const shouldNormalizeBooleans = this.isBooleanField(oldValue, newValue);
        
        details.push({
          fieldName,
          oldValue: shouldNormalizeBooleans ? oldValueUpdated : oldValue, // Store normalized boolean or original value
          newValue: shouldNormalizeBooleans ? newValueUpdated : newValue, // Store normalized boolean or original value
          fieldType: column.type.toString(),
        });
      }
    }

    if (details.length > 0) {
      // Merge dbEntity with updatedData to ensure we have all fields including parentRefId
      const entityForAudit = { ...dbEntity, ...updatedData };

      await this.createAuditLog(
        event,
        AuditHistoryAction.UPDATE,
        entityForAudit,
        event.metadata,
        details,
      );
    }
  }
  async afterRemove(event: RemoveEvent<any>): Promise<void> {
    const entityClass = event.metadata.target as any;
    if (!isAuditable(entityClass)) return;

    const entity = event.entity || event.databaseEntity;
    await this.createAuditLog(event, AuditHistoryAction.DELETE, entity, event.metadata);
  }

  private determineParentRefId(entity: any): number | null {
    // For User entity, parentRefId should be null as mentioned in the requirements
    const entityClassName = entity.constructor.name;
    if (entityClassName === 'User') {
      return null;
    }

    // Check if entity has a registrationId field (most common parent reference)
    if (entity.registrationId) {
      return entity.registrationId;
    }

    // Check if entity has a relation.id pattern (for entities like Preference that use relations)
    if (entity.registration?.id) {
      return entity.registration.id;
    }

    // Check if entity has a programRegistrationId field
    if (entity.programRegistrationId) {
      return entity.programRegistrationId;
    }

    // Check if entity has a programRegistration relation
    if (entity.programRegistration?.id) {
      return entity.programRegistration.id;
    }

    // For entities that don't have a clear parent, use their own ID as parentRefId
    // This follows the pattern seen in ProgramRegistration and other root entities
    if (entity.id) {
      return entity.id;
    }

    return null;
  }

  private normalizeValueForComparison(value: any): any {
    // Handle null and undefined
    if (value === null || value === undefined) {
      return value;
    }

    // Handle objects (like entities) - use their ID for comparison if available
    if (typeof value === 'object' && value !== null) {
      // Handle Date objects first
      if (value instanceof Date) {
        return value.toISOString();
      }
      
      // Handle entity objects - use their ID for comparison
      if (value.id !== undefined) {
        return String(value.id);
      }
      
      // For other objects, convert to JSON string for comparison
      try {
        return JSON.stringify(value);
      } catch (error) {
        return String(value);
      }
    }

    // Handle boolean-string conversions
    if (typeof value === 'string') {
      const trimmedValue = value.trim();
      const lowerValue = trimmedValue.toLowerCase();
      
      // Convert "yes"/"no" to boolean
      if (lowerValue === 'yes' || lowerValue === 'true') {
        return true;
      }
      if (lowerValue === 'no' || lowerValue === 'false') {
        return false;
      }
      
      // Handle empty strings
      if (trimmedValue === '') {
        return '';
      }
      
      // Convert string numbers to actual numbers
      if (!isNaN(Number(trimmedValue)) && trimmedValue !== '') {
        return Number(trimmedValue);
      }
      
      // Handle date strings - try to parse and normalize
      if (this.isDateString(trimmedValue)) {
        try {
          const date = new Date(trimmedValue);
          if (!isNaN(date.getTime())) {
            return date.toISOString();
          }
        } catch (error) {
          // If date parsing fails, return original trimmed value
        }
      }
      
      // Return trimmed string for consistent comparison
      return trimmedValue;
    }

    // Handle boolean values - keep as is
    if (typeof value === 'boolean') {
      return value;
    }

    // Handle numbers - keep as is
    if (typeof value === 'number') {
      return value;
    }

    return value;
  }

  private isBooleanField(oldValue: any, newValue: any): boolean {
    // Check if either value is already a boolean
    if (typeof oldValue === 'boolean' || typeof newValue === 'boolean') {
      return true;
    }
    
    // Check if either value is a boolean-like string
    if (typeof oldValue === 'string' || typeof newValue === 'string') {
      const booleanStrings = ['yes', 'no', 'true', 'false'];
      const oldStr = typeof oldValue === 'string' ? oldValue.toLowerCase().trim() : '';
      const newStr = typeof newValue === 'string' ? newValue.toLowerCase().trim() : '';
      
      return booleanStrings.includes(oldStr) || booleanStrings.includes(newStr);
    }
    
    return false;
  }

  private isDateString(value: string): boolean {
    // Don't treat very short strings or purely numeric strings as dates
    if (value.length < 8 || /^\d+$/.test(value)) {
      return false;
    }
    
    // Check if string looks like a date using common patterns
    const datePatterns = [
      /^\d{4}-\d{2}-\d{2}/, // YYYY-MM-DD
      /^\d{2}\/\d{2}\/\d{4}/, // MM/DD/YYYY or DD/MM/YYYY
      /^\d{2}-\d{2}-\d{4}/, // MM-DD-YYYY or DD-MM-YYYY
      /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}/, // ISO datetime
    ];
    
    const matchesPattern = datePatterns.some(pattern => pattern.test(value));
    
    // If it matches a pattern, also check if it's parseable
    if (matchesPattern) {
      return !isNaN(Date.parse(value));
    }
    
    return false;
  }
}
