import { Test, TestingModule } from '@nestjs/testing';
import { DataSource, QueryRunner } from 'typeorm';
import { AuditHistorySubscriber } from './audit-history.subscriber';
import { AuditHistoryLog } from 'src/common/entities/audit-history-log.entity';
import { AuditHistoryLogDetail } from 'src/common/entities/audit-history-log-detail.entity';
import { AuditHistoryAction, AUDIT_CONTEXT_KEY } from '../audit-history.constants';
import { Auditable } from '../decorators/auditable.decorator';
import { SkipAudit } from '../decorators/skip-audit.decorator';

@Auditable({ name: 'TestEntity' })
class TestEntity {
  id: string = '123';
  name: string = 'Test';
  @SkipAudit()
  password: string = 'secret';
}

describe('AuditSubscriber', () => {
  let auditSubscriber: AuditHistorySubscriber;
  let dataSource: DataSource;
  let queryRunner: QueryRunner;
  let mockManager: any;

  beforeEach(async () => {
    mockManager = {
      save: jest.fn().mockResolvedValue({}),
      update: jest.fn().mockResolvedValue({}),
      findOne: jest.fn().mockResolvedValue({}),
      connection: {
        getMetadata: jest.fn().mockReturnValue({
          relations: []
        })
      }
    };

    queryRunner = {
      data: {
        [AUDIT_CONTEXT_KEY]: {
          userId: 'user123',
          ipAddress: '127.0.0.1',
          userAgent: 'Jest Test',
          requestId: 'req123',
        },
      },
      manager: mockManager,
    } as any;

    dataSource = {
      subscribers: [],
    } as any;

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        {
          provide: AuditHistorySubscriber,
          useFactory: () => new AuditHistorySubscriber(dataSource),
        },
      ],
    }).compile();

    auditSubscriber = module.get<AuditHistorySubscriber>(AuditHistorySubscriber);
  });

  describe('afterInsert', () => {
    it('should create audit log for insert action', async () => {
      const entity = new TestEntity();
      const mockMetadata = {
        target: TestEntity,
        columns: [
          { propertyName: 'id', type: 'varchar' },
          { propertyName: 'name', type: 'varchar' },
          { propertyName: 'password', type: 'varchar' },
          { propertyName: 'auditRefId', type: 'varchar' },
          { propertyName: 'parentRefId', type: 'varchar' },
        ],
      } as any;

      const insertEvent = {
        entity,
        metadata: mockMetadata,
        queryRunner,
      } as any;

      await auditSubscriber.afterInsert(insertEvent);

      // Expect update call to set audit fields
      expect(mockManager.update).toHaveBeenCalledTimes(1);
      expect(mockManager.update).toHaveBeenCalledWith(TestEntity, '123', {
        auditRefId: '123',
        parentRefId: '123',
      });

      expect(mockManager.save).toHaveBeenCalledTimes(2);

      const auditLogCall = mockManager.save.mock.calls[0];
      expect(auditLogCall[0]).toBe(AuditHistoryLog);
      expect(auditLogCall[1]).toMatchObject({
        action: AuditHistoryAction.INSERT,
        entityType: 'TestEntity',
        entityName: 'TestEntity',
        entityId: '123',
        parentRefId: '123', // Updated to expect the entity's own ID
        userId: 'user123',
        ipAddress: '127.0.0.1',
        userAgent: 'Jest Test',
        requestId: 'req123',
      });

      const auditDetailCall = mockManager.save.mock.calls[1];
      expect(auditDetailCall[0]).toBe(AuditHistoryLogDetail);
      expect(auditDetailCall[1]).toHaveLength(2); // password field should be skipped
      expect(auditDetailCall[1][0]).toMatchObject({
        fieldName: 'id',
        oldValue: null,
        newValue: '123',
        fieldType: 'varchar',
      });
      expect(auditDetailCall[1][1]).toMatchObject({
        fieldName: 'name',
        oldValue: null,
        newValue: 'Test',
        fieldType: 'varchar',
      });
    });

    it('should skip non-auditable entities', async () => {
      class NonAuditableEntity {
        id: string = '456';
      }

      const entity = new NonAuditableEntity();
      const insertEvent = {
        entity,
        metadata: { columns: [] },
        queryRunner,
      } as any;

      await auditSubscriber.afterInsert(insertEvent);

      expect(mockManager.save).not.toHaveBeenCalled();
      expect(mockManager.update).not.toHaveBeenCalled();
    });

    it('should set correct parentRefId for entities with registrationId', async () => {
      @Auditable({ name: 'TestChildEntity' })
      class TestChildEntity {
        id: string = '456';
        name: string = 'Child Entity';
        registrationId: number = 123;
      }

      const entity = new TestChildEntity();
      const mockMetadata = {
        target: TestChildEntity,
        columns: [
          { propertyName: 'id', type: 'varchar' },
          { propertyName: 'name', type: 'varchar' },
          { propertyName: 'registrationId', type: 'bigint' },
          { propertyName: 'auditRefId', type: 'varchar' },
          { propertyName: 'parentRefId', type: 'varchar' },
        ],
      } as any;

      const insertEvent = {
        entity,
        metadata: mockMetadata,
        queryRunner,
      } as any;

      await auditSubscriber.afterInsert(insertEvent);

      // Expect update call to set audit fields with registrationId as parentRefId
      expect(mockManager.update).toHaveBeenCalledWith(TestChildEntity, '456', {
        auditRefId: '456',
        parentRefId: 123,
      });

      const auditLogCall = mockManager.save.mock.calls[0];
      expect(auditLogCall[1]).toMatchObject({
        action: AuditHistoryAction.INSERT,
        entityType: 'TestChildEntity',
        entityName: 'TestChildEntity',
        entityId: '456',
        parentRefId: 123,
      });
    });
  });

  describe('beforeUpdate', () => {
    it('should create audit log for update action with field changes', async () => {
      const entity = new TestEntity();
      entity.name = 'Updated Test';

      const databaseEntity = new TestEntity();
      databaseEntity.name = 'Original Test';

      const mockMetadata = {
        target: TestEntity,
        columns: [
          { propertyName: 'id', type: 'varchar' },
          { propertyName: 'name', type: 'varchar' },
          { propertyName: 'password', type: 'varchar' },
        ],
      } as any;

      // Mock the loadEntityWithAllRelations method
      jest.spyOn(auditSubscriber, 'loadEntityWithAllRelations').mockResolvedValue(databaseEntity);

      const updateEvent = {
        entity,
        metadata: mockMetadata,
        manager: mockManager,
        queryRunner,
      } as any;

      await auditSubscriber.beforeUpdate(updateEvent);

      expect(mockManager.save).toHaveBeenCalledTimes(2);

      const auditLogCall = mockManager.save.mock.calls[0];
      expect(auditLogCall[1]).toMatchObject({
        action: AuditHistoryAction.UPDATE,
        entityType: 'TestEntity',
        entityId: '123',
        parentRefId: '123', // TestEntity now uses its own ID as parentRefId
      });

      const auditDetailCall = mockManager.save.mock.calls[1];
      expect(auditDetailCall[1]).toHaveLength(1); // Only name changed
      expect(auditDetailCall[1][0]).toMatchObject({
        fieldName: 'name',
        oldValue: 'Original Test',
        newValue: 'Updated Test',
        fieldType: 'varchar',
      });
    });

    it('should not create audit log if no fields changed', async () => {
      const entity = new TestEntity();
      const databaseEntity = new TestEntity();

      const updateEvent = {
        entity,
        databaseEntity,
        metadata: { columns: [] },
        queryRunner,
      } as any;

      await auditSubscriber.beforeUpdate(updateEvent);

      expect(mockManager.save).not.toHaveBeenCalled();
    });

    it('should set auditRefId and parentRefId for ProgramRegistration updates', async () => {
      @Auditable({ name: 'ProgramRegistration' })
      class ProgramRegistration {
        id: number = 123;
        name: string = 'Test Registration';
        auditRefId: number;
        parentRefId: number;
      }

      const entity = new ProgramRegistration();
      const dbEntity = { id: 123, name: 'Old Registration', auditRefId: null, parentRefId: null };
      
      // Mock loadEntityWithAllRelations to return the dbEntity
      jest.spyOn(auditSubscriber, 'loadEntityWithAllRelations').mockResolvedValue(dbEntity);

      const updateEvent = {
        entity: { id: 123, name: 'Test Registration' },
        metadata: {
          target: ProgramRegistration,
          columns: [
            { propertyName: 'id', type: 'number' },
            { propertyName: 'name', type: 'string' },
            { propertyName: 'auditRefId', type: 'number' },
            { propertyName: 'parentRefId', type: 'number' },
          ],
        },
        queryRunner,
        manager: mockManager,
      } as any;

      await auditSubscriber.beforeUpdate(updateEvent);

      // The merged entity should have { id: 123, name: 'Test Registration', auditRefId: null, parentRefId: null }
      // And since it has an id (123), determineParentRefId should return 123
      expect(mockManager.save).toHaveBeenCalledWith(
        AuditHistoryLog,
        expect.objectContaining({
          action: AuditHistoryAction.UPDATE,
          entityType: 'ProgramRegistration',
          entityId: '123',
          parentRefId: 123, // ProgramRegistration uses its own ID as parentRefId
        }),
      );
    });
  });

  describe('afterRemove', () => {
    it('should create audit log for delete action', async () => {
      const entity = new TestEntity();

      const removeEvent = {
        entity,
        metadata: { target: TestEntity },
        queryRunner,
      } as any;

      await auditSubscriber.afterRemove(removeEvent);

      expect(mockManager.save).toHaveBeenCalledTimes(1);
      expect(mockManager.save).toHaveBeenCalledWith(
        AuditHistoryLog,
        expect.objectContaining({
          action: AuditHistoryAction.DELETE,
          entityType: 'TestEntity',
          entityId: '123',
          parentRefId: '123', // TestEntity uses its own ID as parentRefId
        }),
      );
    });
  });
});
