import { ApiProperty } from '@nestjs/swagger';
import { IsNotEmpty, IsOptional, IsString, Matches, IsEnum } from 'class-validator';
import { DocumentTypeEnum, JourneyTypeEnum } from 'src/common/enum/document-type.enum';

export class ExtractionInfo {
  @ApiProperty({
    description: 'S3 URL of the document to be processed',
    example: 'https://bucket-name.s3.amazonaws.com/path/to/document.pdf',
    type: String,
  })
  @Matches(/^https:\/\/.+\.s3\..+\.amazonaws\.com\/.+/, {
    message: 'Please provide a valid S3 URL (e.g., https://bucket.s3.region.amazonaws.com/file)',
  })
  @IsString()
  url: string;

  @ApiProperty({
    description: 'Type of the document',
    example: 'flight_ticket',
    enum: DocumentTypeEnum,
  })
  @IsEnum(DocumentTypeEnum, {
    message: 'Document type must be one of: flight_ticket, identity_details',
  })
  type: DocumentTypeEnum;

  @ApiProperty({
    description: 'Program Details',
    example: {},
  })
  @IsOptional()
  programDetails: any;

  @ApiProperty({
    description: 'Journey type',
    example: 'Onward',
  })
  @IsOptional()
  journeyType: JourneyTypeEnum;

  @ApiProperty({
    description: 'Name of the user to validate against the flight ticket passenger list',
    example: 'John Smith',
    required: true,
  })
  @IsNotEmpty({ message: 'Username cannot be empty' })
  userName: string;

  @ApiProperty({
    description: 'Type of identity document when processing identity documents',
    example: 'Passport',
    required: false,
  })
  @IsOptional()
  @IsString()
  docType?: string;
}
