import { Injectable } from '@nestjs/common';
import DocumentIntelligence, {
  getLongRunningPoller,
  isUnexpected,
} from '@azure-rest/ai-document-intelligence';
import { AzureOpenAI } from 'openai';
import axios from 'axios';
import { ExtractionInfo } from './dto/create-ai-communication.dto';
import { DocumentTypeEnum } from 'src/common/enum/document-type.enum';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ERROR_MESSAGES } from 'src/common/i18n/error-messages';
import { AppLoggerService } from 'src/common/services/logger.service';
import {
  extractFlightDataSystemPrompt,
  extractFlightDataUserPrompt,
  extractIdentityDocumentSystemPrompt,
  extractIdentityDocumentUserPrompt,
  extractInvoiceDataSystemPrompt,
  extractInvoiceDataUserPrompt,
  flightTicketAnalyzerSystemPrompt,
  flightTicketAnalyzerUserPrompt,
  extractCitySystemPrompt,
  extractCityUserPrompt,
  lookupMatchingSystemPrompt,
  lookupAirlineMatchingUserPrompt,
  lookupCityMatchingUserPrompt,
} from 'src/common/utils/prompts';
import { azureOpenAIConfig } from 'src/common/config/config';
import { ConfigService } from '@nestjs/config';
import { LookupDataService } from 'src/lookup-data/lookup-data.service';

@Injectable()
export class AiCommunicationService {
  private openAiClient: AzureOpenAI;

  constructor(
    private configService: ConfigService,
    private readonly logger: AppLoggerService,
    private readonly lookupDataService: LookupDataService,
  ) {
    const config = azureOpenAIConfig(this.configService);
    this.openAiClient = new AzureOpenAI({
      endpoint: config.endpoint,
      apiKey: config.key,
      apiVersion: config.apiVersion,
    });
  }

  async processDocumentWithDocumentIntelligence(base64Source: string): Promise<string> {
    const endpoint = this.configService.get<string>('AZURE_DOCUMENT_INTELLIGENCE_ENDPOINT') || '';
    const key = this.configService.get<string>('AZURE_DOCUMENT_INTELLIGENCE_KEY') || '';

    if (!endpoint || !key) {
      this.logger.log('[ERROR] Azure Document Intelligence credentials are missing');
      throw new Error('Azure Document Intelligence credentials are missing');
    }

    this.logger.log('[INFO] Initializing Azure Document Intelligence client', { endpoint, key });

    const client = DocumentIntelligence(endpoint, { key });

    this.logger.log('[INFO] Sending document to Azure Document Intelligence for analysis:', {
      base64Source,
    });
    const initialResponse = await client
      .path('/documentModels/{modelId}:analyze', 'prebuilt-layout')
      .post({ contentType: 'application/json', body: { base64Source } });

    if (isUnexpected(initialResponse)) {
      throw new Error(`Azure returned unexpected response: ${initialResponse.status}`);
    }

    const poller = getLongRunningPoller(client, initialResponse);
    const result: any = await poller.pollUntilDone();
    this.logger.log('[INFO] Document analysis completed:', { status: result.status });

    // Log multi-page document detection for debugging
    if (result.body.analyzeResult.pages && result.body.analyzeResult.pages.length > 1) {
      this.logger.log(
        `[INFO] Multi-page document detected with ${result.body.analyzeResult.pages.length} pages`,
      );
    }

    return result.body.analyzeResult.content as string;
  }

  private async callOpenAIChatCompletion(
    systemPrompt: string,
    userPrompt: string,
    temperature = 0,
    max_tokens = 500,
  ): Promise<string> {
    const response = await this.openAiClient.chat.completions.create({
      model: this.configService.get<string>('AZURE_OPENAI_MODEL_NAME') || '',
      messages: [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: userPrompt },
      ],
      temperature,
      max_tokens,
    });
    return response.choices[0].message.content || '';
  }

  private parseOpenAIJsonResponse(raw: string): any {
    try {
      const markdownMatch = raw.match(/```(?:json)?\s*([\s\S]*?)\s*```/);
      if (markdownMatch) {
        return JSON.parse(markdownMatch[1].trim());
      }

      // Try direct JSON parsing first for valid JSON responses
      const trimmed = raw.trim();
      if (
        (trimmed.startsWith('{') && trimmed.endsWith('}')) ||
        (trimmed.startsWith('[') && trimmed.endsWith(']'))
      ) {
        try {
          return JSON.parse(trimmed);
        } catch (directParseError) {
          // Log the direct parsing failure for debugging, then fall back to original method
          this.logger.log(
            '[DEBUG] Direct JSON parsing failed, falling back to extractJsonFromString',
            {
              error: directParseError instanceof Error ? directParseError.message : 'Unknown error',
              rawLength: raw.length,
            },
          );
        }
      }

      return this.extractJsonFromString(raw);
    } catch (err) {
      this.logger.error('Failed to parse JSON from OpenAI response', '', { err });
      throw new Error('Failed to parse JSON from OpenAI response');
    }
  }

  private async isFlightTicket(content: string): Promise<any> {
    const systemPrompt = flightTicketAnalyzerSystemPrompt;
    const userPrompt = flightTicketAnalyzerUserPrompt(content);

    try {
      const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0, 500);
      return this.parseOpenAIJsonResponse(raw);
    } catch (err) {
      this.logger.error(
        '[ERROR] Failed to parse isFlightTicket response:',
        err instanceof Error ? err.message : 'Unknown error',
      );
      // Return a default response instead of throwing
      return { isFlightTicket: false };
    }
  }

  private async extractFlightData(content: string, journeyType?: string): Promise<any> {
    const systemPrompt = extractFlightDataSystemPrompt;
    const userPrompt = extractFlightDataUserPrompt(content, journeyType);

    this.logger.log(`[DEBUG] Flight data extraction - journeyType: ${journeyType}`);

    try {
      const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0.2, 1200);
      const structuredData = this.parseOpenAIJsonResponse(raw);

      this.logger.log(
        `[DEBUG] Extracted segments: ${JSON.stringify(
          structuredData.map((seg: any) => ({
            type: seg.type,
            comingFrom: seg.comingFrom || 'N/A',
            goingTo: seg.goingTo || 'N/A',
          })),
        )}`,
      );

      const onward = structuredData.find((seg) => seg.type === 'onward');
      const ret = structuredData.find((seg) => seg.type === 'return');

      let tripType = 'one-way';

      // For round-trip detection: verify complete round-trip journey
      // 1. Onward origin should match return destination (starts and ends in same city)
      // 2. Onward destination should match return origin (proper return journey)
      if (onward && ret && onward.comingFrom && onward.goingTo && ret.comingFrom && ret.goingTo) {
        const onwardOriginMatchesReturnDestination =
          onward.comingFrom.toLowerCase().trim() === ret.goingTo.toLowerCase().trim();
        const onwardDestinationMatchesReturnOrigin =
          onward.goingTo.toLowerCase().trim() === ret.comingFrom.toLowerCase().trim();

        if (onwardOriginMatchesReturnDestination && onwardDestinationMatchesReturnOrigin) {
          tripType = 'round-trip';
          this.logger.log(
            `[DEBUG] Round-trip detected: ${onward.comingFrom} → ${onward.goingTo} → ${ret.goingTo}`,
          );
        } else {
          this.logger.log(
            `[DEBUG] Not round-trip: Origin match=${onwardOriginMatchesReturnDestination}, Destination match=${onwardDestinationMatchesReturnOrigin}`,
          );
        }
      }

      this.logger.log(
        `[DEBUG] Final trip type: ${tripType}, onward: ${!!onward}, return: ${!!ret}`,
      );

      return { tripType, segments: structuredData };
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Unknown error occurred';
      this.logger.error('[ERROR] Failed to parse extractFlightData response:', errorMessage);
      // Return a basic structure instead of throwing
      return { tripType: 'unknown', segments: [] };
    }
  }

  private extractJsonFromString(str: string): any {
    str = JSON.stringify(str); // Convert to string if not already
    const jsonString = str
      .replace(/(\w+):/g, '"$1":') // Add quotes around keys
      .replace(/'/g, '"'); // Replace single quotes with double quotes

    return JSON.parse(jsonString);
  }

  private async fetchFileAsBase64(s3Url: string): Promise<string> {
    try {
      this.logger.log(`[INFO] Fetching file from S3: ${s3Url}`);

      const response = await axios.get(s3Url, {
        responseType: 'arraybuffer',
        headers: {
          referer: process.env.BACKEND_URL,
        },
      });

      // Check for empty/blank file
      if (!response.data || response.data.byteLength === 0) {
        this.logger.error('[VALIDATION] File is empty or unreadable');
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.FILE_UNREADABLE);
      }

      // Additional validation for file integrity
      const buffer = Buffer.from(response.data as ArrayBuffer);
      if (buffer.length === 0) {
        this.logger.error('[VALIDATION] Buffer is empty after conversion');
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.FILE_UNREADABLE);
      }

      this.logger.log(`[VALIDATION] File downloaded successfully. Size: ${buffer.length} bytes`);

      // Extract file extension from URL for additional validation
      const urlPath = new URL(s3Url).pathname;
      const fileExtension = urlPath.split('.').pop()?.toLowerCase();
      this.logger.log(`[VALIDATION] File extension detected: ${fileExtension}`);

      // Basic file format validation based on magic numbers
      const fileSignature = buffer.toString('hex', 0, 4);
      const isValidFormat = this.validateFileFormat(fileSignature);

      this.logger.log(
        `[VALIDATION] File signature analysis: ${fileSignature}, Valid: ${isValidFormat}`,
      );

      if (!isValidFormat) {
        // Check if it's a PDF file by extension but has invalid signature (corrupted)
        if (fileExtension === 'pdf') {
          this.logger.error(
            `[VALIDATION] PDF file appears to be corrupted. Expected signature: 25504446, Got: ${fileSignature}`,
          );
          // eslint-disable-next-line @typescript-eslint/only-throw-error
          throw new InifniBadRequestException(ERROR_CODES.PDF_FILE_CORRUPTED);
        }

        // Check if it's a supported extension but corrupted
        if (['jpg', 'jpeg', 'png'].includes(fileExtension || '')) {
          this.logger.error(
            `[VALIDATION] ${fileExtension?.toUpperCase()} file appears to be corrupted. Signature: ${fileSignature}`,
          );
          // eslint-disable-next-line @typescript-eslint/only-throw-error
          throw new InifniBadRequestException(
            ERROR_CODES.FILE_CORRUPTED,
            null,
            null,
            fileExtension?.toUpperCase() || 'Ticket',
          );
        }

        this.logger.error(
          `[VALIDATION] Unsupported file format. Extension: ${fileExtension}, Signature: ${fileSignature}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.UNSUPPORTED_FILE_FORMAT);
      }

      this.logger.log(
        `[VALIDATION] File format validated successfully. Signature: ${fileSignature}`,
      );

      return buffer.toString('base64');
    } catch (err: unknown) {
      const errorMessage = err instanceof Error ? err.message : 'Unknown error occurred';
      this.logger.error('[ERROR] Failed to fetch file from S3:', errorMessage);

      // Always re-throw the error, don't call handleKnownErrors
      throw err;
    }
  }

  private validateFileFormat(fileSignature: string): boolean {
    const supportedFormats = {
      // JPEG: FFD8FF
      jpeg: ['ffd8ffe0', 'ffd8ffe1', 'ffd8ffe2', 'ffd8ffe3', 'ffd8ffe8'],
      // PNG: 89504E47
      png: ['89504e47'],
      // PDF: 25504446
      pdf: ['25504446'],
    };

    const signature = fileSignature.toLowerCase();

    return (
      supportedFormats.jpeg.some((sig) => signature.startsWith(sig)) ||
      supportedFormats.png.some((sig) => signature.startsWith(sig)) ||
      supportedFormats.pdf.some((sig) => signature.startsWith(sig))
    );
  }

  /**
   * Get lookup data for airline and city matching
   */
  private async getLookupData(): Promise<{ airlines: any[]; cities: any[] }> {
    try {
      const lookupData = await this.lookupDataService.findAllWithRM();
      return {
        airlines: lookupData.AIRLINE || [],
        cities: lookupData.CITY_NAME || [],
      };
    } catch (error) {
      this.logger.error(
        '[LOOKUP] Failed to fetch lookup data:',
        String(error?.message || 'Unknown error'),
      );
      return { airlines: [], cities: [] };
    }
  }

  /**
   * AI-powered airline matching using OpenAI
   */
  private async matchAirlineWithAI(
    extractedAirline: string,
    lookupAirlines: any[],
    segmentType: 'onward' | 'return',
  ): Promise<any> {
    try {
      const systemPrompt = lookupMatchingSystemPrompt;
      const userPrompt = lookupAirlineMatchingUserPrompt(
        extractedAirline,
        lookupAirlines,
        segmentType,
      );

      this.logger.log(`[AI-LOOKUP] Matching airline: "${extractedAirline}" using AI`);

      const response = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0.1, 200);
      const result = this.parseOpenAIJsonResponse(response);

      this.logger.log(`[AI-LOOKUP] AI airline matching result: ${JSON.stringify(result)}`);

      // Convert matched value to key from lookup data
      const processedResult = this.convertAirlineValueToKey(result, lookupAirlines, segmentType);
      
      this.logger.log(`[AI-LOOKUP] Processed airline result: ${JSON.stringify(processedResult)}`);
      return processedResult;
    } catch (error) {
      this.logger.error(
        `[AI-LOOKUP] AI airline matching failed:`,
        String(error?.message || 'Unknown error'),
      );

      // Fallback to original value with "Other"
      const airlineField = segmentType === 'onward' ? 'airlineNameOnward' : 'airlineNameReturn';

      const otherField =
        segmentType === 'onward' ? 'otherAirlineNameOnward' : 'otherAirlineNameReturn';

      return {
        [airlineField]: 'Other',
        [otherField]: extractedAirline,
      };
    }
  }

  /**
   * AI-powered city matching using OpenAI
   */
  private async matchCityWithAI(
    extractedCity: string,
    lookupCities: any[],
    fieldType: 'departure' | 'arrival',
    segmentType?: 'onward' | 'return',
  ): Promise<any> {
    try {
      const systemPrompt = lookupMatchingSystemPrompt;
      const userPrompt = lookupCityMatchingUserPrompt(extractedCity, lookupCities, fieldType, segmentType);

      this.logger.log(`[AI-LOOKUP] Matching city: "${extractedCity}" using AI`);

      const response = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0.1, 200);
      const result = this.parseOpenAIJsonResponse(response);

      this.logger.log(`[AI-LOOKUP] AI city matching result: ${JSON.stringify(result)}`);

      // Convert matched value to key from lookup data
      const processedResult = this.convertCityValueToKey(result, lookupCities, fieldType);
      
      this.logger.log(`[AI-LOOKUP] Processed city result: ${JSON.stringify(processedResult)}`);
      return processedResult;
    } catch (error) {
      this.logger.error(
        `[AI-LOOKUP] AI city matching failed:`,
        String(error?.message || 'Unknown error'),
      );

      // Fallback to original value with "Other"
      const cityField = fieldType === 'departure' ? 'comingFrom' : 'goingTo';
      const otherField = fieldType === 'departure' ? 'otherDeparture' : 'otherArrival';

      return {
        [cityField]: 'Other',
        [otherField]: extractedCity,
      };
    }
  }

  /**
   * Convert airline value to key from lookup data
   */
  private convertAirlineValueToKey(
    aiResult: any,
    lookupAirlines: any[],
    segmentType: 'onward' | 'return',
  ): any {
    try {
      const airlineField = segmentType === 'onward' ? 'airlineNameOnward' : 'airlineNameReturn';
      
      // If AI returned "Other", find the actual "Other" key from database
      if (aiResult[airlineField] === 'Other') {
        const otherAirline = lookupAirlines.find(
          (airline) => airline.value?.toLowerCase() === 'other'
        );
        return {
          ...aiResult,
          [airlineField]: otherAirline ? otherAirline.key : 'Other',
        };
      }

      // Find the corresponding key for the matched value
      const matchedAirline = lookupAirlines.find(
        (airline) => airline.value === aiResult[airlineField],
      );

      if (matchedAirline) {
        this.logger.log(
          `[AI-LOOKUP] Converting airline "${aiResult[airlineField]}" to key "${matchedAirline.key}"`,
        );
        return {
          ...aiResult,
          [airlineField]: matchedAirline.key,
        };
      } else {
        this.logger.warn(
          `[AI-LOOKUP] Could not find key for airline value: ${aiResult[airlineField]}`,
        );
        return aiResult;
      }
    } catch (error) {
      this.logger.error(`[AI-LOOKUP] Error converting airline value to key:`, String(error));
      return aiResult;
    }
  }

  /**
   * Convert city value to key from lookup data
   */
  private convertCityValueToKey(
    aiResult: any,
    lookupCities: any[],
    fieldType: 'departure' | 'arrival',
  ): any {
    try {
      const cityField = fieldType === 'departure' ? 'comingFrom' : 'goingTo';
      
      // If AI returned "Other", find the actual "Other" key from database
      if (aiResult[cityField] === 'Other') {
        const otherCity = lookupCities.find(
          (city) => city.value?.toLowerCase() === 'other'
        );
        return {
          ...aiResult,
          [cityField]: otherCity ? otherCity.key : 'Other',
        };
      }

      // Find the corresponding key for the matched value
      const matchedCity = lookupCities.find((city) => city.value === aiResult[cityField]);

      if (matchedCity) {
        this.logger.log(
          `[AI-LOOKUP] Converting city "${aiResult[cityField]}" to key "${matchedCity.key}"`,
        );
        return {
          ...aiResult,
          [cityField]: matchedCity.key,
        };
      } else {
        this.logger.warn(`[AI-LOOKUP] Could not find key for city value: ${aiResult[cityField]}`);
        return aiResult;
      }
    } catch (error) {
      this.logger.error(`[AI-LOOKUP] Error converting city value to key:`, String(error));
      return aiResult;
    }
  }

  /**
   * Apply lookup matching to flight data
   */
  private async applyLookupMatching(flightData: any): Promise<any> {
    this.logger.log('[LOOKUP] Starting lookup matching for flight data');

    try {
      const { airlines, cities } = await this.getLookupData();

      if (!flightData.segments || !Array.isArray(flightData.segments)) {
        this.logger.log('[LOOKUP] No segments found in flight data');
        return flightData;
      }

      let hasAirlineCodeIssues = false;

      const enhancedSegments = await Promise.all(
        flightData.segments.map(async (segment: any) => {
          const enhancedSegment = { ...segment };

          // Match airline names using AI AND clean flight numbers
          if (segment.type === 'onward' && segment.airlineNameOnward) {
            const airlineMatch = await this.matchAirlineWithAI(
              String(segment.airlineNameOnward),
              airlines,
              'onward',
            );
            enhancedSegment.airlineNameOnward = airlineMatch.airlineNameOnward;
            if (airlineMatch.otherAirlineNameOnward) {
              enhancedSegment.otherAirlineNameOnward = airlineMatch.otherAirlineNameOnward;
            }

            // Clean flight number using matched airline info
            if (segment.flightNumberOnward) {
              const originalNumber = segment.flightNumberOnward;
              const matchedAirlineData = airlines.find(
                (airline) => airline.key === enhancedSegment.airlineNameOnward
              );
              const cleaningResult = this.cleanFlightNumberWithLookup(
                originalNumber,
                matchedAirlineData,
              );
              enhancedSegment.flightNumberOnward = cleaningResult.cleanedNumber;
              
              if (cleaningResult.hasAirlineCodeIssue) {
                hasAirlineCodeIssues = true;
              }
              
              if (originalNumber !== cleaningResult.cleanedNumber) {
                this.logger.log(
                  `[LOOKUP] Onward flight number cleaned: "${originalNumber}" → "${cleaningResult.cleanedNumber}" (using ${matchedAirlineData?.key || 'fallback'})`,
                );
              }
            }
          }

          if (segment.type === 'return' && segment.airlineNameReturn) {
            const airlineMatch = await this.matchAirlineWithAI(
              String(segment.airlineNameReturn),
              airlines,
              'return',
            );
            enhancedSegment.airlineNameReturn = airlineMatch.airlineNameReturn;
            if (airlineMatch.otherAirlineNameReturn) {
              enhancedSegment.otherAirlineNameReturn = airlineMatch.otherAirlineNameReturn;
            }

            // Clean flight number using matched airline info
            if (segment.flightNumberReturn) {
              const originalNumber = segment.flightNumberReturn;
              const matchedAirlineData = airlines.find(
                (airline) => airline.key === enhancedSegment.airlineNameReturn
              );
              const cleaningResult = this.cleanFlightNumberWithLookup(
                originalNumber,
                matchedAirlineData,
              );
              enhancedSegment.flightNumberReturn = cleaningResult.cleanedNumber;
              
              if (cleaningResult.hasAirlineCodeIssue) {
                hasAirlineCodeIssues = true;
              }
              
              if (originalNumber !== cleaningResult.cleanedNumber) {
                this.logger.log(
                  `[LOOKUP] Return flight number cleaned: "${originalNumber}" → "${cleaningResult.cleanedNumber}" (using ${matchedAirlineData?.key || 'fallback'})`,
                );
              }
            }
          }

          // Match city names using AI with segment type context
          if (segment.comingFrom) {
            const departureMatch = await this.matchCityWithAI(
              String(segment.comingFrom),
              cities,
              'departure',
              segment.type,
            );
            enhancedSegment.comingFrom = departureMatch.comingFrom;
            
            // For onward segments, comingFrom sets otherArrival
            if (segment.type === 'onward' && departureMatch.otherArrival) {
              enhancedSegment.otherArrival = departureMatch.otherArrival;
            }
          }

          if (segment.goingTo) {
            const arrivalMatch = await this.matchCityWithAI(
              String(segment.goingTo),
              cities,
              'arrival',
              segment.type,
            );
            enhancedSegment.goingTo = arrivalMatch.goingTo;
            
            // For return segments, goingTo sets otherDeparture
            if (segment.type === 'return' && arrivalMatch.otherDeparture) {
              enhancedSegment.otherDeparture = arrivalMatch.otherDeparture;
            }
          }

          return enhancedSegment;
        }),
      );

      const enhancedFlightData = {
        ...flightData,
        segments: enhancedSegments,
        hasAirlineCodeIssues,
      };

      this.logger.log('[LOOKUP] Lookup matching completed successfully');
      return enhancedFlightData;
    } catch (error) {
      this.logger.error(
        '[LOOKUP] Error during lookup matching:',
        String(error?.message || 'Unknown error'),
      );
      // Return original data if lookup fails
      return flightData;
    }
  }

  async analyzeFlightTicket(
    s3Url: string,
    programDetails: any,
    userName: string,
    journeyType?: string,
  ): Promise<any> {
    try {
      const base64 = await this.fetchFileAsBase64(s3Url);
      const content = await this.processDocumentWithDocumentIntelligence(base64);

      // Handle cases with no or very little content
      if (!content || content.trim().length < 10) {
        this.logger.error(
          `[VALIDATION] Document content too short or empty. Content length: ${content?.length || 0}, Trimmed length: ${content?.trim().length || 0}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.UNABLE_TO_EXTRACT_TEXT);
      }

      this.logger.log(
        `[VALIDATION] Document content extracted successfully. Content length: ${content.length}, Trimmed length: ${content.trim().length}`,
      );

      const verification = await this.isFlightTicket(content);

      if (!verification.isFlightTicket) {
        this.logger.error(
          `[VALIDATION] Document is not a flight ticket. Verification result: ${JSON.stringify(verification)}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.FILE_NOT_FLIGHT_TICKET);
      }

      this.logger.log(
        `[VALIDATION] Document verified as flight ticket. Verification result: ${JSON.stringify(verification)}`,
      );

      const structuredResult = await this.extractFlightData(content, journeyType);

      this.logger.log(`[DEBUG] Structured result: ${JSON.stringify(structuredResult)}`);

      this.logger.log(
        `[VALIDATION] Flight data extraction completed. Trip type: ${structuredResult.tripType}, Segments count: ${structuredResult.segments?.length || 0}`,
      );

      // Apply lookup matching to standardize airline and city names AND clean flight numbers
      const enhancedResult = await this.applyLookupMatching(structuredResult);

      this.logger.log(
        `[LOOKUP] Flight data enhanced with lookup matching. Enhanced segments count: ${enhancedResult.segments?.length || 0}`,
      );

      // Validate passenger name - userName is now required
      const passengerValidation = this.validatePassengerName(enhancedResult, userName);

      this.logger.log(
        `[VALIDATION] Passenger name validation result: ${passengerValidation.isValid ? 'PASSED' : 'FAILED'} ${passengerValidation.message || ''}`,
      );

      // Track name validation status for response
      let isNameValid = true;
      let nameValidationMessage = '';

      if (!passengerValidation.isValid) {
        isNameValid = false;
        nameValidationMessage =
          'The name in the flight ticket does not match with the user name in the application';

        this.logger.warn(
          `[VALIDATION] Passenger name validation failed: ${passengerValidation.message}`,
        );
        this.logger.log(
          '[VALIDATION] Continuing with flight ticket processing despite name mismatch - returning data with warning',
        );
      } else {
        this.logger.log('[VALIDATION] Passenger name validation passed successfully');
      }

      // Check for partial data extraction - this will return validation result
      const partialDataValidation = this.checkForPartialData(enhancedResult);

      this.logger.log(
        `[VALIDATION] Partial data validation result: ${partialDataValidation.isValid ? 'PASSED' : 'FAILED'} ${partialDataValidation.message || ''}`,
      );

      if (!partialDataValidation.isValid) {
        this.logger.error(
          `[VALIDATION] Partial data validation failed: ${partialDataValidation.message}`,
        );
        const errorMessage =
          partialDataValidation.message ||
          'All the data cannot be extracted from the ticket. Please upload a valid ticket with clear and complete information.';
        this.logger.error(
          `[VALIDATION] Throwing InifniBadRequestException with message: ${errorMessage}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          partialDataValidation.errorCode || ERROR_CODES.PARTIAL_DATA,
          new Error(errorMessage),
        );
      }

      this.logger.log('[DEBUG] Check Enhanced result', enhancedResult);

      // Validate flight destinations against program venue
      const venueValidation = await this.validateFlightVenue(enhancedResult, programDetails);

      this.logger.log(
        `[VALIDATION] Flight venue validation result: ${venueValidation.isValid ? 'PASSED' : 'FAILED'} ${venueValidation.message || ''}`,
      );

      if (!venueValidation.isValid) {
        this.logger.error(
          `[VALIDATION] Flight venue validation failed: ${venueValidation.message}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          venueValidation.errorCode || ERROR_CODES.FAILED_TO_ANALYZE,
          new Error(venueValidation.message || 'Flight venue validation failed'),
          null,
          venueValidation?.segmentType || '',
          venueValidation?.goingTo || '',
          venueValidation?.programCity || '',
        );
      }

      this.logger.log('[VALIDATION] Flight venue validation passed successfully');

      // Validate flight dates against program dates
      const dateValidation = this.validateFlightDates(enhancedResult, programDetails);

      this.logger.log(
        `[VALIDATION] Flight date validation result: ${dateValidation.isValid ? 'PASSED' : 'FAILED'} ${dateValidation.message || ''}`,
      );

      if (!dateValidation.isValid) {
        this.logger.error(`[VALIDATION] Flight date validation failed: ${dateValidation.message}`);
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          dateValidation.errorCode || ERROR_CODES.FAILED_TO_ANALYZE,
        );
      }

      this.logger.log('[VALIDATION] Flight date validation passed successfully');

      // Check if we have partial data from date validation (round-trip with one invalid segment)
      if (dateValidation.partialData) {
        this.logger.log('[VALIDATION] Returning partial flight data due to date validation issues');
        return {
          ...this.filterFlightDataForUI(dateValidation.partialData),
          message: dateValidation.message || 'Partial flight data returned due to date validation.',
          isNameValid,
        };
      }

      this.logger.log('[VALIDATION] Returning complete flight data successfully');

      // Prepare final response with name validation status
      const finalResult = {
        ...this.filterFlightDataForUI(enhancedResult),
        isNameValid,
      };

      // // Add warning message if name validation failed
      // if (!isNameValid) {
      //   finalResult.message = nameValidationMessage;
      // }

      // Check for airline code issues and add warning message
      if (enhancedResult.hasAirlineCodeIssues) {
        this.logger.log('[VALIDATION] Airline code issues detected - adding warning message');
        finalResult.airlineCodeMessage =
          'Unable to read the airline code properly from the ticket. Please review manually.';
      }

      // Filter the response before returning to UI
      return finalResult;
    } catch (err) {
      // Log the error for debugging
      if (err instanceof InifniBadRequestException) {
        this.logger.error('[ERROR] Flight ticket analysis failed with known error:', err.message);
        // Re-throw the InifniBadRequestException as-is to preserve the error message
        throw err;
      } else {
        const errorMessage = err instanceof Error ? err.message : 'Unknown error occurred';
        this.logger.error(
          '[ERROR] Flight ticket analysis failed with unexpected error:',
          errorMessage,
        );
        // Re-throw as InifniBadRequestException with the original error message
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.FAILED_TO_ANALYZE,
          new Error(errorMessage || 'An unexpected error occurred during flight ticket analysis'),
        );
      }
    }
  }

  /**
   * Clean flight number using matched airline lookup data (integrated approach)
   */
  private cleanFlightNumberWithLookup(
    flightNumber: string,
    airlineData?: any,
  ): { cleanedNumber: string; hasAirlineCodeIssue: boolean } {
    if (!flightNumber || typeof flightNumber !== 'string') {
      return { cleanedNumber: flightNumber, hasAirlineCodeIssue: false };
    }

    const trimmedNumber = flightNumber.trim();

    // If already numeric, AI followed instructions correctly
    if (/^\d+$/.test(trimmedNumber)) {
      return { cleanedNumber: trimmedNumber, hasAirlineCodeIssue: false };
    }

    // AI didn't follow instructions, use lookup-based fallback
    this.logger.warn(
      `[LOOKUP] AI returned non-numeric flight number "${trimmedNumber}". Using lookup-based cleaning.`,
    );

    // Try to remove airline code using lookup data
    if (airlineData && airlineData.key) {
      const airlineCode = airlineData.key;
      
      // If airline is "OTHER", we don't need to parse airline codes - just return as is
      if (airlineCode.toLowerCase() === 'other') {
        this.logger.log(
          `[LOOKUP] Airline is "OTHER" - returning flight number as-is without parsing: "${trimmedNumber}"`,
        );
        return { cleanedNumber: trimmedNumber, hasAirlineCodeIssue: false };
      }
      
      // Handle various separators: "6E123", "6E 123", "6E-123", "6E_123", etc.
      const pattern = new RegExp(`^${airlineCode}[\\s\\-_]*(\\d+)`, 'i');
      const match = trimmedNumber.match(pattern);
      
      if (match) {
        this.logger.log(
          `[LOOKUP] Removed airline code "${airlineCode}" from "${trimmedNumber}" → "${match[1]}" (${airlineData.value})`,
        );
        return { cleanedNumber: match[1], hasAirlineCodeIssue: false };
      }
    }

    // For flights not in lookup data, indicate airline code issue
    this.logger.warn(
      `[LOOKUP] Unable to parse airline code from flight number: "${trimmedNumber}". This may require manual review.`,
    );
    return { cleanedNumber: trimmedNumber, hasAirlineCodeIssue: true };
  }

  private checkForPartialData(structuredResult: any): {
    isValid: boolean;
    message?: string;
    errorCode?: string;
  } {
    if (structuredResult?.segments) {
      for (const segment of structuredResult.segments) {
        const segmentType = segment.type === 'onward' ? 'Onward' : 'Return';

        // Check for airline and flight number based on segment type
        if (segment.type === 'onward') {
          if (!segment.airlineNameOnward) {
            this.logger.error(`[VALIDATION] ${segmentType} flight: Airline information is missing`);
            return {
              isValid: false,
              errorCode: ERROR_CODES.AIRLINE_NUMBER_EXTRACTION_FAILED,
              message: ERROR_MESSAGES.AIRLINE_NUMBER_EXTRACTION_FAILED,
            };
          }

          if (!segment.flightNumberOnward) {
            this.logger.error(`[VALIDATION] ${segmentType} flight: Flight number is missing`);
            return {
              isValid: false,
              errorCode: ERROR_CODES.FLIGHT_NUMBER_EXTRACTION_FAILED,
              message: ERROR_MESSAGES.FLIGHT_NUMBER_EXTRACTION_FAILED,
            };
          } else if (
            segment.flightNumberOnward.includes('?') ||
            segment.flightNumberOnward.includes('*')
          ) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Flight number is partially readable: "${segment.flightNumberOnward}"`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.FLIGHT_NUMBER_EXTRACTION_FAILED,
              message: ERROR_MESSAGES.FLIGHT_NUMBER_EXTRACTION_FAILED,
            };
          }

          // City name validation - onward segments need both comingFrom and goingTo
          if (!segment.comingFrom) {
            this.logger.error(`[VALIDATION] ${segmentType} flight: Origin location is missing`);
            return {
              isValid: false,
              errorCode: ERROR_CODES.ORIGIN_LOCATION_MISSING,
              message: ERROR_MESSAGES.ORIGIN_LOCATION_MISSING,
            };
          } else if (this.isAirportCode(segment.comingFrom as string)) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Origin shows airport code instead of city name`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.ORIGIN_LOCATION_MISSING,
              message: ERROR_MESSAGES.ORIGIN_LOCATION_MISSING,
            };
          }

          if (!segment.goingTo) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Destination location is missing`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.DESTINATION_LOCATION_MISSING,
              message: ERROR_MESSAGES.DESTINATION_LOCATION_MISSING,
            };
          } else if (this.isAirportCode(segment.goingTo as string)) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Destination shows airport code instead of city name`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.DESTINATION_LOCATION_MISSING,
              message: ERROR_MESSAGES.DESTINATION_LOCATION_MISSING,
            };
          }

          // Time validation - both departure and arrival times are required
          if (!segment.departureDateTime || !segment.arrivalDateTime) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Missing departure or arrival time`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.DEPARTURE_ARRIVAL_TIME_MISSING,
              message: ERROR_MESSAGES.DEPARTURE_ARRIVAL_TIME_MISSING,
            };
          }
        } else if (segment.type === 'return') {
          if (!segment.airlineNameReturn) {
            this.logger.error(`[VALIDATION] ${segmentType} flight: Airline information is missing`);
            return {
              isValid: false,
              errorCode: ERROR_CODES.AIRLINE_NUMBER_EXTRACTION_FAILED,
              message: ERROR_MESSAGES.AIRLINE_NUMBER_EXTRACTION_FAILED,
            };
          }

          if (!segment.flightNumberReturn) {
            this.logger.error(`[VALIDATION] ${segmentType} flight: Flight number is missing`);
            return {
              isValid: false,
              errorCode: ERROR_CODES.FLIGHT_NUMBER_EXTRACTION_FAILED,
              message: ERROR_MESSAGES.FLIGHT_NUMBER_EXTRACTION_FAILED,
            };
          } else if (
            segment.flightNumberReturn.includes('?') ||
            segment.flightNumberReturn.includes('*')
          ) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Flight number is partially readable: "${segment.flightNumberReturn}"`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.FLIGHT_NUMBER_EXTRACTION_FAILED,
              message: ERROR_MESSAGES.FLIGHT_NUMBER_EXTRACTION_FAILED,
            };
          }

          // City name validation - return segments need both comingFrom and goingTo
          if (!segment.comingFrom) {
            this.logger.error(`[VALIDATION] ${segmentType} flight: Origin location is missing`);
            return {
              isValid: false,
              errorCode: ERROR_CODES.ORIGIN_LOCATION_MISSING,
              message: ERROR_MESSAGES.ORIGIN_LOCATION_MISSING,
            };
          } else if (this.isAirportCode(segment.comingFrom as string)) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Origin shows airport code instead of city name`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.ORIGIN_LOCATION_MISSING,
              message: ERROR_MESSAGES.ORIGIN_LOCATION_MISSING,
            };
          }

          if (!segment.goingTo) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Destination location is missing`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.DESTINATION_LOCATION_MISSING,
              message: ERROR_MESSAGES.DESTINATION_LOCATION_MISSING,
            };
          } else if (this.isAirportCode(segment.goingTo as string)) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Destination shows airport code instead of city name`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.DESTINATION_LOCATION_MISSING,
              message: ERROR_MESSAGES.DESTINATION_LOCATION_MISSING,
            };
          }

          // Time validation - both departure and arrival times are required
          if (!segment.departureDateTime || !segment.arrivalDateTime) {
            this.logger.error(
              `[VALIDATION] ${segmentType} flight: Missing departure or arrival time`,
            );
            return {
              isValid: false,
              errorCode: ERROR_CODES.DEPARTURE_ARRIVAL_TIME_MISSING,
              message: ERROR_MESSAGES.DEPARTURE_ARRIVAL_TIME_MISSING,
            };
          }
        }
      }
    }

    this.logger.log('[VALIDATION] All required flight data extracted successfully');
    return { isValid: true };
  }

  private async validateFlightVenue(
    structuredResult: any,
    programDetails: any,
  ): Promise<{
    isValid: boolean;
    message?: string;
    segmentType?: string;
    goingTo?: string;
    programCity?: string;
    errorCode?: string;
  }> {
    this.logger.log('[VALIDATION] Starting flight venue validation');

    if (!programDetails) {
      this.logger.error('[VALIDATION] Program details are missing for venue validation');
      return {
        isValid: false,
        message: ERROR_MESSAGES.PROGRAM_VENUE_CITY_REQUIRED,
        errorCode: ERROR_CODES.PROGRAM_VENUE_CITY_REQUIRED,
      };
    }

    // Extract program city from venue details
    const programCity = await this.extractProgramCity(programDetails);

    if (!programCity) {
      this.logger.log('[VALIDATION] Could not extract program city - skipping venue validation');
      return { isValid: true }; // Skip validation if we can't determine program city
    }

    this.logger.log(`[VALIDATION] Program city extracted: ${programCity}`);

    // Validate each segment's destination against program city
    if (structuredResult?.segments) {
      for (const segment of structuredResult.segments) {
        // For onward flights, check if goingTo matches program city
        if (segment.type === 'onward' && segment.goingTo) {
          const isValidDestination = this.isCityMatch(segment.goingTo, programCity);

          this.logger.log(
            `[VALIDATION] Onward flight destination check: ${segment.goingTo} vs ${programCity} = ${isValidDestination}`,
          );

          if (!isValidDestination) {
            return {
              isValid: false,
              message: ERROR_MESSAGES.FLIGHT_DESTINATION_MISMATCH,
              errorCode: ERROR_CODES.FLIGHT_DESTINATION_MISMATCH,
              segmentType: 'onward',
              goingTo: segment.goingTo,
              programCity: programCity,
            };
          }
        }

        // For return flights, check if comingFrom matches program city
        if (segment.type === 'return' && segment.comingFrom) {
          const isValidOrigin = this.isCityMatch(segment.comingFrom, programCity);

          this.logger.log(
            `[VALIDATION] Return flight origin check: ${segment.comingFrom} vs ${programCity} = ${isValidOrigin}`,
          );

          if (!isValidOrigin) {
            return {
              isValid: false,
              message: ERROR_MESSAGES.FLIGHT_ORIGIN_MISMATCH,
              errorCode: ERROR_CODES.FLIGHT_ORIGIN_MISMATCH,
              segmentType: 'return',
              goingTo: segment.comingFrom,
              programCity: programCity,
            };
          }
        }
      }
    }

    this.logger.log('[VALIDATION] Flight venue validation passed successfully');
    return { isValid: true };
  }

  private validateFlightDates(
    structuredResult: any,
    programDetails: any,
  ): { isValid: boolean; message?: string; errorCode?: string; partialData?: any } {
    // Check if program dates are available
    if (!programDetails) {
      return {
        isValid: false,
        errorCode: ERROR_CODES.PROGRAM_DATES_REQUIRED,
        message: ERROR_MESSAGES.PROGRAM_DATES_REQUIRED,
      };
    }

    const programStartDate = programDetails.startsAt || programDetails.StartsAt;
    const programEndDate = programDetails.endsAt || programDetails.EndsAt;

    if (!programStartDate || !programEndDate) {
      this.logger.error('[VALIDATION] Program start or end date is missing from programDetails');
      return {
        isValid: false,
        errorCode: ERROR_CODES.PROGRAM_DATES_REQUIRED,
        message: ERROR_MESSAGES.PROGRAM_DATES_REQUIRED,
      };
    }

    this.logger.log(
      `[VALIDATION] Program dates - Start: ${programStartDate}, End: ${programEndDate}`,
    );

    // Parse program dates
    const programStart = new Date(programStartDate as string);
    const programEnd = new Date(programEndDate as string);

    this.logger.log(
      `[VALIDATION] Parsed program dates - Start: ${programStart.toISOString()}, End: ${programEnd.toISOString()}`,
    );

    // Validate flight segments based on trip type
    if (structuredResult?.segments) {
      const tripType = structuredResult.tripType;
      this.logger.log(`[VALIDATION] Validating ${tripType} flight dates based on segment types`);

      // For round-trip tickets, always validate both segments
      if (tripType === 'round-trip') {
        this.logger.log('[VALIDATION] Round-trip ticket detected - validating both segments');

        const onwardSegment = structuredResult.segments.find((seg: any) => seg.type === 'onward');
        const returnSegment = structuredResult.segments.find((seg: any) => seg.type === 'return');

        if (!onwardSegment || !returnSegment) {
          this.logger.error('[VALIDATION] Round-trip flight missing onward or return segment');
          return {
            isValid: false,
            errorCode: ERROR_CODES.ROUND_TRIP_SEGMENTS_MISSING,
            message: ERROR_MESSAGES.ROUND_TRIP_SEGMENTS_MISSING,
          };
        }

        // Validate onward flight
        const onwardArrivalDate = this.extractDateFromDateTime(
          onwardSegment.arrivalDateTime as string,
        );
        const isValidOnward = onwardArrivalDate
          ? this.isValidOnwardDate(onwardArrivalDate, programStart)
          : false;

        // Validate return flight
        let isValidReturn = false;
        const returnDepartureDate = this.extractDateFromDateTime(
          returnSegment.departureDateTime as string,
        );
        
        this.logger.log(
          `[DEBUG] Return flight validation details - Departure: ${returnSegment.departureDateTime}, Parsed: ${returnDepartureDate?.toISOString() || 'null'}, Program End: ${programEnd.toISOString()}`,
        );
        
        if (returnDepartureDate) {
          isValidReturn = this.isValidReturnDate(returnDepartureDate, programEnd);
        }

        this.logger.log(
          `[VALIDATION] Round-trip validation - Onward: ${isValidOnward}, Return: ${isValidReturn}`,
        );

        if (isValidOnward && isValidReturn) {
          // Both are valid - return success
          this.logger.log('[VALIDATION] Round-trip flight dates validated successfully');
          return { isValid: true };
        } else if (isValidOnward && !isValidReturn) {
          // Onward valid, return invalid - return partial data with warning
          this.logger.log(
            '[VALIDATION] Round-trip onward valid, return invalid - returning partial data',
          );
          const onwardOnlyData = {
            ...structuredResult,
            tripType: 'one-way', // Change to one-way since we're only returning onward
            segments: [onwardSegment], // Only include valid onward segment
            warnings: [
              'Return flight dates do not align with program schedule. Only onward flight details are included.',
            ],
          };
          return {
            isValid: true,
            partialData: onwardOnlyData,
            errorCode: ERROR_CODES.RETURN_FLIGHT_DETAILS_MISSING,
            message: ERROR_MESSAGES.RETURN_FLIGHT_DETAILS_MISSING,
          };
        } else if (!isValidOnward && isValidReturn) {
          // Onward invalid, return valid - return partial data with warning
          this.logger.log(
            '[VALIDATION] Round-trip return valid, onward invalid - returning partial data',
          );
          const returnOnlyData = {
            ...structuredResult,
            tripType: 'one-way', // Change to one-way since we're only returning return
            segments: [returnSegment], // Only include valid return segment
            warnings: [
              'Onward flight dates do not align with program schedule. Only return flight details are included.',
            ],
          };
          return {
            isValid: true,
            partialData: returnOnlyData,
            errorCode: ERROR_CODES.ONWARD_FLIGHT_DETAILS_MISSING,
            message: ERROR_MESSAGES.ONWARD_FLIGHT_DETAILS_MISSING,
          };
        } else {
          // Both invalid - return error
          this.logger.error('[VALIDATION] Both onward and return flight dates are invalid');
          return {
            isValid: false,
            errorCode: ERROR_CODES.ONWARD_RETURN_FLIGHT_MISMATCH,
            message: ERROR_MESSAGES.ONWARD_RETURN_FLIGHT_MISMATCH,
          };
        }
      } else {
        // For one-way tickets, validate all segments present
        this.logger.log('[VALIDATION] One-way ticket detected - validating all segments present');

        for (const segment of structuredResult.segments) {
          if (segment.type === 'onward') {
            const arrivalDate = this.extractDateFromDateTime(segment.arrivalDateTime as string);
            if (arrivalDate) {
              const isValidOnward = this.isValidOnwardDate(arrivalDate, programStart);
              if (!isValidOnward) {
                this.logger.error(
                  `[VALIDATION] Onward flight date validation failed: ${arrivalDate.toISOString()}`,
                );
                return {
                  isValid: false,
                  errorCode: ERROR_CODES.INVALID_FLIGHT_DATE_ONWARD,
                  message: ERROR_MESSAGES.INVALID_FLIGHT_DATE_ONWARD,
                };
              }
            }
          } else if (segment.type === 'return') {
            const departureDate = this.extractDateFromDateTime(segment.departureDateTime as string);
            if (departureDate) {
              const isValidReturn = this.isValidReturnDate(departureDate, programEnd);
              if (!isValidReturn) {
                this.logger.error(
                  `[VALIDATION] Return flight date validation failed: ${departureDate.toISOString()}`,
                );
                return {
                  isValid: false,
                  errorCode: ERROR_CODES.INVALID_FLIGHT_DATE_RETURN,
                  message: ERROR_MESSAGES.INVALID_FLIGHT_DATE_RETURN,
                };
              }
            }
          }
        }

        this.logger.log('[VALIDATION] One-way flight date validated successfully');
      }
    }

    return { isValid: true };
  }

  private extractDateFromDateTime(dateTimeString: string): Date | null {
    this.logger.log(`[VALIDATION] Date extraction initiated for: ${dateTimeString}`);
    if (!dateTimeString) return null;
    try {
      // Parse the datetime string to extract the local date components
      // This preserves the original date regardless of timezone conversion
      const parsedDate = new Date(dateTimeString);
      
      // Extract the original date components from the input string
      // to avoid timezone conversion issues
      const dateMatch = dateTimeString.match(/^(\d{4})-(\d{2})-(\d{2})/);
      if (dateMatch) {
        const [, year, month, day] = dateMatch;
        const localDate = new Date(parseInt(year), parseInt(month) - 1, parseInt(day));
        
        this.logger.log(
          `[VALIDATION] Date extraction successful for: ${dateTimeString} - Original UTC: ${parsedDate.toISOString()}, Local date preserved: ${localDate.toISOString()}`,
        );
        
        return localDate;
      } else {
        // Fallback to original method if regex doesn't match
        this.logger.log(
          `[VALIDATION] Date extraction using fallback for: ${dateTimeString} returning: ${parsedDate.toISOString()}`,
        );
        return parsedDate;
      }
    } catch {
      this.logger.log(`[VALIDATION] Date extraction failed for: ${dateTimeString}`);
      return null;
    }
  }

  private isValidOnwardDate(arrivalDate: Date, programStart: Date): boolean {
    // Onward flight can arrive:
    // 1. Any time on the day before program starts, OR
    // 2. Any time on the program start date

    // Extract date-only components to avoid timezone issues
    const arrivalDateOnly = new Date(
      arrivalDate.getFullYear(),
      arrivalDate.getMonth(),
      arrivalDate.getDate(),
    );
    
    // For program dates, extract from ISO string to avoid timezone conversion
    const programStartStr = programStart.toISOString();
    const startDateMatch = programStartStr.match(/^(\d{4})-(\d{2})-(\d{2})/);
    let programStartDateOnly: Date;
    
    if (startDateMatch) {
      const [, year, month, day] = startDateMatch;
      programStartDateOnly = new Date(parseInt(year), parseInt(month) - 1, parseInt(day));
    } else {
      // Fallback to original method
      programStartDateOnly = new Date(
        programStart.getFullYear(),
        programStart.getMonth(),
        programStart.getDate(),
      );
    }
    
    const oneDayBefore = new Date(programStartDateOnly);
    oneDayBefore.setDate(programStartDateOnly.getDate() - 1);

    this.logger.log(
      `[VALIDATION] Onward validation - Arrival: ${arrivalDate.toISOString()}, Program Start: ${programStart.toISOString()}`,
    );

    // Case 1: Arriving on day before program starts (any time is OK)
    if (arrivalDateOnly.getTime() === oneDayBefore.getTime()) {
      this.logger.log('[VALIDATION] Onward flight arrives day before program - VALID');
      return true;
    }

    // Case 2: Arriving on program start date (any time is OK)
    if (arrivalDateOnly.getTime() === programStartDateOnly.getTime()) {
      this.logger.log('[VALIDATION] Onward flight arrives on program start date - VALID');
      return true;
    }

    // Case 3: Arriving on other dates
    this.logger.log('[VALIDATION] Onward flight arrival date outside valid range - INVALID');
    return false;
  }

  private isValidReturnDate(departureDate: Date, programEnd: Date): boolean {
    this.logger.log(
      `[VALIDATION] Return validation - Departure Date validation initiated: ${departureDate.toISOString()}, Program End: ${programEnd.toISOString()}`,
    );
    // Return flight can depart:
    // Any time on the program end date ONLY

    // Extract date-only components to avoid timezone issues
    const departureDateOnly = new Date(
      departureDate.getFullYear(),
      departureDate.getMonth(),
      departureDate.getDate(),
    );
    
    // For program dates, extract from ISO string to avoid timezone conversion
    const programEndStr = programEnd.toISOString();
    const endDateMatch = programEndStr.match(/^(\d{4})-(\d{2})-(\d{2})/);
    let programEndDateOnly: Date;
    
    if (endDateMatch) {
      const [, year, month, day] = endDateMatch;
      programEndDateOnly = new Date(parseInt(year), parseInt(month) - 1, parseInt(day));
    } else {
      // Fallback to original method
      programEndDateOnly = new Date(
        programEnd.getFullYear(),
        programEnd.getMonth(),
        programEnd.getDate(),
      );
    }

    this.logger.log(
      `[VALIDATION] Return validation - Departure: ${departureDate.toISOString()}, Program End: ${programEnd.toISOString()}`,
    );
    
    this.logger.log(
      `[DEBUG] Date comparison - Departure date only: ${departureDateOnly.toISOString()}, Program end date only: ${programEndDateOnly.toISOString()}`,
    );
    
    this.logger.log(
      `[DEBUG] Time comparison - Departure timestamp: ${departureDateOnly.getTime()}, Program end timestamp: ${programEndDateOnly.getTime()}, Equal: ${departureDateOnly.getTime() === programEndDateOnly.getTime()}`,
    );

    // Valid case: Departing on program end date (any time is OK)
    if (departureDateOnly.getTime() === programEndDateOnly.getTime()) {
      this.logger.log('[VALIDATION] Return flight departs on program end date - VALID');
      return true;
    }

    // All other cases: Departing on different dates is not allowed
    this.logger.log(
      '[VALIDATION] Return flight departure date must be on program end date only - INVALID',
    );
    return false;
  }

  private validatePassengerName(
    structuredResult: any,
    userName: string,
  ): { isValid: boolean; message?: string; errorCode?: string } {
    this.logger.log(`[VALIDATION] Starting passenger name validation for user: ${userName}`);

    // Check if any segment has passenger names
    const hasPassengerData = structuredResult.segments?.some(
      (segment: any) =>
        segment.passengerNames &&
        Array.isArray(segment.passengerNames) &&
        segment.passengerNames.length > 0,
    );

    if (!hasPassengerData) {
      this.logger.error('[VALIDATION] No passenger names found in ticket - validation failed');
      return {
        isValid: false,
        message:
          'No passenger names found in the ticket. Please upload a valid flight ticket with passenger information.',
        errorCode: ERROR_CODES.NO_PASSENGER_NAMES_FOUND,
      };
    }

    // Normalize userName for comparison (remove extra spaces, convert to lowercase)
    const normalizedUserName = userName.trim().toLowerCase();

    // Split username into parts for partial matching
    const userNameParts = normalizedUserName.split(/\s+/).filter((part) => part.length > 1);

    this.logger.log(
      `[VALIDATION] Normalized user name: "${normalizedUserName}", Parts: [${userNameParts.join(', ')}]`,
    );

    // Check all segments for passenger name match
    for (const segment of structuredResult.segments) {
      if (segment.passengerNames && Array.isArray(segment.passengerNames)) {
        this.logger.log(
          `[VALIDATION] Checking ${segment.type} segment with ${segment.passengerNames.length} passenger(s): [${segment.passengerNames.join(', ')}]`,
        );

        for (const passengerName of segment.passengerNames) {
          if (this.isNameMatch(passengerName, normalizedUserName, userNameParts)) {
            this.logger.log(
              `[VALIDATION] Passenger name match found: "${passengerName}" matches user "${userName}"`,
            );
            return { isValid: true };
          }
        }
      }
    }

    this.logger.error(
      `[VALIDATION] Passenger name validation failed - no match found for user: ${userName}`,
    );
    return {
      isValid: false,
      message:
        'The uploaded flight ticket does not belong to the specified user. Please upload a ticket with your name.',
      errorCode: ERROR_CODES.PASSENGER_NAME_MISMATCH,
    };
  }

  private isNameMatch(
    passengerName: string,
    normalizedUserName: string,
    userNameParts: string[],
  ): boolean {
    if (!passengerName || typeof passengerName !== 'string') return false;

    const normalizedPassengerName = passengerName.trim().toLowerCase();

    // Remove common prefixes like Mr., Mrs., Ms., Dr., etc.
    const cleanPassengerName = normalizedPassengerName.replace(
      /^(mr\.?|mrs\.?|ms\.?|dr\.?|prof\.?|capt\.?|col\.?|lt\.?|maj\.?)\s+/i,
      '',
    );

    this.logger.log(
      `[DEBUG] Comparing: User="${normalizedUserName}" vs Passenger="${cleanPassengerName}"`,
    );

    // Exact match
    if (cleanPassengerName === normalizedUserName) {
      this.logger.log(`[VALIDATION] Exact match found: "${cleanPassengerName}"`);
      return true;
    }

    // Check if passenger name contains all parts of user name
    const allPartsMatch = userNameParts.every((part) => cleanPassengerName.includes(part));

    if (allPartsMatch && userNameParts.length > 0) {
      this.logger.log(`[VALIDATION] All name parts match: ${userNameParts.join(', ')}`);
      return true;
    }

    // Enhanced matching for various name patterns
    const cleanPassengerParts = cleanPassengerName.split(/\s+/).filter((part) => part.length > 0);

    // CASE 1: First name + abbreviated last name (e.g., "Venkat T" vs "Venkat Tammareddy")
    if (userNameParts.length >= 2 && cleanPassengerParts.length >= 2) {
      const userFirstName = userNameParts[0];
      const userLastName = userNameParts[userNameParts.length - 1]; // Get the last part as surname

      const passengerFirstName = cleanPassengerParts[0];
      const passengerLastPart = cleanPassengerParts[cleanPassengerParts.length - 1];

      // Check if first names match and last name starts with same letter
      if (
        passengerFirstName === userFirstName &&
        passengerLastPart.length === 1 &&
        passengerLastPart.toLowerCase() === userLastName.charAt(0).toLowerCase()
      ) {
        this.logger.log(
          `[VALIDATION] Name match found with abbreviated surname: "${cleanPassengerName}" matches "${normalizedUserName}" (${userFirstName} + ${userLastName} → ${passengerFirstName} + ${passengerLastPart})`,
        );
        return true;
      }

      // CASE 2: Check if first names match and passenger has abbreviated middle + last name
      if (passengerFirstName === userFirstName && cleanPassengerParts.length >= 2) {
        // Check if any subsequent part in passenger name could be an abbreviation of user's name parts
        for (let i = 1; i < cleanPassengerParts.length; i++) {
          const passengerPart = cleanPassengerParts[i];
          if (passengerPart.length === 1) {
            // Check against all user name parts (middle names, last name)
            for (let j = 1; j < userNameParts.length; j++) {
              if (passengerPart.toLowerCase() === userNameParts[j].charAt(0).toLowerCase()) {
                this.logger.log(
                  `[VALIDATION] Name match found with abbreviated name part: "${cleanPassengerName}" matches "${normalizedUserName}" (${passengerPart} = ${userNameParts[j][0]})`,
                );
                return true;
              }
            }
          }
        }
      }
    }

    // CASE 3: Middle name handling (e.g., "John Michael Smith" vs "John M Smith" or "J M Smith")
    if (userNameParts.length >= 3 && cleanPassengerParts.length >= 2) {
      const userFirst = userNameParts[0];
      const userMiddle = userNameParts[1];
      const userLast = userNameParts[userNameParts.length - 1];

      // Check patterns like "John M Smith" vs "John Michael Smith"
      if (cleanPassengerParts.length === 3) {
        const [passengerFirst, passengerMiddle, passengerLast] = cleanPassengerParts;

        if (
          passengerFirst === userFirst &&
          passengerLast === userLast &&
          (passengerMiddle === userMiddle ||
            (passengerMiddle.length === 1 &&
              passengerMiddle.toLowerCase() === userMiddle.charAt(0).toLowerCase()))
        ) {
          this.logger.log(
            `[VALIDATION] Name match found with middle name pattern: "${cleanPassengerName}" matches "${normalizedUserName}"`,
          );
          return true;
        }
      }
    }

    // CASE 4: First letter abbreviations (e.g., "V R Tammareddy" vs "Venkat Ram Tammareddy")
    if (userNameParts.length >= 2 && cleanPassengerParts.length >= 2) {
      let matchCount = 0;
      const minLength = Math.min(userNameParts.length, cleanPassengerParts.length);

      for (let i = 0; i < minLength; i++) {
        const userPart = userNameParts[i];
        const passengerPart = cleanPassengerParts[i];

        // Exact match or abbreviation match
        if (
          userPart === passengerPart ||
          (passengerPart.length === 1 &&
            passengerPart.toLowerCase() === userPart.charAt(0).toLowerCase()) ||
          (userPart.length === 1 &&
            userPart.toLowerCase() === passengerPart.charAt(0).toLowerCase())
        ) {
          matchCount++;
        }
      }

      // If most parts match (at least 2 and >= 70% of parts)
      if (matchCount >= 2 && matchCount >= Math.ceil(minLength * 0.7)) {
        this.logger.log(
          `[VALIDATION] Name match found with abbreviation pattern: "${cleanPassengerName}" matches "${normalizedUserName}" (${matchCount}/${minLength} parts match)`,
        );
        return true;
      }
    }

    // CASE 5: Check reverse - if user name contains all significant parts of passenger name
    const passengerParts = cleanPassengerName.split(/\s+/).filter((part) => part.length > 1);
    const passengerPartsInUser = passengerParts.filter((part) => normalizedUserName.includes(part));

    // If at least 50% of passenger name parts are in user name and we have at least 2 matching parts
    if (
      passengerParts.length > 0 &&
      passengerPartsInUser.length >= Math.max(2, Math.ceil(passengerParts.length * 0.5))
    ) {
      this.logger.log(
        `[VALIDATION] Name match found with partial matching: ${passengerPartsInUser.length}/${passengerParts.length} parts match`,
      );
      return true;
    }

    // CASE 6: Last resort - fuzzy matching for typos (if names are similar length and have significant overlap)
    if (userNameParts.length === 1 && cleanPassengerParts.length === 1) {
      const userName = userNameParts[0];
      const passengerName = cleanPassengerParts[0];

      if (userName.length >= 4 && passengerName.length >= 4) {
        const similarity = this.calculateSimilarity(userName, passengerName);
        if (similarity >= 0.85) {
          // 85% similarity threshold
          this.logger.log(
            `[VALIDATION] Name match found with fuzzy matching: "${cleanPassengerName}" matches "${normalizedUserName}" (${Math.round(similarity * 100)}% similar)`,
          );
          return true;
        }
      }
    }

    this.logger.log(
      `[DEBUG] No match found for: "${cleanPassengerName}" vs "${normalizedUserName}"`,
    );
    return false;
  }

  // Helper method for fuzzy string matching
  private calculateSimilarity(str1: string, str2: string): number {
    const longer = str1.length > str2.length ? str1 : str2;
    const shorter = str1.length > str2.length ? str2 : str1;

    if (longer.length === 0) return 1.0;

    const editDistance = this.levenshteinDistance(longer, shorter);
    return (longer.length - editDistance) / longer.length;
  }

  // Levenshtein distance for fuzzy matching
  private levenshteinDistance(str1: string, str2: string): number {
    const matrix: number[][] = Array(str2.length + 1)
      .fill(null)
      .map(() => Array(str1.length + 1).fill(0) as number[]);

    for (let i = 0; i <= str1.length; i++) matrix[0][i] = i;
    for (let j = 0; j <= str2.length; j++) matrix[j][0] = j;

    for (let j = 1; j <= str2.length; j++) {
      for (let i = 1; i <= str1.length; i++) {
        const substitutionCost = str1[i - 1] === str2[j - 1] ? 0 : 1;
        matrix[j][i] = Math.min(
          matrix[j][i - 1] + 1, // insertion
          matrix[j - 1][i] + 1, // deletion
          matrix[j - 1][i - 1] + substitutionCost, // substitution
        );
      }
    }

    return matrix[str2.length][str1.length];
  }

  private filterFlightDataForUI(structuredResult: any): any {
    this.logger.log('[VALIDATION] Filtering flight data for UI response');

    if (!structuredResult?.segments) {
      return structuredResult;
    }

    // Create a deep copy and filter each segment
    const filteredResult = {
      ...structuredResult,
      segments: structuredResult.segments.map((segment: any) => {
        const filteredSegment = { ...segment };

        // For onward segments, remove goingTo and departureDateTime
        if (segment.type === 'onward') {
          if (filteredSegment.goingTo) {
            delete filteredSegment.goingTo;
            this.logger.log('[VALIDATION] Removed goingTo from onward segment for UI');
          }
          if (filteredSegment.departureDateTime) {
            delete filteredSegment.departureDateTime;
            this.logger.log('[VALIDATION] Removed departureDateTime from onward segment for UI');
          }
        }

        // For return segments, remove comingFrom and arrivalDateTime
        if (segment.type === 'return') {
          if (filteredSegment.comingFrom) {
            delete filteredSegment.comingFrom;
            this.logger.log('[VALIDATION] Removed comingFrom from return segment for UI');
          }
          if (filteredSegment.arrivalDateTime) {
            delete filteredSegment.arrivalDateTime;
            this.logger.log('[VALIDATION] Removed arrivalDateTime from return segment for UI');
          }
        }

        return filteredSegment;
      }),
    };

    this.logger.log('[VALIDATION] Flight data filtering completed for UI');
    return filteredResult;
  }

  private async extractProgramCity(programDetails: any): Promise<string | null> {
    if (!programDetails) return null;

    // Get venue address from program details
    const venueAddress = programDetails.venue || programDetails.Venue;

    if (venueAddress && typeof venueAddress === 'string' && venueAddress.trim()) {
      // Use OpenAI to extract city from the venue address string
      const extractedCity = await this.extractCityUsingOpenAI(venueAddress.trim());
      return extractedCity;
    }

    return null;
  }

  private async extractCityUsingOpenAI(venueAddress: string): Promise<string | null> {
    if (!venueAddress || typeof venueAddress !== 'string') return null;

    const systemPrompt = extractCitySystemPrompt;
    const userPrompt = extractCityUserPrompt(venueAddress);

    try {
      const response = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0.1, 50);
      const cityName = response.trim();

      // Validate the response
      if (cityName && cityName !== 'Unknown' && cityName.length > 2 && !/^\d+$/.test(cityName)) {
        return cityName;
      }

      return null;
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Unknown error occurred';
      this.logger.error('[ERROR] Failed to extract city using OpenAI:', errorMessage);
      return null;
    }
  }

  private isCityMatch(flightCity: string, programCity: string): boolean {
    if (!flightCity || !programCity) return false;

    const normalizeCity = (city: string): string => {
      return city
        .toLowerCase()
        .trim()
        .replace(/[^\w\s]/g, '') // Remove special characters
        .replace(/\s+/g, ' '); // Normalize spaces
    };

    const normalizedFlightCity = normalizeCity(flightCity);
    const normalizedProgramCity = normalizeCity(programCity);

    // Direct match
    if (normalizedFlightCity === normalizedProgramCity) {
      return true;
    }

    // Check if one city contains the other (for cases like "New Delhi" vs "Delhi")
    if (
      normalizedFlightCity.includes(normalizedProgramCity) ||
      normalizedProgramCity.includes(normalizedFlightCity)
    ) {
      return true;
    }

    // Handle common city name variations
    const cityVariations: Record<string, string[]> = {
      'new delhi': ['delhi', 'newdelhi'],
      delhi: ['new delhi', 'newdelhi'],
      mumbai: ['bombay'],
      bombay: ['mumbai'],
      bangalore: ['bengaluru', 'bangluru'],
      bengaluru: ['bangalore', 'bangluru'],
      kolkata: ['calcutta'],
      calcutta: ['kolkata'],
      chennai: ['madras'],
      madras: ['chennai'],
    };

    const variations = cityVariations[normalizedFlightCity] || [];
    return variations.includes(normalizedProgramCity);
  }

  private isAirportCode(location: string): boolean {
    // Check if the location looks like an airport code (3-4 uppercase letters)
    if (!location || typeof location !== 'string') return false;
    const airportCodePattern = /^[A-Z]{3,4}$/;
    return airportCodePattern.test(location.trim());
  }

  async analyzeTheDocument(info: ExtractionInfo): Promise<any> {
    this.logger.log(`[INFO] Starting document analysis. Type: ${info.type}, URL: ${info.url}`);

    let result;
    if (info.type === DocumentTypeEnum.FLIGHT_TICKET) {
      this.logger.log('[INFO] Processing flight ticket document');
      result = await this.analyzeFlightTicket(
        info.url,
        info.programDetails,
        info.userName,
        info.journeyType,
      );
    } else if (info.type === DocumentTypeEnum.IDENTITY_DOCUMENT) {
      this.logger.log('[INFO] Processing identity document');

      // Validate that docType is provided for identity documents
      if (!info.docType) {
        this.logger.error(
          '[VALIDATION] Document type is required for identity document processing',
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.IDENTITY_DOCUMENT_TYPE_REQUIRED);
      }

      this.logger.log(`[INFO] Processing identity document of type: ${info.docType}`);
      result = await this.analyzeIdentityDocumentFromS3(info.url, info.userName, info.docType);
    } else {
      this.logger.error(`[VALIDATION] Unsupported document type: ${info.type}`);
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniBadRequestException(
        ERROR_CODES.FAILED_TO_ANALYZE,
        new Error(`Unsupported document type: ${info.type}`),
      );
    }

    this.logger.log('[INFO] Document analysis completed successfully');
    return result;
  }

  async analyzeIdentityDocumentFromS3(
    s3Url: string,
    userName: string,
    docType: string,
  ): Promise<any> {
    try {
      this.logger.log(`[IDENTITY_DOC] Starting identity document analysis for user: ${userName}`);
      this.logger.log(`[IDENTITY_DOC] Expected document type: ${docType}`);
      this.logger.log(`[IDENTITY_DOC] S3 URL: ${s3Url}`);

      const base64 = await this.fetchFileAsBase64(s3Url);
      this.logger.log(
        `[IDENTITY_DOC] Successfully fetched file as base64, length: ${base64.length}`,
      );

      // Note: File format validation is already done in fetchFileAsBase64 method
      this.logger.log('[IDENTITY_DOC] File format validation passed for identity document');

      const content = await this.processDocumentWithDocumentIntelligence(base64);
      this.logger.log(
        `[IDENTITY_DOC] Document Intelligence processed, content length: ${content?.length || 0}`,
      );

      // Handle cases with no or very little content
      if (!content || content.trim().length < 10) {
        this.logger.error(
          `[IDENTITY_DOC] Identity document content too short or empty. Content length: ${content?.length || 0}, Trimmed length: ${content?.trim().length || 0}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.FAILED_TO_EXTRACT_IDENTITY_DATA);
      }

      this.logger.log(
        `[IDENTITY_DOC] Identity document content extracted successfully. Content length: ${content.length}`,
      );

      // First detect what type of document was actually uploaded
      this.logger.log(
        '[IDENTITY_DOC] About to call _getIdentityDocumentType to detect actual document type',
      );
      const detectedDocumentType = await this._getIdentityDocumentType(content);
      this.logger.log(`[IDENTITY_DOC] _getIdentityDocumentType returned: ${detectedDocumentType}`);

      // Validate that the detected document type matches the user-selected type
      const allowedTypes = ['Passport', 'Voter id', 'Aadhar', 'Pan card', 'Driving license'];

      if (!allowedTypes.includes(detectedDocumentType)) {
        this.logger.error(
          `[IDENTITY_DOC] Unsupported identity document type detected: ${detectedDocumentType}. Allowed types: ${allowedTypes.join(', ')}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_IDENTITY_DOCUMENT,
          new Error('The uploaded document is not a supported identity document.'),
        );
      }

      // Simple document type comparison - case insensitive
      const isDocumentTypeMatch = (selected: string, detected: string): boolean => {
        const normalize = (type: string) => type.toLowerCase().trim();
        return normalize(selected) === normalize(detected);
      };

      if (!isDocumentTypeMatch(docType, detectedDocumentType)) {
        this.logger.error(
          `[VALIDATION] Document type mismatch: User selected '${docType}' but detected '${detectedDocumentType}'`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(ERROR_CODES.IDENTITY_DOCUMENT_TYPE_MISMATCH);
      }

      this.logger.log(
        `[VALIDATION] Document type validation passed: Selected '${docType}' matches detected '${detectedDocumentType}'`,
      );

      // Use the detected document type for processing (which should match the selected type)
      const documentType = detectedDocumentType;

      this.logger.log('[IDENTITY_DOC] About to call _extractIdentityDocumentFields');
      const extractedData = await this._extractIdentityDocumentFields(content, documentType);
      this.logger.log(
        `[IDENTITY_DOC] _extractIdentityDocumentFields returned data with keys: ${Object.keys((extractedData as Record<string, any>) || {}).join(', ')}`,
      );

      this.logger.log('[VALIDATION] Identity document fields extracted successfully');

      // Check if the document is valid based on the enhanced validation
      if (!extractedData.isValidDocument) {
        this.logger.error(
          `[VALIDATION] Document validation failed: ${extractedData.validationReason}`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_IDENTITY_DOCUMENT,
          new Error(
            (extractedData.validationReason as string) ||
              'The uploaded document is not a valid identity document. Please upload a valid government-issued identity document.',
          ),
        );
      }

      // Check if the document number was extracted
      if (!extractedData.travelInfoNumber) {
        this.logger.error(
          `[VALIDATION] No ${documentType} number could be extracted from the document`,
        );
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.FAILED_TO_EXTRACT_IDENTITY_DATA,
          new Error(
            `Could not extract ${documentType.toLowerCase()} number from the document. Please ensure the document is clear and readable.`,
          ),
        );
      }

      // For Aadhar cards, log whether we got a full or masked number
      if (documentType === 'Aadhar') {
        const aadharNumber = extractedData.travelInfoNumber as string;
        const isMasked =
          aadharNumber.includes('X') || aadharNumber.includes('*') || aadharNumber.includes('x');
        this.logger.log(
          `[VALIDATION] Aadhar number extracted: ${isMasked ? 'Masked format detected' : 'Full number detected'} - ${aadharNumber}`,
        );

        // Validate that we have at least some meaningful number information
        const hasVisibleDigits = /\d{4}/.test(aadharNumber); // At least 4 consecutive digits
        if (!hasVisibleDigits) {
          this.logger.error(
            `[VALIDATION] Aadhar number does not contain sufficient visible digits: ${aadharNumber}`,
          );
          // eslint-disable-next-line @typescript-eslint/only-throw-error
          throw new InifniBadRequestException(
            ERROR_CODES.FAILED_TO_EXTRACT_IDENTITY_DATA,
            new Error(
              'Could not extract sufficient Aadhar number information from the document. Please ensure the document shows at least the last 4 digits clearly.',
            ),
          );
        }
      }

      // Validate document holder name against provided username
      const nameValidation = this.validateIdentityDocumentName(extractedData, userName);

      this.logger.log(
        `[VALIDATION] Identity document name validation result: ${nameValidation.isValid ? 'PASSED' : 'FAILED'} ${nameValidation.message || ''}`,
      );

      // Prepare response object with name validation results
      const responseData = {
        ...extractedData,
        isNameValid: nameValidation.isValid,
      };

      // Add warning message if name validation failed
      if (!nameValidation.isValid) {
        this.logger.warn(
          `[VALIDATION] Identity document name validation failed: ${nameValidation.message}`,
        );

        responseData.message =
          'The Name in the document does not match with the user name in the application';

        this.logger.log(
          '[VALIDATION] Continuing with document processing despite name mismatch - returning data with warning',
        );
      } else {
        this.logger.log('[VALIDATION] Identity document name validation passed successfully');
      }

      this.logger.log(
        `[VALIDATION] Document validation completed successfully. Type: ${extractedData.documentType}, Country: ${extractedData.issuingCountry || 'Unknown'}`,
      );

      return responseData;
    } catch (err: any) {
      // Handle InifniBadRequestException first - it has its own message property
      if (err instanceof InifniBadRequestException) {
        this.logger.error(`[ERROR] Re-throwing InifniBadRequestException: ${err.message}`);
        throw err;
      }

      const errorMessage = err instanceof Error ? err.message : 'Unknown error occurred';
      this.logger.error(`[ERROR] Identity document analysis failed: ${errorMessage}`);

      // Handle specific JSON parsing errors
      if (errorMessage.includes('Failed to parse JSON from OpenAI response')) {
        this.logger.error('[ERROR] JSON parsing failed - OpenAI response format issue');
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.FAILED_TO_EXTRACT_IDENTITY_DATA,
          new Error(
            'Failed to process the document. The AI service returned an invalid response format. Please try again or contact support if the issue persists.',
          ),
        );
      }

      // Handle document type determination errors
      if (errorMessage.includes('Failed to determine document type')) {
        this.logger.error('[ERROR] Document type determination failed');
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_IDENTITY_DOCUMENT,
          new Error(
            'Could not identify the document type. Please ensure you have uploaded a clear, valid identity document.',
          ),
        );
      }

      // Handle field extraction errors
      if (errorMessage.includes('Failed to extract') && errorMessage.includes('document fields')) {
        this.logger.error('[ERROR] Document field extraction failed');
        // eslint-disable-next-line @typescript-eslint/only-throw-error
        throw new InifniBadRequestException(
          ERROR_CODES.FAILED_TO_EXTRACT_IDENTITY_DATA,
          new Error(
            'Could not extract information from the document. Please ensure the document is clear, complete, and readable.',
          ),
        );
      }

      // For any other errors, wrap in a generic error
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniBadRequestException(
        ERROR_CODES.FAILED_TO_ANALYZE,
        new Error(errorMessage || 'An unexpected error occurred during identity document analysis'),
      );
    }
  }

  private validateIdentityDocumentName(
    extractedData: any,
    userName: string,
  ): { isValid: boolean; message?: string; errorCode?: string } {
    this.logger.log(
      `[VALIDATION] Starting identity document name validation for user: ${userName}`,
    );

    // Check if holder name was extracted from the document
    if (!extractedData.holderName || !extractedData.holderName.trim()) {
      this.logger.error('[VALIDATION] No holder name found in identity document');
      return {
        isValid: false,
        message:
          'Could not extract holder name from the identity document. Please ensure the document is clear and readable.',
        errorCode: ERROR_CODES.FAILED_TO_EXTRACT_IDENTITY_DATA,
      };
    }

    // Use the same name matching logic as flight tickets
    const normalizedUserName = userName.trim().toLowerCase();
    const normalizedDocumentName = (extractedData.holderName as string).trim().toLowerCase();

    // Split names into parts for partial matching
    const userNameParts = normalizedUserName.split(/\s+/).filter((part) => part.length > 1);

    this.logger.log(
      `[VALIDATION] Identity document name comparison: User="${normalizedUserName}" vs Document="${normalizedDocumentName}"`,
    );

    // Check if document name matches user name using fuzzy matching logic
    if (this.isNameMatch(normalizedDocumentName, normalizedUserName, userNameParts)) {
      this.logger.log(
        `[VALIDATION] Identity document name match found: "${extractedData.holderName}" matches user "${userName}"`,
      );
      return { isValid: true };
    }

    this.logger.error(
      `[VALIDATION] Identity document name validation failed - no match found for user: ${userName}`,
    );
    return {
      isValid: false,
      message:
        'The uploaded identity document does not belong to the specified user. Please upload a document with your name.',
      errorCode: ERROR_CODES.IDENTITY_DOCUMENT_NAME_MISMATCH,
    };
  }

  private async _getIdentityDocumentType(content: string): Promise<string> {
    try {
      this.logger.log('[IDENTITY_DOC] _getIdentityDocumentType: Starting document type detection');

      const systemPrompt = extractInvoiceDataSystemPrompt;
      const userPrompt = extractInvoiceDataUserPrompt(content);

      this.logger.log(
        `[IDENTITY_DOC] _getIdentityDocumentType: System prompt length: ${systemPrompt.length}`,
      );
      this.logger.log(
        `[IDENTITY_DOC] _getIdentityDocumentType: User prompt length: ${userPrompt.length}`,
      );
      this.logger.log(
        `[IDENTITY_DOC] _getIdentityDocumentType: Content sample: ${content.substring(0, 200)}...`,
      );

      this.logger.log('[IDENTITY_DOC] _getIdentityDocumentType: Calling OpenAI...');
      const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0, 100);

      this.logger.log(`[IDENTITY_DOC] _getIdentityDocumentType: OpenAI raw response: ${raw}`);

      this.logger.log('[IDENTITY_DOC] _getIdentityDocumentType: About to parse JSON response...');
      const docTypeResult = this.parseOpenAIJsonResponse(raw);

      this.logger.log(
        `[IDENTITY_DOC] _getIdentityDocumentType: Parsed result: ${JSON.stringify(docTypeResult)}`,
      );

      if (!docTypeResult || typeof docTypeResult !== 'object') {
        this.logger.error('[IDENTITY_DOC] _getIdentityDocumentType: Invalid response format');
        throw new Error('Invalid response format from AI service for document type detection');
      }

      const documentType = (docTypeResult.documentType?.trim() as string) || '';
      this.logger.log(
        `[IDENTITY_DOC] _getIdentityDocumentType: Final document type: ${documentType}`,
      );

      if (!documentType) {
        this.logger.error('[IDENTITY_DOC] _getIdentityDocumentType: No document type extracted');
        throw new Error('Could not determine document type from the uploaded file');
      }

      return documentType;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error occurred';
      this.logger.error(
        `[IDENTITY_DOC] _getIdentityDocumentType: Failed with error: ${errorMessage}`,
      );

      // Check if it's a JSON parsing error specifically
      if (errorMessage.includes('Failed to parse JSON from OpenAI response')) {
        this.logger.error('[IDENTITY_DOC] _getIdentityDocumentType: JSON parsing error detected');
      }

      throw new Error(`Failed to determine document type: ${errorMessage}`);
    }
  }

  private async _extractIdentityDocumentFields(
    content: string,
    documentType: string,
  ): Promise<any> {
    try {
      this.logger.log(
        `[IDENTITY_DOC] _extractIdentityDocumentFields: Starting field extraction for ${documentType}`,
      );

      const systemPrompt = extractIdentityDocumentSystemPrompt;
      const userPrompt = extractIdentityDocumentUserPrompt(content, documentType);

      this.logger.log(
        `[IDENTITY_DOC] _extractIdentityDocumentFields: System prompt length: ${systemPrompt.length}`,
      );
      this.logger.log(
        `[IDENTITY_DOC] _extractIdentityDocumentFields: User prompt length: ${userPrompt.length}`,
      );

      this.logger.log(
        `[IDENTITY_DOC] _extractIdentityDocumentFields: Calling OpenAI for ${documentType} field extraction...`,
      );
      const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0, 300);

      this.logger.log(`[IDENTITY_DOC] _extractIdentityDocumentFields: OpenAI raw response: ${raw}`);

      this.logger.log(
        '[IDENTITY_DOC] _extractIdentityDocumentFields: About to parse JSON response...',
      );
      const extractedData = this.parseOpenAIJsonResponse(raw);

      this.logger.log(
        `[IDENTITY_DOC] _extractIdentityDocumentFields: Parsed result: ${JSON.stringify(extractedData)}`,
      );

      if (!extractedData || typeof extractedData !== 'object') {
        this.logger.error('[IDENTITY_DOC] _extractIdentityDocumentFields: Invalid response format');
        throw new Error('Invalid response format from AI service for document field extraction');
      }

      return extractedData;
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error occurred';
      this.logger.error(
        `[IDENTITY_DOC] _extractIdentityDocumentFields: Failed with error: ${errorMessage}`,
      );

      // Check if it's a JSON parsing error specifically
      if (errorMessage.includes('Failed to parse JSON from OpenAI response')) {
        this.logger.error(
          '[IDENTITY_DOC] _extractIdentityDocumentFields: JSON parsing error detected',
        );
      }

      throw new Error(`Failed to extract ${documentType} document fields: ${errorMessage}`);
    }
  }
}
