import * as winston from 'winston'; // Import winston logger for logs
import * as fs from 'fs'; // Import fs module
import * as path from 'path'; // Import path module
import WinstonCloudWatch from 'winston-cloudwatch';

// Function to generate log file paths dynamically
function getLogFilePaths() {
  const now = new Date();
  const year = now.getFullYear();
  const month = String(now.getMonth() + 1).padStart(2, '0');
  const day = String(now.getDate()).padStart(2, '0');

  // Define log directory relative to the project root
  const logDir = path.join(process.cwd(), `logs/${year}-${month}-logs`);
  if (!fs.existsSync(logDir)) {
    fs.mkdirSync(logDir, { recursive: true });
  }

  return {
    errorLog: path.join(logDir, `${year}-${month}-${day}-error.log`),
    systemLog: path.join(logDir, `${year}-${month}-${day}-system.log`),
    dateString: `${year}-${month}-${day}` // Used for CloudWatch log stream names
  };
}

// Keep track of current date
let { errorLog, systemLog, dateString } = getLogFilePaths();

// Function to update transports dynamically
function updateTransports() {
  const newPaths = getLogFilePaths();
  if (newPaths.dateString !== dateString) {
    console.log(`[Logger] Updating log file paths for ${newPaths.dateString}`);
    
    // Update global log file paths
    errorLog = newPaths.errorLog;
    systemLog = newPaths.systemLog;
    dateString = newPaths.dateString;

    // Update Winston transports
    createLogger.clear();
    createLogger.add(new winston.transports.Console({
      level: 'silly',
      format: winston.format.combine(winston.format.colorize(), logFormat),
    }));
    createLogger.add(new winston.transports.File({ filename: errorLog, level: 'error' }));
    createLogger.add(new winston.transports.File({ filename: systemLog, level: 'silly' }));
    createLogger.add(new WinstonCloudWatch({
      logGroupName: `WebSocket-Error-Logs-${process.env.NODE_ENV}`,
      logStreamName: `WebSocket-Error-${process.env.NODE_ENV}-${dateString}`,
      awsAccessKeyId: process.env.AWS_ACCESS_KEY_ID,
      awsSecretKey: process.env.AWS_SECRET_ACCESS_KEY,
      awsRegion: process.env.AWS_REGION,
      messageFormatter: ({ level, message }) => `[${level}] : ${message}`,
    }));
    createLogger.add(new WinstonCloudWatch({
      logGroupName: `WebSocket-System-Logs-${process.env.NODE_ENV}`,
      logStreamName: `WebSocket-System-${process.env.NODE_ENV}-${dateString}`,
      awsAccessKeyId: process.env.AWS_ACCESS_KEY_ID,
      awsSecretKey: process.env.AWS_SECRET_ACCESS_KEY,
      awsRegion: process.env.AWS_REGION,
      messageFormatter: ({ level, message }) => `[${level}] : ${message}`,
    }));
  }
}

// Create a log format
const logFormat = winston.format.printf(({ timestamp, level, message }) => {
  return `[Infinipath-Websocket] ${timestamp} [${level.toUpperCase()}]: ${message}`;
});

// Create Winston logger instance
export const createLogger = winston.createLogger({
  level: 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.json(),
    logFormat
  ),
  transports: [
    new winston.transports.Console({
      level: 'silly',
      format: winston.format.combine(winston.format.colorize(), logFormat),
    }),
    new winston.transports.File({ filename: errorLog, level: 'error' }),
    new winston.transports.File({ filename: systemLog, level: 'silly' }),
    new WinstonCloudWatch({
      logGroupName: `WebSocket-Error-Logs-${process.env.NODE_ENV}`,
      logStreamName: `WebSocket-Error-${process.env.NODE_ENV}-${dateString}`,
      awsAccessKeyId: process.env.AWS_ACCESS_KEY_ID,
      awsSecretKey: process.env.AWS_SECRET_ACCESS_KEY,
      awsRegion: process.env.AWS_REGION,
      messageFormatter: ({ level, message }) => `[${level}] : ${message}`,
    }),
    new WinstonCloudWatch({
      logGroupName: `WebSocket-System-Logs-${process.env.NODE_ENV}`,
      logStreamName: `WebSocket-System-${process.env.NODE_ENV}-${dateString}`,
      awsAccessKeyId: process.env.AWS_ACCESS_KEY_ID,
      awsSecretKey: process.env.AWS_SECRET_ACCESS_KEY,
      awsRegion: process.env.AWS_REGION,
      messageFormatter: ({ level, message }) => `[${level}] : ${message}`,
    }),
  ],
});

// Check every minute if the date has changed, and update log files accordingly
setInterval(updateTransports, 60 * 1000);
