import { io } from "../app"; // Import io server from main app file
import { getUserWebinar } from "../controllers"; // Import getUserWebinar function from controllers
import { toCamelCase, scheduleJob } from "./helpers"; // Import toCamelCase and scheduleJob functions from helpers
import { createLogger } from "./winstonLogger"; // Import the logger function
const logger = createLogger; // create a logger instance
import { cancelScheduledJobs } from "../utils/helpers";
/**
 * Creates a webinar and schedules related jobs for registration start, registration end,
 * webinar start, webinar end, and join reminders.
 * 
 * @async
 * @function createWebinar
 * @param {Object} webinarDetails - Details of the webinar including time schedules
 * @param {Date} webinarDetails.registrationStartsAt - Registration start time
 * @param {Date} webinarDetails.registrationEndsAt - Registration end time
 * @param {Date} webinarDetails.startDate - Webinar start date and time
 * @param {number} webinarDetails.duration - Duration of the webinar in minutes
 * @param {Date} webinarDetails.endDate - Webinar end date and time
 * @param {string} webinarDetails.webinarId - Unique identifier for the webinar
 * @param {number} webinarDetails.joinTimeEnable - Time before the webinar when users can join
 * @param {Map<string, string>} userSocketMap - A map of user IDs to their socket IDs
 * @returns {Promise<void>}
 */
export async function createWebinar(
  webinarDetails: {
    registrationStartsAt: Date;
    registrationEndsAt: Date;
    startDate: Date;
    duration: number;
    endDate: Date;
    webinarId: string;
    joinTimeEnable: number;
  },
  userSocketMap: any
) {
  logger.info(`Creating webinar and scheduling jobs...:${JSON.stringify(webinarDetails, null, 2)}`);
  // Convert all times to UTC
  const registrationStartsAtUTC = new Date(
    webinarDetails.registrationStartsAt.getTime() - 5.5 * 60 * 60 * 1000
  );
  const registrationEndsAtUTC = new Date(
    webinarDetails.registrationEndsAt.getTime() - 5.5 * 60 * 60 * 1000
  );
  const startDateUTC = new Date(
    webinarDetails.startDate.getTime() - 5.5 * 60 * 60 * 1000
  );
  const endDateUTC = new Date(
    webinarDetails.endDate.getTime() - 5.5 * 60 * 60 * 1000
  );

  const joinReminderUTC = new Date(
    webinarDetails.startDate.getTime() -
      webinarDetails.joinTimeEnable * 60 * 1000 -
      5.5 * 60 * 60 * 1000
  );

  let webinarId = webinarDetails.webinarId;
  cancelScheduledJobs(webinarId);
  // Schedule registration start job
  scheduleJob(webinarId, registrationStartsAtUTC, "Registration Start", async () => {
    logger.info(`Registration start job triggered.`);
    try {
      const registeredUserIds = Array.from(userSocketMap.keys());
      logger.info(`Registered user IDs for Registration start job triggered: ${registeredUserIds}`);
      registeredUserIds.forEach(async (userId: string) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: webinarId,
        });
        const socketIds = userSocketMap.get(userId);
        if (getUserWebinarData) {
          socketIds.forEach((socketId: string) => {
          io.to(socketId).emit("webinarData", toCamelCase(getUserWebinarData));
          logger.info(
            `Emitted webinar data to user ${userId} with socket ID ${socketId} for registration start triggered${webinarId}`
          );
        })
        } else {
          logger.info(
            `User ${userId} is not registered for registration start ${webinarId}`
          );
        }
      });
    } catch (error) {
      logger.info(`Error in registration start job: ${error}`);
    }
  });

  // Schedule registration end job
  scheduleJob(webinarId, registrationEndsAtUTC, "Registration End", async () => {
    logger.info(`Registration end job triggered.`);
    try {
      const registeredUserIds = Array.from(userSocketMap.keys());
      logger.info(`Registered user IDs for Registration end job triggered: ${registeredUserIds}`);
      registeredUserIds.forEach(async (userId: string) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: webinarId,
        });
        const socketIds = userSocketMap.get(userId);
        if (getUserWebinarData) {
          socketIds.forEach((socketId: string) => {
          io.to(socketId).emit("webinarData", toCamelCase(getUserWebinarData));
          logger.info(
            `Emitted webinar data to user ${userId} with socket ID ${socketId} for registration end triggered ${webinarId}`
          );
        });
        } else {
          logger.info(
            `User ${userId} is not registered for registration end ${webinarId}`
          );
        }
      });
    } catch (error) {
      logger.info(`Error in registration end job: ${error}`);
    }
  });

  // Schedule webinar start job
  scheduleJob(webinarId, startDateUTC, "Webinar Start", async () => {
    logger.info(`Webinar start job triggered.`);
    try {
      const registeredUserIds = Array.from(userSocketMap.keys());
      logger.info(`Registered user IDs for Webinar start job triggered: ${registeredUserIds}`);
      registeredUserIds.forEach(async (userId: string) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: webinarId,
        });
        const socketIds = userSocketMap.get(userId);
        if (getUserWebinarData && getUserWebinarData[0].is_registered) {
          socketIds.forEach((socketId: string) => {
          io.to(socketId).emit("webinarData", toCamelCase(getUserWebinarData));
          logger.info(
            `Emitted webinar data to user ${userId} with socket ID ${socketId} for webinar start triggered ${webinarId}`
          );
        });
        } else {
          logger.info(
            `User ${userId} is not registered for webinar start ${webinarId}`
          );
        }
      });
    } catch (error) {
      logger.info(`Error in webinar start job: ${error}`);
    }
  });

  // Schedule webinar end job
  scheduleJob(webinarId, endDateUTC, "Webinar End", async () => {
    logger.info(`Webinar end job triggered.`);
    try {
      const registeredUserIds = Array.from(userSocketMap.keys());
      logger.info(`Registered user IDs for Webinar end job triggered: ${registeredUserIds}`);
      registeredUserIds.forEach(async (userId: string) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: webinarId,
          end: true,
        });

        const socketIds = userSocketMap.get(userId);

        if (getUserWebinarData) {
          socketIds.forEach((socketId: string) => {
          io.to(socketId).emit("webinarData", toCamelCase(getUserWebinarData));
          logger.info(
            `Emitted webinar data to user ${userId} with socket ID ${socketId} for webinar end triggered ${webinarId}`
          );
        });
        } else {
          logger.info(
            `User ${userId} is not registered for webinar end ${webinarId}`
          );
        }
      });
    } catch (error) {
      logger.info(`Error in webinar end job: ${error}`);
    }
  });
                
  // Schedule join reminder job
  scheduleJob(webinarId, joinReminderUTC, "Join Reminder", async () => {
    logger.info(`Join reminder job triggered.`);
    try {
      const registeredUserIds = Array.from(userSocketMap.keys());
      logger.info(`Registered user IDs for Join reminder job triggered: ${registeredUserIds}`);
      registeredUserIds.forEach(async (userId: string) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: webinarId,
        });

        const socketIds = userSocketMap.get(userId);
        if (getUserWebinarData && getUserWebinarData[0].is_registered) {
          socketIds.forEach((socketId: string) => {
          io.to(socketId).emit("webinarData", toCamelCase(getUserWebinarData));
          logger.info(
            `Emitted webinar data to user ${userId} with socket ID ${socketId} for webinar join reminder triggered ${webinarId}`
          );
        });
        } else {
          logger.info(
            `User ${userId} is not registered for webinar join reminder ${webinarId}`
          );
        }
      });
    } catch (error) {
      logger.info(`Error in join reminder job: ${error}`);
    }
  });
}
