import fs from "fs"; // Import fs module
import path from "path"; // Import path module
import schedule from "node-schedule"; // Import node-schedule module
import { io } from "../app"; // Import io server from main app file
import { webinarInfo } from "../services"; // Import webinarInfo from services
import { modifyWebinarData } from "../controllers"; // Import modifyWebinarData function from controllers
import { createLogger } from "./winstonLogger"; // Import the logger function
const logger = createLogger; // create a logger instance
const webinarFilePath = path.join(__dirname, "../../webinar.json"); // Path to the webinar JSON file
const scheduledJobs = new Map<string, schedule.Job[]>(); // Stores webinar jobs by webinarId
/**
 * Reads webinar data from a JSON file or database and emits it via WebSocket.
 * If the file is empty or an error occurs, data is fetched from the database.
 *
 * @async
 * @function emitWebinarData
 * @returns {Promise<void>} - Emits webinar data via WebSocket
 */
export const emitWebinarData = async () => {
  fs.readFile(webinarFilePath, "utf8", async (err, data) => {
    if (err || !data) {
      logger.info(`File data not present`);
      logger.error(`Error reading webinar file or file is empty: ${err}`);
      // If there's an error reading the file or the file is empty, fetch data from the database
      try {
        logger.info(`Fetch webinar data from the database`);
        const result = await webinarInfo(); // Fetch webinar data from the database
        logger.info(`Emitting the webinarData1 : ${result}`);
        io.emit("webinarData", result); // Emit the webinar data
      } catch (dbErr) {
        logger.error(`Database query error:: ${dbErr}`);
      }
    } else {
      logger.info(`File data is present`);
      const webinars = await modifyWebinarData(JSON.parse(data)); // Modify the webinar data from the file
      if (webinars.length === 0) {
        try {
          logger.info(`Fetch webinar data from the database`);
          const result = await webinarInfo(); // Fetch webinar data from the database
          logger.info(`Emitting the webinarData : ${result}`);
          io.emit("webinarData1", result); // Emit the webinar data
        } catch (dbErr) {
          logger.error(`Database query error:: ${dbErr}`);
        }
      } else {
        // Emit the data from the file
        logger.info(`Emitting webinar data from file"`);
        io.emit("webinarData1", webinars); // Emit the webinar data
      }
    }
  });
};

/**
 * Converts an object's keys from snake_case to camelCase recursively.
 * 
 * @function toCamelCase
 * @param {any} obj - The object or array to be converted
 * @returns {any} - The object with camelCased keys
 */
export const toCamelCase = (obj: any): any => {
  if (Array.isArray(obj)) {
    return obj.map((v) => toCamelCase(v));
  } else if (obj !== null && obj.constructor === Object) {
    return Object.keys(obj).reduce(
      (result, key) => ({
        ...result,
        [key.replace(/_([a-z])/g, (g) => g[1].toUpperCase())]: toCamelCase(
          obj[key]
        ),
      }),
      {}
    );
  }
  return obj;
};

/**
 * Schedules a job to be executed at a specific date and time.
 * 
 * @function scheduleJob
 * @param {Date} date - The date and time when the job should run
 * @param {string} jobName - A name for the scheduled job
 * @param {Function} callback - The function to execute when the job runs
 * @returns {void}
 */
// export function scheduleJob(date: Date, jobName: string, callback: () => void) {
//   schedule.scheduleJob(date, callback);
//   logger.info(`${jobName} scheduled for ${date}`);
// }

export function scheduleJob(
  webinarId: string,
  date: Date,
  jobName: string,
  callback: () => void
) {
  const job = schedule.scheduleJob(date, callback);
  logger.info(`${jobName} scheduled for ${date}`);

  if (!scheduledJobs.has(webinarId)) {
    scheduledJobs.set(webinarId, []);
  }
  scheduledJobs.get(webinarId)?.push(job); // Store job reference
}

export function cancelScheduledJobs(webinarId: string) {
  if (scheduledJobs.has(webinarId)) {
    scheduledJobs.get(webinarId)?.forEach((job) => job.cancel());
    scheduledJobs.delete(webinarId);
    logger.info(`All scheduled jobs for webinar ${webinarId} have been canceled.`);
  }
}