import { Request, Response } from "express"; // Import express module for request and response
import fs from "fs"; // Import fs module
import path from "path"; // Import path module
import { registrationStatus } from "../common/constants"; // Import registration status from constants
import {
  getUserData,
  getUserWebinarData,
  getExpiredUserWebinarData,
  getUserWebinarDetails,
  getUserWebinarRegistrationAttempts,
  getWebinarDataById,
  getWebinarRegistrationCounts,
} from "../services"; // Import socket service funtions from services
import { io } from "../app"; // Import io server from main app file
import { createWebinar } from "../utils/scheduler"; // Import createWebinar from scheduler
import { toCamelCase } from "../utils/helpers"; // Import toCamelCase functions from helpers
import { createLogger } from "../utils/winstonLogger"; // Import the logger function
const logger = createLogger; // create a logger instance
const webinarFilePath = path.join(__dirname, "../../webinar.json"); // Path to the webinar JSON file

export const helloWorld = (req: Request, res: Response) => {
  try {
    logger.info(`helloWorld function data: ${req}`);
    res.send("Hello World");
  } catch (error) {
    logger.error(`Error in helloWorld: ${error}`);
    res.status(500).send("Internal Server Error");
  }
};

export const updateWebinarDataInFile = async (newWebinarData: any) => {
  try {
    logger.info(`updateWebinarDataInFile function data :${newWebinarData}`);
    // Read the existing webinar data from the file
    return fs.readFile(webinarFilePath, "utf8", (err, data) => {
      if (err) {
        logger.error(`Error reading webinar file: ${err}`);
        throw new Error("Internal Server Error");
      }

      let webinars = [];
      if (data) {
        webinars = JSON.parse(data);
      }

      // Check if the webinar with the same id already exists
      const existingWebinar = webinars.find(
        (webinar: any) => webinar.id === newWebinarData.id
      );

      if (existingWebinar) {
        throw new Error("Webinar with this ID already exists");
      }

      // Add the new webinar data to the list
      webinars.push(newWebinarData);

      // Filter out webinars with endDate greater than the current date
      const currentDate: Date = new Date();
      currentDate.setTime(currentDate.getTime() + 5.5 * 60 * 60 * 1000);
      webinars = webinars.filter(
        (webinar: any) => new Date(webinar.endDate) >= currentDate
      );

      // Write the updated data back to the file
      fs.writeFile(
        webinarFilePath,
        JSON.stringify(webinars, null, 2),
        (err) => {
          if (err) {
            logger.error(`Error writing to webinar file: ${err}`);
            throw new Error("Internal Server Error");
          }
          logger.info(`Webinar data updated successfully`);
        }
      );
    });
  } catch (error) {
    logger.error(`Error in updateWebinarData : ${error}`);
    throw error;
  }
};

export const updataWebinarData = async (req: Request, res: Response) => {
  try {
    logger.info(`updataWebinarData function data: ${JSON.parse(req.body)}`);
    const newWebinarData = req.body;

    // Read the existing webinar data from the file
    fs.readFile(webinarFilePath, "utf8", (err, data) => {
      if (err) {
        logger.error(`Error reading webinar file: ${err}`);
        return res.status(500).send("Internal Server Error");
      }

      let webinars = [];
      if (data) {
        webinars = JSON.parse(data);
      }

      // Check if the webinar with the same id already exists
      const existingWebinar = webinars.find(
        (webinar: any) => webinar.id === newWebinarData.id
      );

      if (existingWebinar) {
        return res.status(400).send("Webinar with this ID already exists");
      }

      // Add the new webinar data to the list
      webinars.push(newWebinarData);

      // Write the updated data back to the file
      fs.writeFile(
        webinarFilePath,
        JSON.stringify(webinars, null, 2),
        (err) => {
          if (err) {
            logger.error(`Error writing to webinar file: ${err}`);
            return res.status(500).send("Internal Server Error");
          }

          return res.status(200).send("Webinar data updated successfully");
        }
      );
      return 1;
    });
  } catch (error) {
    logger.error(`Error in updataWebinarData: ${error}`);
    res.status(500).send("Internal Server Error");
  }
};

export const modifyWebinarData = async (data: any) => {
  try {
    logger.info(`modifyWebinarData function data:${data}`);
    let currentDateTime: any = new Date();
    // UTC to IST
    currentDateTime = new Date(
      currentDateTime.getTime() + 5.5 * 60 * 60 * 1000
    );
    const responseArray = [];

    for (const userMeeting of data) {
      let responseObject: any = {};
      let registration_status: string = "";
      let is_panelist_registration_available: boolean = false;
      let is_attendee_registration_available: boolean = false;
      let is_join: boolean = false;

      const meetingStartTime = new Date(userMeeting.startAt);
      let meetingEndTime = new Date(
        meetingStartTime.getTime() + userMeeting.duration * 60 * 1000
      );
      let joinStartTime = userMeeting.joinEnableTime
        ? new Date(
            meetingStartTime.getTime() - userMeeting.joinEnableTime * 60 * 1000
          )
        : new Date(meetingStartTime.getTime() - 15 * 60 * 1000);
      const joinEndTime = meetingEndTime;

      if (currentDateTime < meetingStartTime) {
        logger.info(`modifyWebinarData currentDateTime: ${currentDateTime}`);
        if (currentDateTime < new Date(userMeeting.registrationStartsAt)) {
          logger.info(`modifyWebinarData notOpened: ${currentDateTime}`);
          registration_status = registrationStatus.notOpened;
          is_panelist_registration_available = false;
          is_attendee_registration_available = false;
          is_join = false;
          responseObject.registrationStartsAt =
            userMeeting.registrationStartsAt;
          responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
          responseObject.startDate = userMeeting.startAt;
          responseObject.duration = userMeeting.duration;
        } else if (
          currentDateTime >= new Date(userMeeting.registrationStartsAt) &&
          currentDateTime <= new Date(userMeeting.registrationEndsAt)
        ) {
          logger.info(`modifyWebinarData opened: ${registrationStatus.opened}`);
          const registrationCount = await getWebinarRegistrationCounts(
            userMeeting.id
          );
          const nonPanelistCount = registrationCount.attendeeCount;
          const panelistRegistrations = registrationCount.panelistCount;

          if (panelistRegistrations < userMeeting.max_video_limit) {
            is_panelist_registration_available = true;
          } else {
            is_panelist_registration_available = false;
          }
          // responseObject.is_panelist_registration_available =
          //   is_panelist_registration_available;

          if (nonPanelistCount < userMeeting.max_non_video_limit) {
            is_attendee_registration_available = true;
          } else {
            is_attendee_registration_available = false;
          }
          // responseObject.is_attendee_registration_available =
          //   is_attendee_registration_available;

          registration_status = registrationStatus.opened;
          is_join = false;
          responseObject.registrationStartsAt =
            userMeeting.registrationStartsAt;
          responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
          responseObject.startDate = userMeeting.startAt;
          responseObject.duration = userMeeting.duration;
          responseObject["meetingId"] = userMeeting.id;

          if (currentDateTime >= joinStartTime) {
            responseObject["meetingId"] = userMeeting.id;
            is_join = true;
          } else {
            is_join = false;
          }
        } else if (currentDateTime > new Date(userMeeting.registrationEndsAt)) {
          logger.info(`modifyWebinarData closed: ${registrationStatus.closed}`);
          registration_status = registrationStatus.closed;
          is_panelist_registration_available = false;
          is_attendee_registration_available = false;
          responseObject["meetingId"] = userMeeting.id;
          if (currentDateTime >= joinStartTime) {
            is_join = true;
          } else {
            is_join = false;
          }
          responseObject.registrationStartsAt =
            userMeeting.registrationStartsAt;
          responseObject.startDate = userMeeting.startAt;
          responseObject.duration = userMeeting.duration;
          responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
        }
      } else if (
        currentDateTime >= meetingStartTime &&
        currentDateTime < joinEndTime
      ) {
        logger.info(`modifyWebinarData started`);
        registration_status = registrationStatus.closed;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        responseObject["meetingId"] = userMeeting.id;
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.duration = userMeeting.duration;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
      } else if (currentDateTime > joinEndTime) {
        registration_status = registrationStatus.ended;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        // is_join = false;
      }

      responseObject.registrationStatus = registration_status;
      responseObject.isPanelistRegistrationAvailable =
        is_panelist_registration_available;
      responseObject.isAttendeeRegistrationAvailable =
        is_attendee_registration_available;
      responseObject.isJoin = is_join;
      responseObject.title = userMeeting.title;
      responseObject.id = userMeeting.id;
      responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
      responseArray.push(responseObject);
    }
    // logger.info(`modifyWebinarData responseArray: ${responseArray}`);
    return responseArray;
  } catch (error) {
    logger.error(`Error in modifyWebinarData: ${error}`);
    return error;
  }
};

export const modifySingleWebinarData = async (id: any, userSocketMap: any) => {
  try {
    logger.info(`modifySingleWebinarData function data id: ${id}`);
    const userMeeting = await getWebinarDataById(id);

    let currentDateTime: any = new Date();
    // UTC to IST
    currentDateTime = new Date(
      currentDateTime.getTime() + 5.5 * 60 * 60 * 1000
    );
    const responseArray = [];
    logger.info(`modifySingleWebinarData userMeeting data: ${JSON.stringify(userMeeting, null, 2)}`);
    let responseObject: any = {};
    let registration_status: string = "";
    let is_panelist_registration_available: boolean = false;
    let is_attendee_registration_available: boolean = false;
    let is_join: boolean = false;

    const meetingStartTime = new Date(userMeeting.startAt);
    let meetingEndTime = new Date(
      meetingStartTime.getTime() + userMeeting.duration * 60 * 1000
    );
    let joinStartTime = userMeeting.joinEnableTime
      ? new Date(
          meetingStartTime.getTime() - userMeeting.joinEnableTime * 60 * 1000
        )
      : new Date(meetingStartTime.getTime() - 15 * 60 * 1000);
    const joinEndTime = meetingEndTime;

    if (currentDateTime < meetingStartTime) {
      logger.info(
        `modifySingleWebinarData currentDateTime: ${currentDateTime}`
      );
      if (currentDateTime < new Date(userMeeting.registrationStartsAt)) {
        logger.info(
          `modifySingleWebinarData notOpened: ${registrationStatus.notOpened}`
        );
        registration_status = registrationStatus.notOpened;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        is_join = false;
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.endAt = joinEndTime;
        responseObject.duration = userMeeting.duration;
        responseObject["meetingId"] = userMeeting.id;
      } else if (
        currentDateTime >= new Date(userMeeting.registrationStartsAt) &&
        currentDateTime <= new Date(userMeeting.registrationEndsAt)
      ) {
        logger.info(
          `modifySingleWebinarData opened: ${registrationStatus.opened}`
        );
        const registrationCount = await getWebinarRegistrationCounts(
          userMeeting.id
        );
        const nonPanelistCount = registrationCount.attendeeCount;
        const panelistRegistrations = registrationCount.panelistCount;

        if (panelistRegistrations < userMeeting.maxVideoLimit) {
          is_panelist_registration_available = true;
        } else {
          is_panelist_registration_available = false;
        }
        // responseObject.is_panelist_registration_available =
        //   is_panelist_registration_available;

        if (nonPanelistCount < userMeeting.maxNonVideoLimit) {
          is_attendee_registration_available = true;
        } else {
          is_attendee_registration_available = false;
        }
        // responseObject.is_attendee_registration_available =
        //   is_attendee_registration_available;

        registration_status = registrationStatus.opened;
        is_join = false;
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.endAt = joinEndTime;
        responseObject.duration = userMeeting.duration;
        responseObject["meetingId"] = userMeeting.id;

        if (currentDateTime >= joinStartTime) {
          responseObject["meetingId"] = userMeeting.id;
          is_join = true;
        } else {
          is_join = false;
        }
      } else if (currentDateTime > new Date(userMeeting.registrationEndsAt)) {
        logger.info(
          `modifySingleWebinarData closed: ${registrationStatus.closed}`
        );
        registration_status = registrationStatus.closed;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        responseObject["meetingId"] = userMeeting.id;
        if (currentDateTime >= joinStartTime) {
          is_join = true;
        } else {
          is_join = false;
        }
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.endAt = joinEndTime;
        responseObject.duration = userMeeting.duration;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
      }
    } else if (
      currentDateTime >= meetingStartTime &&
      currentDateTime < joinEndTime
    ) {
      logger.info(`modifySingleWebinarData started`);
      registration_status = registrationStatus.closed;
      is_panelist_registration_available = false;
      is_attendee_registration_available = false;
      responseObject["meetingId"] = userMeeting.id;
      responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
      responseObject.startDate = userMeeting.startAt;
      responseObject.endAt = joinEndTime;
      responseObject.duration = userMeeting.duration;
      responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
    } else if (currentDateTime > joinEndTime) {
      // registration_status = registrationStatus.notOpened;
      // is_panelist_registration_available = false;
      // is_attendee_registration_available = false;
      // is_join = false;
      registration_status = registrationStatus.ended;
      is_panelist_registration_available = false;
      is_attendee_registration_available = false;
      is_join = false;
      responseObject["meetingId"] = userMeeting.id;
      responseObject.endAt = joinEndTime;
      responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
      responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
      responseObject.startDate = userMeeting.startAt;
      responseObject.duration = userMeeting.duration;
    }

    responseObject.registrationStatus = registration_status;
    responseObject.isPanelistRegistrationAvailable =
      is_panelist_registration_available;
    responseObject.isAttendeeRegistrationAvailable =
      is_attendee_registration_available;
    responseObject.isJoin = is_join;
    responseObject.title = userMeeting.title;
    responseObject.id = userMeeting.id;
    responseArray.push(responseObject);
    logger.info(`modifySingleWebinarData responseArray: ${JSON.stringify(responseArray, null, 2)}`);
    // Example usage
    const webinarDetails = {
      registrationStartsAt: responseObject.registrationStartsAt,
      registrationEndsAt: responseObject.registrationEndsAt,
      startDate: responseObject.startDate,
      duration: responseObject.duration,
      endDate: responseObject.endAt,
      webinarId: responseObject.id,
      joinTimeEnable: userMeeting.joinEnableTime,
    };

    createWebinar(webinarDetails, userSocketMap);

    const registeredUserIds = Array.from(userSocketMap.keys());
    logger.info(`modifySingleWebinarData Emit data only to connected users`);
    await Promise.all(
      registeredUserIds.map(async (userId) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: webinarDetails.webinarId,
        });
        const socketIds = userSocketMap.get(userId);
        if (getUserWebinarData) {
          socketIds.forEach((socketId: string) => {
            io.to(socketId).emit(
              "webinarData",
              toCamelCase(getUserWebinarData)
            );
            logger.info(
              `Emitted webinar data to user ${userId} with socket ID ${socketId} for webinar start ${webinarDetails.webinarId}`
            );
          });
        } else {
          logger.info(
            `User ${userId} is not registered for webinar start ${webinarDetails.webinarId}`
          );
        }
      })
    );
    return responseArray;
  } catch (error) {
    logger.error(`Error in modifyWebinarData: ${error}`);
    return error;
  }
};

export const getUserWebinar = async (data: any) => {
  logger.info(`getUserWebinar function data: ${JSON.stringify(data, null, 2)}`);
  const responseArray = [];
  const webinarId = data.webinarId;
  const userId = data.userId;
  const userData = await getUserData(userId);
  let userMeeting;
  if (!data.end) {
    userMeeting = await getUserWebinarData(webinarId);
  } else {
    userMeeting = await getExpiredUserWebinarData(webinarId);
  }
  let responseObject: any = {};

  let registration_status: string = "";
  let is_panelist_registration_available: boolean = false;
  let is_attendee_registration_available: boolean = false;
  let is_join: boolean = false;

  if (userMeeting) {
    const meetingStartTime = userMeeting.start_at;
    responseObject["title"] = userMeeting?.title || "Weekly Growth Session";
    let meetingEndTime = new Date(
      meetingStartTime.getTime() + userMeeting.duration * 60 * 1000
    );

    let joinStartTime: Date;
    if (userMeeting.joinEnableTime) {
      joinStartTime = new Date(
        meetingStartTime.getTime() - userMeeting.joinEnableTime * 60 * 1000
      );
    } else {
      joinStartTime = new Date(meetingStartTime.getTime() - 15 * 60 * 1000);
    }

    const joinEndTime = meetingEndTime;
    let currentDateTime: any = new Date();
    currentDateTime = new Date(
      currentDateTime.getTime() + 5.5 * 60 * 60 * 1000
    );

    if (currentDateTime < meetingStartTime) {
      if (currentDateTime < userMeeting.registration_starts_at) {
        registration_status = registrationStatus.notOpened;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        is_join = false;
        responseObject.registration_starts_at =
          userMeeting.registration_starts_at;
        responseObject.registration_ends_at = userMeeting.registration_ends_at;
        responseObject.start_date = userMeeting.start_at;
        responseObject.duration = userMeeting.duration;
        responseObject["meeting_id"] = userMeeting.id;
        responseObject.is_panelist_registration_available =
          is_panelist_registration_available;
        responseObject.is_attendee_registration_available =
          is_attendee_registration_available;
        responseObject.is_join = is_join;
        responseObject.is_registered = false;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
      } else if (
        currentDateTime >= userMeeting.registration_starts_at &&
        currentDateTime <= userMeeting.registration_ends_at
      ) {
        const registrationCount = await getWebinarRegistrationCounts(
          userMeeting.id
        );
        const nonPanelistCount = registrationCount.attendeeCount;
        const panelistRegistrations = registrationCount.panelistCount;

        if (panelistRegistrations < userMeeting.max_video_limit) {
          is_panelist_registration_available = true;
        } else {
          is_panelist_registration_available = false;
        }
        // responseObject.is_panelist_registration_available =
        //   is_panelist_registration_available;

        if (nonPanelistCount < userMeeting.max_non_video_limit) {
          is_attendee_registration_available = true;
        } else {
          is_attendee_registration_available = false;
        }
        // responseObject.is_attendee_registration_available =
        //   is_attendee_registration_available;
        responseObject.is_panelist_registration_available =
          is_panelist_registration_available;
        responseObject.is_attendee_registration_available =
          is_attendee_registration_available;

        registration_status = registrationStatus.opened;
        is_join = false;
        responseObject.registration_starts_at =
          userMeeting.registration_starts_at;
        responseObject.registration_ends_at = userMeeting.registration_ends_at;
        responseObject.start_date = userMeeting.start_at;
        responseObject.duration = userMeeting.duration;
        responseObject["meeting_id"] = userMeeting.id;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
        if (currentDateTime >= joinStartTime) {
          responseObject["meeting_id"] = userMeeting.id;
          is_join = true;
        } else {
          is_join = false;
        }
      } else if (currentDateTime > userMeeting.registration_ends_at) {
        registration_status = registrationStatus.closed;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        responseObject["meeting_id"] = userMeeting.id;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
        if (currentDateTime >= joinStartTime) {
          responseObject["meeting_id"] = userMeeting.id;
          is_join = true;
        } else {
          is_join = false;
        }

        responseObject.registration_starts_at =
          userMeeting.registration_starts_at;
        responseObject.start_date = userMeeting.start_at;
        responseObject.duration = userMeeting.duration;
        responseObject.registration_ends_at = userMeeting.registration_ends_at;
        responseObject.is_panelist_registration_available =
          is_panelist_registration_available;
        responseObject.is_attendee_registration_available =
          is_attendee_registration_available;
      }
    } else if (
      currentDateTime >= meetingStartTime &&
      currentDateTime < joinEndTime
    ) {
      registration_status = registrationStatus.closed;
      is_panelist_registration_available = false;
      is_attendee_registration_available = false;
      responseObject["meeting_id"] = userMeeting.id;
      responseObject.join_enable_minutes = userMeeting.joinEnableTime;
      responseObject.registration_starts_at =
        userMeeting.registration_starts_at;
      responseObject.start_date = userMeeting.start_at;
      responseObject.registration_ends_at = userMeeting.registration_ends_at;
      responseObject.duration = userMeeting.duration;
      responseObject.is_panelist_registration_available =
        is_panelist_registration_available;
      responseObject.is_attendee_registration_available =
        is_attendee_registration_available;
    } else if (currentDateTime > joinEndTime) {
      is_join = false;
      registration_status = registrationStatus.ended;
      is_panelist_registration_available = false;
      is_attendee_registration_available = false;
      responseObject["meetingId"] = data.webinarId;
      responseObject.join_enable_minutes = userMeeting.joinEnableTime;
      responseObject.endAt = joinEndTime;
      responseObject.end_date = joinEndTime;
      responseObject.join_end_time = joinEndTime;
      responseObject.registration_ends_at = userMeeting.registration_ends_at;
      responseObject.registration_starts_at =
        userMeeting.registration_starts_at;
      responseObject.start_date = userMeeting.start_at;
      responseObject.duration = userMeeting.duration;
    }

    responseObject["registration_status"] = registration_status;

    let is_registered: boolean = true;
    let is_eligible: boolean = true;

    if (
      currentDateTime >= userMeeting.registration_starts_at &&
      currentDateTime < joinEndTime
    ) {
      const userMeetingDetails = await getUserWebinarDetails(
        userData.id,
        userMeeting.id
      );
      if (!userMeetingDetails) {
        is_registered = false;
        const registrationCount = await getWebinarRegistrationCounts(
          userMeeting.id
        );
        const nonPanelistCount = registrationCount.attendeeCount;
        const panelistRegistrations = registrationCount.panelistCount;

        if (currentDateTime < userMeeting.registration_ends_at) {
          if (panelistRegistrations < userMeeting.max_video_limit) {
            is_panelist_registration_available = true;
          } else {
            is_panelist_registration_available = false;
          }
          responseObject.is_panelist_registration_available =
            is_panelist_registration_available;

          if (nonPanelistCount < userMeeting.max_non_video_limit) {
            is_attendee_registration_available = true;
          } else {
            is_attendee_registration_available = false;
          }
          responseObject.is_attendee_registration_available =
            is_attendee_registration_available;
        }
      } else if (userMeetingDetails.deleted_at) {
        is_registered = false;
        is_eligible = true;
        const registrationCount = await getWebinarRegistrationCounts(
          userMeeting.id
        );
        const nonPanelistCount = registrationCount.attendeeCount;
        const panelistRegistrations = registrationCount.panelistCount;

        if (currentDateTime < userMeeting.registration_ends_at) {
          if (panelistRegistrations < userMeeting.max_video_limit) {
            is_panelist_registration_available = true;
          } else {
            is_panelist_registration_available = false;
          }
          responseObject.is_panelist_registration_available =
            is_panelist_registration_available;

          if (nonPanelistCount < userMeeting.max_non_video_limit) {
            is_attendee_registration_available = true;
          } else {
            is_attendee_registration_available = false;
          }
          responseObject.is_attendee_registration_available =
            is_attendee_registration_available;
        }
      } else {
        responseObject.start_date = meetingStartTime;
        responseObject.duration = userMeeting.duration;
        responseObject.end_date = meetingEndTime;
        responseObject.join_end_time = joinEndTime;
        responseObject["title"] = userMeeting.title;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;

        if (
          currentDateTime >= meetingStartTime &&
          currentDateTime < joinEndTime
        ) {
          is_join = true;
        }

        responseObject["is_panelist"] = userMeetingDetails.is_panelist;
        responseObject["meeting_id"] = userMeeting.id;
      }

      responseObject["is_registered"] = is_registered;
      responseObject["is_eligible"] = is_eligible;
      responseObject["is_join"] = is_join;
    }
    if (
      currentDateTime > userMeeting.registration_starts_at &&
      currentDateTime > joinEndTime
    ) {
      const userMeetingDetails = await getUserWebinarDetails(
        userData.id,
        userMeeting.id
      );
      userMeetingDetails && userMeetingDetails.deleted_at == null
        ? (is_registered = true)
        : (is_registered = false);
      is_eligible = is_join = false;
      responseObject["is_registered"] = is_registered;
      responseObject["is_eligible"] = is_eligible;
      responseObject["is_join"] = is_join;
      responseObject["is_panelist_registration_available"] =
        is_panelist_registration_available;
      responseObject["is_attendee_registration_available"] =
        is_attendee_registration_available;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
    }
    if (
      currentDateTime >= userMeeting.registration_starts_at &&
      currentDateTime < userMeeting.registration_ends_at
    ) {
      const { attendee_attempts, panelist_attempts } =
        await getUserWebinarRegistrationAttempts(userData.id, userMeeting.id);
      responseObject["attendee_registration_attempts"] = attendee_attempts;
      responseObject["panelist_registration_attempts"] = panelist_attempts;
    }
  }
  logger.info(`getUserWebinar responseObject : ${JSON.stringify(responseObject, null, 2)}`);
  responseArray.push(responseObject);
  return responseArray;
};

export const modifySingleWebinarDataClone = async (id: any) => {
  try {
    logger.info(`modifySingleWebinarDataClone function data id: ${id}`);
    const userMeeting = await getWebinarDataById(id);
    logger.info(`modifySingleWebinarDataClone userMeeting: ${JSON.parse(JSON.stringify(userMeeting))}`);
    let currentDateTime: any = new Date();
    // UTC to IST
    currentDateTime = new Date(
      currentDateTime.getTime() + 5.5 * 60 * 60 * 1000
    );
    const responseArray = [];
    let responseObject: any = {};
    let registration_status: string = "";
    let is_panelist_registration_available: boolean = false;
    let is_attendee_registration_available: boolean = false;
    let is_join: boolean = false;

    const meetingStartTime = new Date(userMeeting.startAt);
    let meetingEndTime = new Date(
      meetingStartTime.getTime() + userMeeting.duration * 60 * 1000
    );
    let joinStartTime = userMeeting.joinEnableTime
      ? new Date(
          meetingStartTime.getTime() - userMeeting.joinEnableTime * 60 * 1000
        )
      : new Date(meetingStartTime.getTime() - 15 * 60 * 1000);
    const joinEndTime = meetingEndTime;

    if (currentDateTime < meetingStartTime) {
      logger.info(
        `modifySingleWebinarDataClone currentDateTime: ${currentDateTime}`
      );
      if (currentDateTime < new Date(userMeeting.registrationStartsAt)) {
        logger.info(
          `modifySingleWebinarDataClone notOpened: ${registrationStatus.notOpened}`
        );
        registration_status = registrationStatus.notOpened;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        is_join = false;
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.duration = userMeeting.duration;
        responseObject["meetingId"] = userMeeting.id;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
      } else if (
        currentDateTime >= new Date(userMeeting.registrationStartsAt) &&
        currentDateTime <= new Date(userMeeting.registrationEndsAt)
      ) {
        logger.info(
          `modifySingleWebinarDataClone opened: ${registrationStatus.opened}`
        );
        const registrationCount = await getWebinarRegistrationCounts(
          userMeeting.id
        );
        const nonPanelistCount = registrationCount.attendeeCount;
        const panelistRegistrations = registrationCount.panelistCount;

        if (panelistRegistrations < userMeeting.maxVideoLimit) {
          is_panelist_registration_available = true;
        } else {
          is_panelist_registration_available = false;
        }
        // responseObject.is_panelist_registration_available =
        //   is_panelist_registration_available;

        if (nonPanelistCount < userMeeting.maxNonVideoLimit) {
          is_attendee_registration_available = true;
        } else {
          is_attendee_registration_available = false;
        }
        // responseObject.is_attendee_registration_available =
        //   is_attendee_registration_available;

        registration_status = registrationStatus.opened;
        is_join = false;
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.duration = userMeeting.duration;
        responseObject["meetingId"] = userMeeting.id;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
        if (currentDateTime >= joinStartTime) {
          responseObject["meetingId"] = userMeeting.id;
          is_join = true;
        } else {
          is_join = false;
        }
      } else if (currentDateTime > new Date(userMeeting.registrationEndsAt)) {
        logger.info(
          `modifySingleWebinarDataClone closed: ${registrationStatus.closed}`
        );
        registration_status = registrationStatus.closed;
        is_panelist_registration_available = false;
        is_attendee_registration_available = false;
        responseObject["meetingId"] = userMeeting.id;
        if (currentDateTime >= joinStartTime) {
          is_join = true;
        } else {
          is_join = false;
        }
        responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
        responseObject.startDate = userMeeting.startAt;
        responseObject.duration = userMeeting.duration;
        responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
        responseObject.join_enable_minutes = userMeeting.joinEnableTime;
      }
    } else if (
      currentDateTime >= meetingStartTime &&
      currentDateTime < joinEndTime
    ) {
      logger.info(`modifySingleWebinarDataClone started`);
      registration_status = registrationStatus.closed;
      is_panelist_registration_available = false;
      is_attendee_registration_available = false;
      responseObject["meetingId"] = userMeeting.id;
      responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
      responseObject.startDate = userMeeting.startAt;
      responseObject.duration = userMeeting.duration;
      responseObject.registrationEndsAt = userMeeting.registrationEndsAt;
      responseObject.join_enable_minutes = userMeeting.joinEnableTime;
    } else if (currentDateTime > joinEndTime) {
      registration_status = registrationStatus.ended;
      is_panelist_registration_available = false;
      is_attendee_registration_available = false;
      is_join = false;
      responseObject["meetingId"] = userMeeting.id;
      responseObject.join_enable_minutes = userMeeting.joinEnableTime;
      responseObject.registrationStartsAt = userMeeting.registrationStartsAt;
      responseObject.startDate = userMeeting.startAt;
      responseObject.duration = userMeeting.duration;
    }

    responseObject.registrationStatus = registration_status;
    responseObject.isPanelistRegistrationAvailable =
      is_panelist_registration_available;
    responseObject.isAttendeeRegistrationAvailable =
      is_attendee_registration_available;
    responseObject.isJoin = is_join;
    responseObject.title = userMeeting.title;
    responseObject.id = userMeeting.id;
    responseArray.push(responseObject);
    logger.info(`modifySingleWebinarDataClone responseArray: ${responseArray}`);
    return responseArray;
  } catch (error) {
    logger.error(`Error in modifyWebinarData: ${error}`);
    return error;
  }
};

export const handleRegisterWebinar = async (
  data: any,
  userSocketMap: any,
  socket: any
) => {
  try {
    logger.info(`handleRegisterWebinar function data: ${JSON.stringify(data, null, 2)}`);
    const registeredUserIds = Array.from(userSocketMap.keys());
    await Promise.all(
      registeredUserIds.map(async (userId) => {
        const getUserWebinarData = await getUserWebinar({
          userId: userId,
          webinarId: data.webinarId,
        });
        const socketIds = userSocketMap.get(userId);
        logger.info(`handleRegisterWebinar socketIds : ${JSON.stringify(socketIds, null, 2)}`);
        if (getUserWebinarData) {
          socketIds.forEach((socketId: string) => {
            io.to(socketId).emit(
              "webinarData",
              toCamelCase(getUserWebinarData)
            );
            logger.info(
              `Emitted webinar data to user ${userId} with socket ID ${socketId} for registrer webinar start ${data.webinarId}`
            );
          });
        } else {
          logger.info(
            `User ${userId} is not registered for registrer webinar start ${data.webinarId}`
          );
        }
      })
    );
  } catch (error) {
    logger.error(`Error updating webinar data: ${JSON.stringify(error, null, 2)}`);
    socket.emit("error", { message: "Failed to update webinar data" });
  }
};
