import {
  Column,
  CreateDateColumn,
  Entity,
  PrimaryGeneratedColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
} from "typeorm";
import { Webinar } from "./webinar";

/**
 * Entity definition for webinar attendee table.
 * Represents a Webinar Attandance in the system
 */
@Entity({ name: "webinar_attendee" })
export class WebinarAttendee {
  @PrimaryGeneratedColumn()
  id: number;

  @Column({ name: "webinar_id" })
  webinarId: number;

  @Column({ name: "user_id" })
  userId: number;

  @Column({ name: "related_user_id", nullable: true })
  relatedUserId: number;

  @Column({ name: "mode_of_joining", type: "varchar", length: 50 })
  modeOfJoining: string;

  @Column({ name: "face_verification_status", type: "boolean", default: false })
  faceVerificationStatus: boolean;

  @Column({ name: "attendance_time", type: "timestamptz", nullable: true })
  attendanceTime: Date;

  @Column({ name: "device_type" })
  deviceType: string;

  @Column({ name: "device_info" })
  deviceInfo: string;

  @CreateDateColumn({
    name: "created_at",
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
  })
  createdAt: Date;

  @UpdateDateColumn({
    name: "updated_at",
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
    onUpdate: "CURRENT_TIMESTAMP",
  })
  updatedAt: Date;

  // Relationship with Webinar table
  @ManyToOne(() => Webinar)
  @JoinColumn({ name: "webinar_id" })
  webinar: Webinar;

  /**
   * Represents a constructor for a webinar attendance object.
   *
   * @constructor
   * @param {number} webinarId - The unique identifier of the webinar.
   * @param {number} userId - The unique identifier of the user attending the webinar.
   * @param {number} relatedUserId - The unique identifier of the related user (if any).
   * @param {string} modeOfJoining - The mode in which the user joins the webinar (e.g., 'online', 'phone').
   * @param {boolean} faceVerificationStatus - The status of the face verification (true if verified, false otherwise).
   * @param {Date} attendanceTime - The timestamp of when the user attends the webinar.
   * @param {string} deviceType - The type of device the user is using (e.g., 'mobile', 'desktop').
   * @param {string} deviceInfo - Additional details about the user's device (e.g., device model, operating system).
   */

  constructor(
    webinarId: number,
    userId: number,
    relatedUserId: number,
    modeOfJoining: string,
    faceVerificationStatus: boolean,
    attendanceTime: Date,
    deviceType: string,
    deviceInfo: string
  ) {
    this.webinarId = webinarId;
    this.userId = userId;
    this.relatedUserId = relatedUserId;
    this.modeOfJoining = modeOfJoining;
    this.faceVerificationStatus = faceVerificationStatus;
    this.attendanceTime = attendanceTime;
    this.deviceType = deviceType;
    this.deviceInfo = deviceInfo;
  }
}
