import { Injectable, Inject, forwardRef } from "@nestjs/common";
import { DataSource } from "typeorm";
import { UserRepository } from "./user.repository";
import { User } from "src/common/entities/user.entity";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import InifniBadRequestException from "src/common/exceptions/infini-badrequest-exception";
import { RegistrationRepository } from "src/registration/registration.repository";
import { ROLE_KEYS } from "src/common/constants/strings-constants";
import { UpdateUserDto } from "./dto/update-user.dto";
import { CreateUserDto } from "./dto/create-user.dto";
import { AppLoggerService } from "src/common/services/logger.service";

@Injectable()
export class UserService {
  constructor(
    private readonly userRepository: UserRepository,
    @Inject(forwardRef(() => RegistrationRepository))
    private readonly registrationRepository: RegistrationRepository,
    private readonly dataSource: DataSource,
     private readonly logger: AppLoggerService

  ) {}

  private readonly ROLE_NAME_TO_KEY: Record<string, string> = {
    viewer: ROLE_KEYS.VIEWER,
    admin: ROLE_KEYS.ADMIN,
    mahatria: ROLE_KEYS.MAHATRIA,
    rm: ROLE_KEYS.RM,
    finance_manager: ROLE_KEYS.FINANCE_MANAGER,
    relational_manager: ROLE_KEYS.RELATIONAL_MANAGER,
    shoba: ROLE_KEYS.SHOBA,
    operational_manger: ROLE_KEYS.OPERATIONAL_MANAGER,
  };

  /**
   * Fetches a user by phone number.
   * @param phoneNumber - The phone number to filter by.
   * @returns The user entity if found.
   */
  async getUserByPhoneNumber(phoneNumber: string): Promise<User> {
    try {
      return await this.userRepository.getUserByPhoneNumber(phoneNumber);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   * Fetches a user by combined phone number.
   * @param combinedPhone - The combined phone number to filter by.
   * @returns The user entity if found.
   * @throws {Error} If the user is not found.
   */
  async getUserByCombinedPhoneNumber(
    combinedPhone: string,
  ): Promise<User | null> {
    try {
      const formattedCombined = combinedPhone.trim();
      return await this.userRepository.getUserByCombinedPhoneNumber(
        formattedCombined,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  async getUserByCountryCodeAndPhoneNumber(
    countryCode: string,
    phoneNumber: string,
  ): Promise<User | null> {
    try {
      const formattedCountryCode = countryCode?.trim();
      const formattedPhoneNumber = phoneNumber?.trim();
      return await this.userRepository.simpleUserLookupByPhone(formattedCountryCode, formattedPhoneNumber);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  async getUserByEmail(email: string): Promise<User | null> {
    try {
      return await this.userRepository.getUserByEmail(email);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   * Fetches a user by ID.
   * @param id - The user ID.
   * @returns The user entity if found.
   * @throws {Error} If the user is not found.
   */
  async getUserById(id: number): Promise<User> {
    try {
      return await this.userRepository.getUserByField("id", id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   * Fetches a user by Firebase ID.
   * @param firebaseId - The Firebase ID to filter by.
   * @returns The user entity if found.
   * @throws {Error} If the user is not found.
   */
  async getUserByFirebaseId(firebaseId: string): Promise<User> {
    try {
      return await this.userRepository.getUserByFirebaseId(firebaseId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   * Fetches a user by ID.
   * @param id - The user ID.
   * @returns The user entity if found.
   */
  async getUserPrefilDataById(id: number): Promise<Partial<User> | null> {
    try {
      const data = await this.userRepository.getUserByField('id', id);
      if (!data) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }
      const formattedDataToPrefill: Record<string, any> = {
        id: data.id,
        name: data.fullName,
        gender: data.gender,
        mobileNumber: data.countryCode+data.phoneNumber,
        countryCode: data.countryCode,
        email: data.email,
        dob: data.dob,
        city: data.address,
        otherCityName: data.otherAddress,
        profileUrl: data.profileUrl,
      }
      const lastInvoice =
        await this.registrationRepository.findLastInvoiceDetailByUserId(id);
      const lastPayment =
        await this.registrationRepository.findLastPaymentDetailByUserId(id);
      const lastTravelPlan =
        await this.registrationRepository.findLastTravelPlanByUserId(id);
      const lastTravelInfo =
        await this.registrationRepository.findLastTravelInfoByUserId(id);

      // if (lastInvoice) {
      //   formattedDataToPrefill.invoiceEmail = lastInvoice.invoiceEmail;
      //   formattedDataToPrefill.tdsApplicable = lastInvoice.tdsApplicable;
      //   formattedDataToPrefill.invoiceName = lastInvoice.invoiceName;
      //   formattedDataToPrefill.tanNumber = lastInvoice.tanNumber;
      //   formattedDataToPrefill.invoiceAddress = lastInvoice.invoiceAddress;
      //   formattedDataToPrefill.isGstRegistered = lastInvoice.isGstRegistered;
      // }
      // if (lastPayment) {
      //   formattedDataToPrefill.originalPayment = lastPayment.originalAmount;
        // formattedDataToPrefill.paymentMode = lastPayment.paymentMode;
      // }
      // if (lastTravelInfo) {
      //   formattedDataToPrefill.userPictureUrl = lastTravelInfo.userPictureUrl;
      //   formattedDataToPrefill.pictureUrl = lastTravelInfo.idPictureUrl;
      //   formattedDataToPrefill.tShirtSize = lastTravelInfo.tshirtSize;
      //   formattedDataToPrefill.travelInfoType = lastTravelInfo.idType;
      //   formattedDataToPrefill.travelInfoNumber = lastTravelInfo.idNumber;
      // }
      // if (lastTravelPlan) {
      //   formattedDataToPrefill.airlineNumber = lastTravelPlan.airlineName;
      //   formattedDataToPrefill.flightNumber = lastTravelPlan.flightNumber;
      // }

      return formattedDataToPrefill;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  async createMinimalUser(data: {
    phoneNumber?: string;
    countryCode?: string;
    email?: string;
  }): Promise<User | null> {
    try {
      let user: User | null = null;
      await this.dataSource.transaction(async (transactionalEntityManager) => {
        // Create user within the transaction
        user = await this.userRepository.createUser(
          {
            phoneNumber: data.phoneNumber,
            countryCode: data.countryCode,
            email: data.email,
            role: 'viewer',
            termsAndConditions: true,
          },
          transactionalEntityManager,
        );

        if (!user) {
          throw new Error(ERROR_CODES.USER_SAVE_FAILED);
        }
        // Also create a UserRoleMap for the user within the transaction
        await this.userRepository.createUserRoleMap(
          user.id,
          ROLE_KEYS.VIEWER,
          transactionalEntityManager,
        );
      });
      return user;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SAVE_FAILED, error);
    }
  }

  async createUser(createUserDto: CreateUserDto): Promise<User | null> {
    try {
      let user: User | null = null;
      await this.dataSource.transaction(async (manager) => {
        user = await this.userRepository.createUser(
          {
            ...createUserDto,
            dob: createUserDto.dob ? new Date(createUserDto.dob) : undefined,
          },
          manager,
        );
        if (!user) {
          throw new Error(ERROR_CODES.USER_SAVE_FAILED);
        }
        const viewerRole = ROLE_KEYS.VIEWER;
        const mainRoleKey = this.ROLE_NAME_TO_KEY[createUserDto.role];
        await this.userRepository.createUserRoleMap(user.id, viewerRole, manager);
        if (createUserDto.role !== 'viewer' && mainRoleKey) {
          await this.userRepository.createUserRoleMap(user.id, mainRoleKey, manager);
        }
      });
      return user;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SAVE_FAILED, error);
    }
  }

  async getUserRoles(userId: number): Promise<string[]> {
    return await this.userRepository.getUserRolesById(userId);
  }

  async getRegistrationsByUser(
    userId: number,
    limit: number,
    offset: number,
    searchText: string,
    programId: number | null,
    programSessionId: number | null,
    parsedFilters: Record<string, any>,
  ) {
    try {
      return await this.registrationRepository.findRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText,
        { ...parsedFilters, createdBy: userId },
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async getAllUsers(
    limit: number,
    offset: number,
    searchText: string,
    parsedFilters: Record<string, any>,
  ) {
    try {
      return this.userRepository.getAllUsers(
        limit,
        offset,
        searchText,
        parsedFilters,
      );
    }
    catch (error) {
      handleKnownErrors(ERROR_CODES.USER_GET_ALL_FAILED, error);
    }
  }

  /**
   * Updates a user by ID
   * @param id - User ID to update
   * @param updateUserDto - Data to update
   * @returns Updated user entity
   */
  async updateUser(id: number, updateUserDto: UpdateUserDto): Promise<User> {
    this.logger.log(`Service: Updating user with ID: ${id}`, { updateData: updateUserDto });
    try {
      if ((updateUserDto as any).phoneNumber) {
        this.logger.error(`Attempted phone number update for user ${id}`);
        throw new InifniBadRequestException(
          ERROR_CODES.USER_PHONE_UPDATE_NOT_ALLOWED,
        );
      }

      if (updateUserDto.email) {
        const email = updateUserDto.email.toLowerCase();
        const existing = await this.userRepository.getUsersByEmail(email);
        if (existing && Array.isArray(existing)) {
          const otherUser = existing.find((user) => user.id !== id);
          if (otherUser) {
            this.logger.error(`Email already exists for user ID: ${otherUser.id}`);
            throw new InifniBadRequestException(
              ERROR_CODES.USER_EMAIL_ALREADY_EXISTS,
              null,
              null,
              email
            );
          }
        }
        updateUserDto.email = email;
      }

      const updatedUser = await this.userRepository.updateUser(id, updateUserDto);
      if (!updatedUser) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      if (updateUserDto.role) {
        await this.dataSource.transaction(async (manager) => {
          const existingRoles = updatedUser.userRoleMaps.map((urm) => urm.role.roleKey);
          const viewerRole = ROLE_KEYS.VIEWER;
            if (!existingRoles.includes(viewerRole)) {
            await this.userRepository.createUserRoleMap(updatedUser.id, viewerRole, manager);
            }
            // Remove all roles except viewer if role is being updated
            const rolesToRemove = updatedUser.userRoleMaps
            .filter((urm) => urm.role.roleKey !== viewerRole)
            .map((urm) => urm.role.roleKey);
            if (rolesToRemove.length > 0) {
            await this.userRepository.removeUserRoles(updatedUser.id, rolesToRemove, manager);
            }

          const newRoleKey = updateUserDto.role ? this.ROLE_NAME_TO_KEY[updateUserDto.role] : undefined;
          if (updateUserDto.role && updateUserDto.role !== 'viewer' && newRoleKey && !existingRoles.includes(newRoleKey)) {
            await this.userRepository.createUserRoleMap(updatedUser.id, newRoleKey, manager);
          }
        });
      }
      this.logger.log(`Service: User updated successfully with ID: ${id}`, { updatedUser });
      return updatedUser;
    } catch (error) {
      if (error instanceof InifniNotFoundException) {
        throw error;
      }
      handleKnownErrors(ERROR_CODES.USER_UPDATE_FAILED, error);
    }
  }
}
