import { Injectable, NotFoundException } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { EntityManager, ILike, IsNull, Repository } from "typeorm";
import { User } from "src/common/entities/user.entity";
import {
  userConstMessages as MESSAGES,
  ROLE_KEYS,
  userConstMessages,
} from "src/common/constants/strings-constants";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { AppLoggerService } from "src/common/services/logger.service";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { CommonDataService } from "src/common/services/commonData.service";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import { UserRole, UserRoleMap } from "src/common/entities";
import { parse } from "path";
import { UpdateUserDto } from "./dto/update-user.dto";

@Injectable()
export class UserRepository {
  constructor(
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly logger: AppLoggerService,
    private readonly commonDataService: CommonDataService,
  ) { }

  /**
   * Fetches a user by a specific field name and value.
   * @param fieldName - The name of the field to filter by.
   * @param value - The value of the field to filter by.
   * @returns The user entity if found.
   * @throws NotFoundException if no user is found.
   */
  async getUserByField(fieldName: string, value: any): Promise<User> {
    this.logger.log(
      userConstMessages.FETCHING_USER_BY_FIELD(fieldName, value),
      { value },
    );
    try {
      const user = await this.userRepo.findOne({
        where: { [fieldName]: value },
        relations: ['userRoleMaps', 'userRoleMaps.role'],
      });
      if (!user) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          value
        );
      }
      this.logger.log(
        userConstMessages.USER_FOUND_BY_FIELD(fieldName, value),
        { user },
      );
      return user;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   * Updates a user by ID with the provided data
   * @param id - User ID to update
   * @param updateData - Partial user data to update
   * @returns Updated user entity
   */
  async updateUser(id: number, updateData: Partial<UpdateUserDto>): Promise<User> {
    this.logger.log(`Updating user with ID: ${id}`, { updateData });
    try {
      // First check if user exists
      const existingUser = await this.userRepo.findOne({ where: { id } });
      if (!existingUser) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      // Filter out undefined values from updateData
      const filteredUpdateData = Object.entries(updateData)
        .filter(([key, value]) => value !== undefined)
        .reduce((acc, [key, value]) => {
          acc[key] = value;
          return acc;
        }, {} as any);

      // Convert date strings to Date objects if present
      if (filteredUpdateData.dob) {
        filteredUpdateData.dob = new Date(filteredUpdateData.dob);
      }
      if (filteredUpdateData.associatedSince) {
        filteredUpdateData.associatedSince = new Date(filteredUpdateData.associatedSince);
      }

      // Update the user
      await this.userRepo.update(id, filteredUpdateData);

      // Fetch and return the updated user with relations
      const updatedUser = await this.userRepo.findOne({
        where: { id },
        relations: ['userRoleMaps', 'userRoleMaps.role'],
      });

      if (!updatedUser) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      this.logger.log(`User updated successfully with ID: ${id}`, { updatedUser });
      return updatedUser;
    } catch (error) {
      if (error instanceof InifniNotFoundException) {
        throw error;
      }
      handleKnownErrors(ERROR_CODES.USER_UPDATE_FAILED, error);
    }
  }

  /**
   * Fetches a user by phone number.
   * @param phoneNumber - The phone number to filter by.
   * @returns The user entity if found.
   */
  async getUserByPhoneNumber(phoneNumber: string): Promise<User> {
    try {
      return await this.getUserByField("phoneNumber", phoneNumber);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   *
   * @param fields - Array of field names to concatenate.
   * @param value  - The concatenated value to search for.
   * @param errorField  - The name of the field for error messages.
   * @returns - The user entity if found.
   */
  async getUserByConcatFields(
    fields: string[],
    value: string,
    errorField: string = "concatenated fields",
  ): Promise<User | null> {
    this.logger.log(
      userConstMessages.FETCHING_USER_BY_CONCATED_FIELDS(fields, value),
    );
    try {
      // Build the concatenation expression dynamically based on fields provided
      const concatExpression = fields
        .map((field) => `user.${field}`)
        .join(", ");

      // Create and execute the query with CONCAT function and enable relations
      const user = await this.userRepo
        .createQueryBuilder("user")
        .leftJoinAndSelect("user.userRoleMaps", "userRoleMaps")
        .leftJoinAndSelect("userRoleMaps.role", "role")
        .where(`CONCAT(${concatExpression}) ILIKE :value`, { value })
        .getOne();

      if (!user) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND_BY_CONCAT_FIELDS,
          null, 
          null,
          value
        );
      }

      this.logger.log(
        userConstMessages.USER_FOUND_BY_FIELD(errorField, value),
        { user },
      );
      return user;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   *
   * @param countryCode - The countryCode to filter by.
   * @param phoneNumber - The phone number to filter by.
   * @returns - The user entity if found.
   */
  async simpleUserLookupByPhone(countryCode: string, phoneNumber: string): Promise<User | null> {
    try {
      return await this.userRepo.findOne({ where: { countryCode, phoneNumber } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   *
   * @param combinedPhone - The combined phone number to filter by.
   * @returns - The user entity if found.
   */
  async getUserByCombinedPhoneNumber(
    combinedPhone: string,
  ): Promise<User | null> {
    try {
      return await this.getUserByConcatFields(
        ["country_code", "phone_number"],
        combinedPhone,
        "combinedPhone",
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

async getUserByEmail(email: string): Promise<User | null> {
  try {
    // Trim and lowercase at query level
    return await this.userRepo
      .createQueryBuilder("user")
      .where("LOWER(TRIM(user.email)) = LOWER(TRIM(:email))", { email })
      .getOne();
  } catch (error) {
    handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
  }
}

  async getUsersByEmail(email: string): Promise<User[]> {
    try {
      // check for case-insensitive match
      return await this.userRepo
        .createQueryBuilder("user")
        .where("LOWER(TRIM(user.email)) = LOWER(TRIM(:email))", { email })
        .getMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  async getUserRolesById(userId: number): Promise<string[]> {
    try {
      const user = await this.userRepo.findOne({
        where: { id: userId },
        relations: ['userRoleMaps', 'userRoleMaps.role'],
      });
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
      }
      const roles = user.userRoleMaps?.map((urm) => urm.role?.roleKey).filter(Boolean) || [ROLE_KEYS.VIEWER];
      return roles;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   *
   * @param fullName - The full name to filter by.
   * @returns - The user entity if found.
   */
  async getUserByFullName(fullName: string): Promise<User | null> {
    try {
      return await this.getUserByConcatFields(
        ["first_name", "last_name"],
        fullName,
        "fullName",
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
   * Fetches a user by Firebase ID.
   * @param firebaseId - The Firebase ID to filter by.
   * @returns The user entity if found.
   */
  async getUserByFirebaseId(firebaseId: string): Promise<User> {
    try {
      return await this.getUserByField("firebaseIdExt", firebaseId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  /**
 * Fetches users by a specific role.
 * @param roleName - The name of the role to filter by.
 * @returns An array of user entities with the specified role.
 */
  async getUsersByRoleKey(roleName: string): Promise<User[]> {
    try {
      this.logger.log(`Fetching users with role: ${roleName}`);

      const users = await this.userRepo.find({
        where: {
          userRoleMaps: {
            role: {
              roleKey: roleName,
            },
          },
        },
        relations: ["userRoleMaps", "userRoleMaps.role"],
      });

      if (!users || users.length === 0) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOT_FOUND_BY_ROLE,
          null,
          null,
          roleName
        );
      }

    this.logger.log(`Found ${users.length} users with role: ${roleName}`, { users });
    return users;
  } catch (error) {
    handleKnownErrors(ERROR_CODES.USER_NOT_FOUND_BY_ROLE, error);
  }

  
}

  async createUser(
    data: Partial<User>,
    transactionalEntityManager: EntityManager,
  ): Promise<User> {
    try {
      const entity = transactionalEntityManager.create(User, data);
      return await transactionalEntityManager.save(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SAVE_FAILED, error);
    }
  }

  async createUserRoleMap(
    userId: number,
    roleKey: string,
    transactionalEntityManager: EntityManager,
  ) {
    try {
      const role = await transactionalEntityManager.findOne(UserRole, {
        where: { roleKey },
      });
      if (!role) {
        throw new NotFoundException(userConstMessages.ROLE_NOT_FOUND(roleKey));
      }

      const data = {
        user: { id: userId },
        role: { id: role.id },
      };
      const entity = transactionalEntityManager.create(UserRoleMap, data);
      return await transactionalEntityManager.save(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SAVE_FAILED, error);
    }
  }

  async removeUserRoles(
    userId: number,
    roleKeys: string[],
    transactionalEntityManager: EntityManager,
  ): Promise<void> {
    if (!roleKeys || roleKeys.length === 0) return;

    // Find role ids for the given roleKeys
    const roles = await transactionalEntityManager.find(UserRole, {
      where: roleKeys.map((roleKey) => ({ roleKey })),
    });
    const roleIds = roles.map((role) => role.id);

    if (roleIds.length === 0) return;

    // Delete user_role_map entries for the user and these roles
    await transactionalEntityManager.delete(
      UserRoleMap,
      {
        user: { id: userId },
        role: { id: roleIds.length === 1 ? roleIds[0] : roleIds },
      }
    );
  }

  /**
   * Helper function to get display name from user
   */
  private getDisplayName(user: User): string {
    if (user.fullName && user.fullName.trim()) {
      return user.fullName.trim();
    }
    return `${user.firstName || ''} ${user.lastName || ''}`.trim();
  }

  /**
   * Helper function to check if user is "Other"
   */
  private isOtherUser(user: User): boolean {
    return (user.fullName && user.fullName.trim() === 'Other') || 
          (user.firstName === 'Other');
  }

  /**
   * User sorting comparator function
   */
  private sortUsers(a: User, b: User): number {
    const aDisplayName = this.getDisplayName(a);
    const bDisplayName = this.getDisplayName(b);
    
    const aIsOther = this.isOtherUser(a) ? 1 : 0;
    const bIsOther = this.isOtherUser(b) ? 1 : 0;
    
    if (aIsOther !== bIsOther) {
      return aIsOther - bIsOther;
    }
    
    return aDisplayName.toLowerCase().localeCompare(bDisplayName.toLowerCase());
  }

  /**
   * Build base where clause for user queries
   */
  private buildBaseWhereClause(parsedFilters: Record<string, any>): any {
    const baseWhereClause: any = {
      isSystemUser: false,
      isAiUser: false
    };

    // Role filtering through relation
    if (parsedFilters?.role) {
      const isRoleId = !isNaN(parsedFilters.role);
      if (isRoleId) {
        baseWhereClause.userRoleMaps = {
          role: { id: parseInt(parsedFilters.role) }
        };
      } else {
        baseWhereClause.userRoleMaps = {
          role: { name: parsedFilters.role }
        };
      }
    } else if (parsedFilters?.roleKey) {
      baseWhereClause.userRoleMaps = {
        role: { roleKey: parsedFilters.roleKey },
      };
    }

    return baseWhereClause;
  }

  /**
   * Build search conditions for user queries
   */
  private buildSearchConditions(baseWhereClause: any, searchText: string): any {
    if (!searchText || !searchText.trim()) {
      return baseWhereClause;
    }

    const searchTerm = searchText.trim();
    return [
      { ...baseWhereClause, fullName: ILike(`%${searchTerm}%`) },
      { ...baseWhereClause, email: ILike(`%${searchTerm}%`) },
      { ...baseWhereClause, phoneNumber: ILike(`%${searchTerm}%`) },
    ];
  }

  /**
   * Apply role filters to query builder
   */
  private applyRoleFilters(qb: any, parsedFilters: Record<string, any>): void {
    if (parsedFilters?.role) {
      const isRoleId = !isNaN(parsedFilters.role);
      if (isRoleId) {
        qb.andWhere('role.id = :roleId', { roleId: parseInt(parsedFilters.role) });
      } else {
        qb.andWhere('role.name = :roleName', { roleName: parsedFilters.role });
      }
    } else if (parsedFilters?.roleKey) {
      qb.andWhere('role.role_key = :roleKey', { roleKey: parsedFilters.roleKey });
    }
  }

  /**
   * Apply search filters to query builder
   */
  private applySearchFilters(qb: any, searchText: string): void {
    if (searchText && searchText.trim()) {
      const searchTerm = `%${searchText.trim()}%`;
      qb.andWhere(
        '(user.full_name ILIKE :searchTerm OR user.email ILIKE :searchTerm OR user.phone_number ILIKE :searchTerm)',
        { searchTerm },
      );
    }
  }

  /**
   * Get users using query builder with role filtering
   */
  private async getUsersWithRoleFiltering(
    parsedFilters: Record<string, any>,
    searchText: string,
    limit: number,
    offset: number
  ): Promise<User[]> {
    const qb = this.userRepo
      .createQueryBuilder('user')
      .leftJoinAndSelect('user.userRoleMaps', 'userRoleMaps')
      .leftJoinAndSelect('userRoleMaps.role', 'role')
      .where('user.is_system_user = false')
      .andWhere('user.is_ai_user = false');

    this.applyRoleFilters(qb, parsedFilters);
    this.applySearchFilters(qb, searchText);

    const allResults = await qb.getMany();
    allResults.sort((a, b) => this.sortUsers(a, b));

    return allResults.slice(offset, offset + limit);
  }

  /**
   * Get users using common data service
   */
  private async getUsersWithCommonDataService(
    finalWhereClause: any,
    limit: number,
    offset: number
  ): Promise<User[]> {
    const data = await this.commonDataService.get(
      this.userRepo,
      undefined,
      finalWhereClause,
      limit,
      offset,
      { id: 'ASC' },
      undefined,
      ['userRoleMaps', 'userRoleMaps.role'],
    );

    data.sort((a, b) => this.sortUsers(a, b));
    return data;
  }

  /**
   * Get total count for users with role filtering
   */
  private async getTotalCountWithRoleFiltering(parsedFilters: Record<string, any>): Promise<number> {
    const countQb = this.userRepo
      .createQueryBuilder('user')
      .leftJoin('user.userRoleMaps', 'userRoleMaps')
      .leftJoin('userRoleMaps.role', 'role')
      .where('user.is_system_user = false')
      .andWhere('user.is_ai_user = false');

    this.applyRoleFilters(countQb, parsedFilters);

    return await countQb.getCount();
  }

  /**
   * Build pagination response
   */
  private buildPaginationResponse(
    data: User[],
    total: number,
    limit: number,
    offset: number
  ): any {
    return {
      data,
      pagination: {
        totalPages: Math.ceil(total / limit),
        pageNumber: Math.floor(offset / limit) + 1,
        pageSize: +limit,
        totalRecords: total,
        numberOfRecords: data.length,
      },
    };
  }

  /**
   * Fetches users with filtering, searching, and pagination
   * FIXED: Modularized with proper sorting and role filtering
   */
  async getAllUsers(
    limit: number,
    offset: number,
    searchText: string,
    parsedFilters: Record<string, any>,
  ) {
    try {
      const baseWhereClause = this.buildBaseWhereClause(parsedFilters);
      const finalWhereClause = this.buildSearchConditions(baseWhereClause, searchText);
      const hasRoleFiltering = !!(parsedFilters?.role || parsedFilters?.roleKey);

      let data: User[];
      let total: number;

      if (hasRoleFiltering) {
        // Use query builder for role filtering with custom sorting
        data = await this.getUsersWithRoleFiltering(parsedFilters, searchText, limit, offset);
        total = await this.getTotalCountWithRoleFiltering(parsedFilters);
      } else {
        // Use common data service for non-role filtering
        data = await this.getUsersWithCommonDataService(finalWhereClause, limit, offset);
        total = await this.userRepo.count({ where: baseWhereClause });
      }

      return this.buildPaginationResponse(data, total, limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }
}