import { 
  Controller, 
  Get,
  Post,
  Param,
  ParseIntPipe,
  Res,
  HttpStatus,
  UseGuards,
  Query,
  Req,
  Patch,
  ValidationPipe,
  Body,
  HttpCode,
} from "@nestjs/common";
import { Response } from "express";
import { UserService } from "./user.service";
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBearerAuth, ApiSecurity, ApiQuery, ApiBody } from "@nestjs/swagger";
import { ResponseService } from "src/common/response-handling/response-handler";
import { AppLoggerService } from "src/common/services/logger.service";
import { handleControllerError } from "src/common/utils/controller-response-handling";
import { FirebaseAuthGuard } from "src/auth/firebase-auth.guard";
import { RolesGuard } from "src/common/guards/roles.guard";
import { Roles } from "src/common/decorators/roles.decorator";
import { userConstMessages, SWAGGER_API_RESPONSE, ROLE_KEYS } from "src/common/constants/strings-constants";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { RegistrationStatusEnum } from "src/common/enum/registration-status.enum";
import { ApprovalStatusEnum } from "src/common/enum/approval-status.enum";
import { CombinedAuthGuard } from "src/auth/combined-auth.guard";
import { UpdateUserDto } from "./dto/update-user.dto";
import { CreateUserDto } from "./dto/create-user.dto";


@ApiTags("user")
@Controller("user")
@UseGuards(CombinedAuthGuard,RolesGuard)
@Roles('admin', 'viewer', 'finance_manager', 'relational_manager', 'shoba', 'rm', 'operational_manger','mahatria')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class UserController {
  constructor(
    private readonly userService: UserService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService
  ) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: 'Create user' })
  @ApiBody({ type: CreateUserDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @UseGuards(FirebaseAuthGuard)
  async createUser(
    @Body(ValidationPipe) createUserDto: CreateUserDto,
    @Res() res: Response,
  ) {
    this.logger.log('Creating user', createUserDto);
    try {
      const data = await this.userService.createUser(createUserDto);
      if (!data) {
        this.logger.error(userConstMessages.USER_CREATION_FAILED);
        throw new InifniNotFoundException(
          ERROR_CODES.USER_CREATION_FAILED,
          null,
          null,
          createUserDto.phoneNumber
        );
      }
      return this.responseService.success(res, 'User created successfully', data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }


  /**
   * Get all users with search, pagination, and filters.
   * @param limit - Number of users per page
   * @param offset - Offset for pagination
   * @param searchText - Search text for user fields
   * @param filters - JSON string for additional filters (e.g., role)
   * @returns List of users
   */
  @Get()
  @ApiOperation({ summary: "Get all users with search, pagination, and filters" })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of users per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters (e.g., role)' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getAllUsers(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Res() res: Response,
  ) {
    this.logger.log(userConstMessages.FETCHING_ALL_USERS({
      limit,
      offset,
      searchText,
      filters: filters ? JSON.parse(decodeURIComponent(filters)) : {}
    }));
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const users = await this.userService.getAllUsers(+limit, +offset, searchText, parsedFilters);
      return this.responseService.success(res, "Users fetched successfully", users, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get all relational managers with search and pagination
   * @param limit - Number of users per page
   * @param offset - Offset for pagination
   * @param searchText - Search text for user fields
   * @returns List of relational managers
   */
  @Get('rm-list')
  @ApiOperation({ summary: 'Get relational managers with search and pagination' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of users per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getRMList(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText = '',
    @Res() res: Response,
  ) {
    const filters = { roleKey: ROLE_KEYS.RELATIONAL_MANAGER };
    this.logger.log(userConstMessages.FETCHING_RM_LIST({ limit, offset, searchText }));
    try {
      const users = await this.userService.getAllUsers(+limit, +offset, searchText, filters);
      return this.responseService.success(res, userConstMessages.RM_LIST_RETRIEVED, users, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get user by ID.
   * @param id - User ID
   * @returns User data
   */
  @Get(":id")
  @ApiOperation({ summary: "Get user by ID" })
  @ApiParam({ name: "id", type: Number, description: "User ID" })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getUserById(
    @Param("id", ParseIntPipe) id: number,
    @Res() res: Response
  ) {
    this.logger.log(userConstMessages.FETCHING_USER_BY_FIELD("id", id));
    try {
      const user = await this.userService.getUserById(id);
      if (!user) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      this.logger.log(userConstMessages.USER_FOUND_BY_FIELD("id", id), { user });
      return this.responseService.success(res, "User fetched successfully", user, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get user by phone number.
   * @param phoneNumber - User phone number
   * @returns User data
   */
  @Get("phone-number/:phoneNumber")
  @ApiOperation({ summary: "Get user by phone number" })
  @ApiParam({ name: "phoneNumber", type: String, description: "User phone number" })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getUserByPhoneNumber(
    @Param("phoneNumber") phoneNumber: string,
    @Res() res: Response
  ) {
    this.logger.log(userConstMessages.FETCHING_USER_BY_FIELD("phoneNumber", phoneNumber));
    try {
      const user = await this.userService.getUserByCombinedPhoneNumber(phoneNumber);
      if (!user) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_BY_FIELD("phoneNumber", phoneNumber));
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          phoneNumber
        );
      }
      this.logger.log(userConstMessages.USER_FOUND_BY_FIELD("phoneNumber", phoneNumber), { user });
      return this.responseService.success(res, "User fetched successfully", user, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
  /**
   * Get user by ID for prefill.
   * @param id - User ID
   * @returns User data for prefill
   */
  @Get(":id/prefill")
  @ApiOperation({ summary: "Get user by ID for prefill" })
  @ApiParam({ name: "id", type: Number, description: "User ID" })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getUserByIdPrefill(
    @Param("id", ParseIntPipe) id: number,
    @Res() res: Response
  ) {
    this.logger.log(userConstMessages.FETCHING_USER_BY_FIELD("id", id));
    try {
      const user = await this.userService.getUserPrefilDataById(id);
      if (!user) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      // You can filter/transform user fields here if needed for prefill
      this.logger.log(userConstMessages.USER_FOUND_BY_FIELD("id", id), { user });
      return this.responseService.success(res, "User fetched successfully", user, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(":id/registrations")
  @ApiOperation({ summary: "Get user registrations" })
  @ApiParam({ name: "id", type: Number, description: "User ID" })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Program ID to filter registrations' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Program Session ID to filter registrations' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getUserRegistrations(
    @Param('id', ParseIntPipe) id: number,
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText = '',
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('filters') filters = '',
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const data = await this.userService.getRegistrationsByUser(id, +limit, +offset, searchText, programId, programSessionId, parsedFilters); 
      
     if (data) {
        // for each registration make registration status as pending_approval if it is in pendinG state and approovals status is pending
        data.data.forEach(registration => {
          if (registration.registrationStatus === RegistrationStatusEnum.PENDING && registration.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
            registration.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
          }
        }
      );
    }
      await this.responseService.success(res, 'Registrations fetched successfully', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Update user by ID.
   * @param id - User ID
   * @param updateUserDto - User data to update
   * @returns Updated user data
   */
  @Patch(":id")
  @ApiOperation({ summary: "Update user by ID" })
  @ApiParam({ name: "id", type: Number, description: "User ID" })
  @ApiBody({ type: UpdateUserDto, description: "User data to update" })
  @ApiResponse({ status: HttpStatus.OK, description: "User updated successfully" })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: "Bad Request - Invalid data" })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async updateUser(
    @Param("id", ParseIntPipe) id: number,
    @Body(ValidationPipe) updateUserDto: UpdateUserDto,
    @Res() res: Response
  ) {
    this.logger.log(`Updating user with ID: ${id}`, { updateData: updateUserDto });
    try {
      const updatedUser = await this.userService.updateUser(id, updateUserDto);
      this.logger.log(`User updated successfully with ID: ${id}`, { updatedUser });
      return this.responseService.success(res, "User updated successfully");
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
