import { Injectable } from '@nestjs/common';
import { CreateRegistrationDto } from './dto/create-registration.dto';
import { UpdateRegistrationDto } from './dto/update-registration.dto';
import { RegistrationRepository } from './registration.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationMessages, stateGstCodes, zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { FormSection, Program, ProgramQuestion, ProgramSession, Question, RegistrationCustomResponse, User, ProgramRegistration } from 'src/common/entities';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { RegistrationBasicStatusEnum } from 'src/common/enum/registration-basic-status.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { CommunicationService } from 'src/communication/communication.service';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { TravelStatusEnum } from 'src/common/enum/travel-status.enum';
import { GenderEnum } from 'src/common/enum/gender.enum';
import { In, Not, IsNull } from 'typeorm';
import { TravelTypeEnum } from 'src/common/enum/travel-type.enum';
import { RegistrationAnswerDto } from './dto/create-registration.dto';
import { MajorCities, SEEKER_FE_EDIT_REGISTRATION_PATH, SEEKER_FE_REG_PATH } from 'src/common/constants/constants';
import { ExcelService } from 'src/common/services/excel.service';
import { areGSTsFromSameState } from 'src/common/utils/common.util';
import { ProgramRegistrationSwapRepository } from 'src/program-registration/repositories/program-registration-swap.repository';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { ProgramRegistrationService } from 'src/program-registration/program-registration.service';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';


// Define comprehensive return type interface for registration operations
interface RegistrationResult {
  registrationId: number;
  registrationLevel?: string;
  registrationStatus?: RegistrationStatusEnum;
  basicDetailsStatus?: RegistrationBasicStatusEnum;
  
  // Waitlist related properties
  waitlisted?: boolean;
  waitlistPosition?: number;
  participantRemainingSeats?: number;
  
  // Approval related properties
  pendingApproval?: boolean;
  approvalType?: ApprovalStatusEnum;
  
  // Direct registration properties
  registrationSeqNumber?: string;
  remainingSeats?: number;
  
  // Update related properties
  formSection?: string;
  formSectionId?: number;
  updated?: boolean;
  
  // Cancellation related properties
  cancelled?: boolean;
}

// KPI interfaces for Default List View
interface DefaultKPIResult {
  registrations: {
    categoryLabel: string;
    items: KPIItem[];
  };
  programs?: {
    categoryLabel: string;
    items: KPIItem[];
  };
  payments?: {
    categoryLabel: string;
    items: KPIItem[];
  };
  invoices?: {
    categoryLabel: string;
    items: KPIItem[];
  };
  travelAndLogistics?: {
    categoryLabel: string;
    items: KPIItem[];
  };
}

interface KPIItem {
  label: string;
  value: number;
  kpiCategory: string;
  kpiFilter: string;
  order : number
}

// KPI interfaces for Mahatria
interface AllocatedProgramMetrics {
  programs: {
    programId: number;
    programName: string;
    allocatedCount: number;
    maleCount: number;
    femaleCount: number;
    organisationUserCount: number;
    totalSeatsCount: number;
    maleOrganisationUserCount: number;
    femaleOrganisationUserCount: number;
    roomMatePreferenceCount: number;
  }[];
  holdCount: number;
  rejectCount: number;
  holdMaleCount: number;
  holdFemaleCount: number;
  holdOrganisationUserCount: number;
  rejectMaleCount: number;
  rejectFemaleCount: number;
  rejectOrganisationUserCount: number;
  holdMaleOrganisationUserCount: number;
  holdFemaleOrganisationUserCount: number;
  rejectMaleOrganisationUserCount: number;
  rejectFemaleOrganisationUserCount: number;
}

interface UnallocatedProgramMetrics {
  programs: {
    programId: number;
    programName: string;
    count: number;
    maleCount: number;
    femaleCount: number;
    organisationUserCount: number;
    maleOrganisationUserCount: number;
    femaleOrganisationUserCount: number;
    totalSeatsCount: number;
  }[];
  totalMaleCount: number;
  totalFemaleCount: number;
  totalUnallocatedCount: number;
  totalMahatriaChoiceCount: number;
  totalOrganisationUserCount: number;
  totalMaleOrganisationUserCount: number;
  totalFemaleOrganisationUserCount: number;
  totalMahatriaChoiceOrganisationUserCount: number;
}

interface AllRegistrationMetrics{
  totalRegistrations: number;
  mahatriaChoice: number;
  preferredPrograms: {
    id:string;
    name:string;
    count: number;
  }[];
  maleCount: number;
  femaleCount: number;
}

interface MahatriaKPIResult {
  isGroupedProgram: boolean;
  primaryProgramId?: number;
  primaryProgramName?: string;
  allocated?: AllocatedProgramMetrics;
  unallocated?: UnallocatedProgramMetrics;
  totalRegistrations?: number;
}

@Injectable()
export class RegistrationService {
  constructor(
    private readonly repository: RegistrationRepository,
    private readonly logger: AppLoggerService,
    private readonly communicationService: CommunicationService,
    private readonly excelService: ExcelService,
    private readonly programRegistrationSwapRepository: ProgramRegistrationSwapRepository,
    private readonly programRegistrationService: ProgramRegistrationService
    
  ) {}

  async register(
    dto: CreateRegistrationDto,
    userId: number,
    isSelfRegister: boolean,
  ): Promise<RegistrationResult> {
    this.logger.log(registrationMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      // STEP 1: EARLY VALIDATION - Validate and get program details first
      const program = await this.validateAndGetProgram(dto.programId);
      const registrationLevel = this.getRegistrationLevel(program);

      // STEP 2: EARLY VALIDATION - Validate session if required
      const session = await this.validateSessionIfRequired(dto, program, registrationLevel);

      // STEP 3: EARLY VALIDATION - Validate question answers FIRST
      // This will return immediately if there are any question validation errors
      const questionMappings = await this.validateQuestionAnswers(
        dto,
        program.id,
        dto.programSessionId,
      );

      // STEP 4: CHECK REGISTRATION DATES - Check if registrations are still open
      await this.validateRegistrationDates(program, session, registrationLevel);

      // STEP 5: Business Logic - Check if user is already registered (self-registration only)
      if (isSelfRegister) {
        await this.validateUserNotAlreadyRegistered(userId, dto.programId, dto.programSessionId);
      }

      // STEP 6: Business Logic - Check seat availability and determine registration path
      const seatInfo = await this.checkSeatAvailability(program, session, registrationLevel);

      // STEP 7: Business Logic - Process answers and group by storage location
      const processedAnswers = this.processQuestionAnswers(dto.answers, questionMappings, userId);

      // STEP 8: Business Logic - Determine registration outcome based on seats and requirements
      const registrationOutcome = this.determineRegistrationOutcome(seatInfo, program);

      // STEP 9: Delegate to repository for data persistence
      const registration = await this.repository.createRegistration({
        dto,
        userId,
        isSelfRegister,
        program,
        session,
        registrationLevel,
        seatInfo,
        processedAnswers,
        registrationOutcome,
        registrationStatus: dto.registrationStatus,
      });


      const programRegistration = await this.repository.findRegistrationById(
        registration.registrationId,
      );

      if (registration.basicDetailsStatus === RegistrationBasicStatusEnum.COMPLETED) {
        const mergeInfo = {
          reg_name: programRegistration.fullName,
          reg_edit_link: this.registrationEditLink(programRegistration.program.id, userId),
        };
        const emailData: SendSingleEmailDto = {
          templateKey: process.env.ZEPTO_REGISTRATION_COMPLETED_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name: program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: programRegistration.emailAddress,
            name: programRegistration.fullName,
          },
          mergeInfo: mergeInfo,
          attachments: [],
          subject: '',
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        await this.communicationService.sendSingleEmail(emailData);
        const messageData: SendTemplateMessageDto = {
          whatsappNumber: programRegistration.mobileNumber.slice(0), // Remove + prefix
          templateName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: programRegistration.fullName }],
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        await this.communicationService.sendTemplateMessage(messageData);

        const rmMessageData: SendTemplateMessageDto = {
          whatsappNumber: `${programRegistration.rmContactUser.countryCode}${programRegistration.rmContactUser.phoneNumber}`.slice(0), // Remove + prefix
          templateName: process.env.WATI_RM_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_RM_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: programRegistration.fullName }],
          trackinfo: {
            registrationId: programRegistration.id,
          },
        };
        await this.communicationService.sendTemplateMessage(rmMessageData);
      }

      return registration;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CREATION_FAILED, error);
    }
  }

  async update(dto: UpdateRegistrationDto, user: User): Promise<RegistrationResult> {
    this.logger.log(registrationMessages.UPDATE_REQUEST_RECEIVED, dto);
    try {

      // Business Logic: Validate form section
      let formSection: FormSection | null = null;
      if (dto.formSectionId) {
        formSection = await this.validateFormSection(dto.formSectionId);
      }
      let updateDto:any = { ...dto };
      if(formSection){
        updateDto = {
          ...dto,
          formSection: Object.assign({}, formSection),
        }
      }
      // Business Logic: Validate registration exists and can be updated
      const registration = await this.validateRegistrationForUpdate(
        dto.programRegistrationId,
        user,
        updateDto
      );

      // Business Logic: Handle cancellation request
      if (this.isCancellationRequest(dto)) {
        if (!dto.cancellationDate || !dto.cancelledBy) {
          throw new InifniBadRequestException(
            ERROR_CODES.REGISTRATION_INVALID_STATUS,
            null,
            null,
            'Cancellation date and cancelled by are required for cancellation',
          );
        }
        return await this.repository.cancelRegistration(
          registration.id,
          dto.cancellationDate,
          dto.cancelledBy,
        );
      }

      // if role of user is viewer then form section id should be provided
      if (user.userRoleMaps.some((urm) => urm.role.name === 'viewer') && !dto.formSectionId) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_INVALID_STATUS,
          null,
          null,
          'Form section ID is required for viewer role',
        );
      }


      // Business Logic: Check approval status
      if (
        formSection &&
        formSection.key !== 'FS_BASICDETAILS' &&
        formSection.key !== 'FS_MAHATRIAQUESTIONS'
      ) {
        await this.validateApprovalStatus(registration);
      }

      // Business Logic: Validate program and session
      const program = await this.validateProgramForUpdate(dto.programId, registration);
      const session = await this.validateSessionForUpdate(dto, registration, program);

      if (
        formSection &&
        formSection.key === 'FS_TRAVELPLAN')
      {
        // #codebase if registration_status is save_As_draft then in upserting question check for the section of the question if its section_key is FS_travel.... then check if record is there if there mark that status as save_as_draft and if not there then add a new record there with save as draft directly insert first
      const existingTravelPlan = await this.repository.findTravelPlanByRegistrationId(
        registration.id,
      );
      if (existingTravelPlan) {
        // Update existing travel plan with draft status
        await this.repository.updateTravelPlanStatus(
          existingTravelPlan.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id,
        );
      } else {
        // Create new travel plan with draft status
        await this.repository.createTravelPlan(
          registration.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id);
      }

      const existingTravelInfo = await this.repository.findTravelInfoByRegistrationId(
        registration.id,
      );
      if (existingTravelInfo) {
        // Update existing travel info with draft status
        await this.repository.updateTravelInfoStatus(
          existingTravelInfo.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id,
        );
      } else {
        // Create new travel info with draft status
        await this.repository.createTravelInfo(
          registration.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id);
      }

      }

      // Business Logic: Validate and process section-specific questions
      const sectionQuestions = await this.validateSectionQuestions(
        dto,
        program.id,
        dto.programSessionId,
        formSection,
      );
      const processedAnswers = this.processQuestionAnswers(dto.answers, sectionQuestions, user.id);

      // Delegate to repository for data persistence
      const updateRegistrationData= await this.repository.updateRegistrationData({
        registration,
        processedAnswers,
        userId: user.id,
        registrationStatus: dto.registrationStatus,
      });

      const updatedRegistration = await this.repository.findRegistrationById(
        updateRegistrationData.registrationId,
      );

      if (registration.basicDetailsStatus === RegistrationBasicStatusEnum.DRAFT && updatedRegistration.basicDetailsStatus === RegistrationBasicStatusEnum.COMPLETED) {
        const mergeInfo = {
          reg_name: updatedRegistration.fullName,
          reg_edit_link: this.registrationEditLink(updatedRegistration.program.id, user.id),
        };
        const emailData: SendSingleEmailDto = {
          templateKey: process.env.ZEPTO_REGISTRATION_COMPLETED_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name: program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: updatedRegistration.emailAddress,
            name: updatedRegistration.fullName,
          },
          mergeInfo: mergeInfo,
          attachments: [],
          subject: '',
          trackinfo: {
            registrationId: updatedRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        await this.communicationService.sendSingleEmail(emailData);
        const messageData: SendTemplateMessageDto = {
          whatsappNumber: updatedRegistration.mobileNumber.slice(0), // Remove + prefix
          templateName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: updatedRegistration.fullName }],
          trackinfo: {
            registrationId: updatedRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        await this.communicationService.sendTemplateMessage(messageData);
      }
      return updateRegistrationData;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CREATION_FAILED, error);
    }
  }

  /**
   * Get registrations list view with KPIs based on user roles
   */
  async getRegistrationsListView(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
    downloadType?: string,
  ) {
    this.logger.log('Getting registrations list view', {
      limit,
      offset,
      programId,
      programSessionId,
      searchText,
      parsedFilters,
      userRoles,
      downloadType,
    });
  
    try {
      const isRMUser = userRoles.includes('relational_manager');
  
      // Handle download case - return all data without pagination
      if (downloadType) {
        const registrationData = await this.repository.findRegistrations(
          'all',
          'all',
          programId,
          programSessionId,
          downloadType === 'filtered' ? searchText : '',
          downloadType === 'filtered' ? this.extractDataFilters(parsedFilters) : {},
          userRoles,
          isRMUser ? filterRegistrationsByUserId : undefined,
        );
        
        return {
          data: registrationData.data,
          pagination: {
            totalPages: 1,
            pageNumber: 1,
            pageSize: registrationData.data.length,
            totalRecords: registrationData.data.length,
            numberOfRecords: registrationData.data.length,
          },
        };
      }
  
      // ✅ FIXED: Separate filters for KPIs and data
      const kpiFilters = this.extractKPIFilters(parsedFilters);
      const dataFilters = this.extractDataFilters(parsedFilters);

      // Get registration data with BOTH search + KPI filters applied
      const result = await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText, // ✅ Search applies to data table
        dataFilters,
        userRoles,
        isRMUser ? filterRegistrationsByUserId : undefined,
      );
  
      // ✅ CRITICAL FIX: Generate KPIs WITHOUT search text - always show total state
      const kpis = await this.generateDefaultKPIs({
        programId,
        programSessionId,
        searchText: '', // ✅ NO SEARCH TEXT for KPIs
        parsedFilters: this.removeKPIFilters(parsedFilters), // ✅ Remove KPI filters from KPI generation
        userRoles,
        filterRegistrationsByUserId: isRMUser ? filterRegistrationsByUserId : undefined,
        activeKpiCategory: parsedFilters?.kpiCategory || null,
      });
  
      return {
        data: result.data,
        pagination: result.pagination,
        kpis: {
          default: kpis,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Transform registration list data for the controller response
   */
  async transformRegistrationListData(result: any, userRoles: string[]) {
    const transformedData = {
      data: result.data.map(registration => {
        let paymentStatus = registration?.paymentDetails?.[0]?.paymentStatus;
        // if (!paymentStatus && registration?.allocatedProgram) {
        //   paymentStatus = 'payment_pending';
        // }

        let travelStatus = registration?.travelPlans?.[0]?.travelPlanStatus;
        // if (!travelStatus && registration?.allocatedProgram) {
        //   travelStatus = 'travel_pending';
        // }

        return {
          id: registration.id,
          seekerName: this.capitalizeWords(registration.fullName) ?? '',
          age: this.calculateAge(registration.dob),
          gender: registration?.gender ?? '',
          location: registration?.city ?? '',
          blessedWith: registration?.allocatedProgram?.name ?? '',
          approvalStatus: registration?.approvals?.[0]?.approvalStatus ?? '',
          numberOfHDBs: registration?.noOfHDBs ?? 0,
          rmComments: registration?.rmReview || '',
          paymentStatus: this.formatPaymentStatus(paymentStatus ?? ''),
          invoiceStatus: this.formatInvoiceStatus(registration?.invoiceDetails?.[0]?.invoiceStatus ?? ''),
          invoiceUrl: registration?.invoiceDetails?.[0]?.invoicePdfUrl ?? '',
          travelPlanStatus: this.formatTravelStatus(travelStatus ?? ''),
          mobileNumber: registration?.mobileNumber ?? '',
          registrationStatus: registration?.registrationStatus,
          averageRating: registration?.averageRating || 0,
          ratings: registration?.ratings || [],
          profileUrl: registration?.profileUrl || '',
          emailAddress: registration?.emailAddress,
          programId: registration?.program?.id,
          programSessionId: registration?.programSession?.id,
          preference: registration?.preferences ?? [],
          appliedOn: registration?.createdAt,
          isSwapRequestActive: registration?.swapsRequests?.some(swap => swap.status === SwapRequestStatus.ACTIVE) || false,
          isOrgUser: registration?.user?.userType === UserTypeEnum.ORG ? true : false,
          proFormaInvoicePdfUrl: registration?.proFormaInvoicePdfUrl || '',
          preferredRoomMate: registration?.preferredRoomMate || ''
        };
      }),
      tableHeaders: this.getTableHeaders(userRoles),
      filters: this.getFilterConfiguration(userRoles),
      pagination: result.pagination,
      kpis: this.flattenKPIs(result.kpis?.default ?? {})
    };
    
    // Apply registration status transformation
    const isUserRoleRMorOM = userRoles.includes('finance_manager') || userRoles.includes('operations_manager');
    if (transformedData.data) {
      transformedData.data.forEach((registration, index) => {
        if (isUserRoleRMorOM){
          delete registration.rmComments;
          delete registration.ratings;
        }
        const originalReg = result.data[index];
        if (originalReg.registrationStatus === RegistrationStatusEnum.PENDING && 
            originalReg.approvals && 
            originalReg.approvals[0] && 
            originalReg.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
          registration.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
        }
      });
    }

    return transformedData;
  }

  // transformDraftRegistrationListData
  async transformDraftRegistrationListData(result: any, userRoles: string[]) {
    const data = {
      data: result.data.map(registration => ({
        id: registration.id,
        seekerName: this.capitalizeWords(registration.fullName) ?? '',
        age: this.calculateAge(registration.dob),
        gender: registration.gender ?? '',
        location: registration.city ?? '',
        numberOfHDBs: registration.noOfHDBs ?? 0,
        mobileNumber: registration.mobileNumber ?? '',
        emailAddress: registration.emailAddress ?? '',
        profileUrl: registration.profileUrl || '',
        rmContact: registration.rmContactUser ? registration.rmContactUser.fullName : '',
        programId: registration.program?.id,
        programSessionId: registration.programSession?.id,
        preferences: registration.preferences ?? [],
      }))
    };
    const transformedData = {
      data: data.data,
      tableHeaders: this.getDraftRegTableHeaders(userRoles),
      pagination: result.pagination,
    };
    return transformedData;
  }

  // transformMahatriaKPIData
  async transformMahatriaKPIData(result: any, userRoles: string[]) {
    const transformedData = result.data.map((item: any) => ({
      ...item,
      isOrganizationUser: item.user?.userType === UserTypeEnum.ORG ? true : false,
    }));
    return transformedData;
  }

/**
 * Generate Excel file for download and upload to S3 with new format
 * @param registrations - Raw registration data from repository
 * @param userRoles - User roles for determining visible columns
 * @param downloadType - Type of download (all/filtered)
 * @returns S3 URL of the generated Excel file
 */
async generateExcelDownload(
  registrations: any[], 
  userRoles: string[], 
  downloadType: string
): Promise<string> {
  try {    
    // Transform data for Excel according to the new specified format
    const excelData = registrations.map((registration, index) => {
      let igstCheck = (registration?.invoiceDetails?.[0]?.gstNumber && registration?.program?.gstNumber) ? !areGSTsFromSameState(registration.program?.gstNumber, registration?.invoiceDetails?.[0]?.gstNumbe) : false;
      let transformedReg: any = {
        'S.No.': index + 1,
        'Seeker Email': registration?.emailAddress ?? '',
        'Seeker Name': registration?.fullName ?? '',
        'Blessed': registration?.allocatedProgram?.name ?? '',
        'No of HDB / MSD done': registration?.noOfHDBs ?? 0,
        'Mobile #': registration?.mobileNumber ?? '',
        'RM': registration?.rmContactUser?.fullName ?? '',
        // 'Country': registration?.country ?? 'India', 
        'City': registration?.city ?? '',
        'Date of blessing': registration?.allocatedProgram ? (registration?.approvals[0]?.approvalDate ? new Date(registration?.approvals[0]?.approvalDate).toLocaleDateString('en-GB') : '') : '',
        // 'Payment link completion date': registration?.paymentDetails?.[0]?.paymentStatus === 'online_completed' ? (registration?.paymentDetails?.[0]?.updatedAt ? new Date(registration.paymentDetails[0].updatedAt).toLocaleDateString('en-GB') : '') : '',
        'Programme Name with period': this.formatProgrammeNameWithPeriod(registration?.program, registration?.allocatedProgram) ?? ''
      };

      if(userRoles.includes('finance_manager')) {
        transformedReg = {
          ...transformedReg,
          'Mode of payment': this.formatPaymentMode(registration?.paymentDetails?.[0]?.paymentMode,registration?.paymentDetails[0]?.offlineMeta) ?? '',
          'Cheque/DD #': registration?.paymentDetails?.[0]?.offlineMeta?.chequeNo ?? '',
          'Cheque/DD Date': registration?.paymentDetails?.[0]?.offlineMeta?.chequeDate ? new Date(registration.paymentDetails[0].offlineMeta.chequeDate).toLocaleDateString('en-GB') : '',
          'UTR/Reference #': registration?.paymentDetails?.[0]?.offlineMeta?.utrNumber ?? registration?.paymentDetails?.[0]?.offlineMeta?.referenceNumber ?? '',
          'NEFT / Bank Transfer Date': registration?.paymentDetails?.[0]?.offlineMeta?.bankTransferDate ? new Date(registration.paymentDetails[0].offlineMeta.bankTransferDate).toLocaleDateString('en-GB') : '',
          [`CGST ${registration?.program?.cgst}%`]: igstCheck ? '' : this.calculateTaxAmount((registration?.paymentDetails?.[0]?.originalAmount ? Number(registration?.paymentDetails?.[0]?.originalAmount) : 0), (registration?.program?.cgst ?? 9)),
          [`SGST ${registration?.program?.sgst}%`]: igstCheck ? '' : this.calculateTaxAmount((registration?.paymentDetails?.[0]?.originalAmount ? Number(registration?.paymentDetails?.[0]?.originalAmount) : 0), (registration?.program?.sgst ?? 9)),
          [`IGST ${registration?.program?.igst}%`]: igstCheck ? this.calculateTaxAmount(registration?.paymentDetails?.[0]?.originalAmount ? Number(registration?.paymentDetails?.[0]?.originalAmount) : 0, (registration?.program?.igst ?? 18)) : '',
          'Total amount': registration?.paymentDetails?.[0]?.subTotal ?? registration?.paymentDetails?.[0]?.subTotal ?? '',
          'TDS': registration?.paymentDetails?.[0]?.tds ?? '',
          'Amount Received': registration?.paymentDetails?.[0]?.subTotal ? registration?.paymentDetails?.[0]?.subTotal + registration?.paymentDetails?.[0]?.tds : '',
          'Fee Amount': registration?.invoiceDetails?.[0]?.baseAmount ?? registration?.paymentDetails?.[0]?.originalAmount ?? '',
          'GSTIN': registration?.invoiceDetails?.[0]?.gstNumber ?? '',
          'TAN': registration?.invoiceDetails?.[0]?.tanNumber ?? '',
          'Payment ID (Razor pay)': registration?.paymentDetails?.[0]?.offlineMeta?.razropayPaymentId ?? '',
          'Order ID (Razor pay)': registration?.paymentDetails?.[0]?.razorpayId ?? '',
          'Invoice Type': registration?.invoiceDetails?.[0] ? (igstCheck ? 'IGST' : 'CGST AND SGST') : '',
          'Invoice Name': registration?.invoiceDetails?.[0]?.invoiceName ?? '',
          'Invoice Email': registration?.invoiceDetails?.[0]?.invoiceEmail ?? '',
          'Invoice Address': registration?.invoiceDetails?.[0]?.invoiceAddress ?? '',
          'Pincode': registration?.invoiceDetails?.[0]?.zip ?? '',
          'Place of Supply':  registration?.invoiceDetails?.gstNumber ? stateGstCodes.find(code => code.code === registration.invoiceDetails?.gstNumber.substring(0, 2))?.stateName || null : null,
          'Invoice #': registration?.invoiceDetails?.[0]?.invoiceSequenceNumber ?? '',
          // 'Invoice Date': registration?.invoiceDetails?.[0]?.invoiceIssuedDate ? new Date(registration.invoiceDetails[0].invoiceIssuedDate).toLocaleDateString('en-GB') : '',
          'e-Invoice Ack #': registration?.invoiceDetails?.[0]?.einvoiceAckNumber ?? '',
          'e-Invoice Date': registration?.invoiceDetails?.[0]?.einvoiceAckDate ? new Date(registration.invoiceDetails[0].einvoiceAckDate).toLocaleDateString('en-GB') : '',
          'Invoice Order Type': registration?.paymentDetails?.[0]?.paymentMode === 'online' ? 'Online' : 'Offline',
        }
      } else {
        transformedReg = {
          ...transformedReg,
          'Payment Status': registration?.paymentDetails?.[0]?.paymentStatus ? registration?.paymentDetails?.[0]?.paymentStatus : '',
          'Invoice Status': registration?.invoiceDetails?.[0]?.invoiceStatus ? registration?.invoiceDetails?.[0]?.invoiceStatus : '',
          'Invoice Created Date': registration?.invoiceDetails?.[0]?.createdAt ? new Date(registration.invoiceDetails[0].createdAt).toLocaleDateString('en-GB') : '',
          'Travel Status': registration?.travelPlans?.[0]?.travelPlanStatus ? registration?.travelPlans?.[0]?.travelPlanStatus : '',
          'Type of ID': registration.travelInfo?.[0]?.idType ?? '',
          'ID Number': registration.travelInfo?.[0]?.idNumber ?? '',
          'T-Shirt Size': registration.travelInfo?.[0]?.tshirtSize ?? '',
          'Arrival Airline Name': registration.travelPlans?.[0]?.airlineName ?? '',
          'Flight Number': registration.travelPlans?.[0]?.flightNumber ?? '',
          'Arrival Date': registration.travelPlans?.[0]?.arrivalDatetime ? new Date(registration.travelPlans[0].arrivalDatetime).toLocaleDateString('en-GB') : '',
          'Arrival Time': registration.travelPlans?.[0]?.arrivalDatetime ? new Date(registration.travelPlans[0].arrivalDatetime).toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit' }) : '',
          'Arrival Location': registration.travel,
          'Preferred Roommate Name': registration.preferredRoomMate ?? '',
        }
      }

      return transformedReg;
    });
    // Generate filename
    const timestamp = new Date().toISOString().split('T')[0];
    const filename = `Registrations_${timestamp.replace(/-/g, '')}.xlsx`;

      // Excel options
      const excelOptions = {
        sheetName: 'Registrations Export',
        makeHeaderBold: true,
        autoWidth: true
      };

      // Generate Excel and upload to S3
      const s3Url = await this.excelService.jsonToExcelAndUpload(
        excelData,
        filename,
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        excelOptions
      );

      return s3Url;
    } catch (error) {
      this.logger.error('Error generating Excel download', error);
      throw new Error('Failed to generate Excel file');
    }
  }

/**
 * Helper method to format payment mode for display
 */
private formatPaymentMode(paymentMode: string,offlineMeta:any): string {
  if (!paymentMode) return '';
  
  switch (paymentMode.toLowerCase()) {
    case 'online':
      return 'Online ( Razorpay )';
    case 'offline':
      return offlineMeta?.paymentMode ?? 'Offline';
    default:
      return paymentMode;
  }
}

/**
 * Helper method to calculate tax amount based on base amount and tax percentage
 */
private calculateTaxAmount(baseAmount: number, taxPercentage: number): number {
  if (!baseAmount || !taxPercentage) return 0;
  return (baseAmount * taxPercentage) / 100;
}

/**
 * Helper method to format programme name with period
 */
private formatProgrammeNameWithPeriod(program: any, allocatedProgram: any): string {
  const programName = allocatedProgram?.name || program?.name || '';
  
  if (!programName) return '';
  
  // You can customize this format based on how you want to display the period
  const startDate = program?.startsAt ? new Date(program.startsAt).toLocaleDateString('en-GB') : '';
  const endDate = program?.endsAt ? new Date(program.endsAt).toLocaleDateString('en-GB') : '';
  
  if (startDate && endDate) {
    return `${programName} (${startDate} to ${endDate})`;
  } else if (startDate) {
    return `${programName} (From ${startDate})`;
  } else {
    return programName;
  }
}

  /**
   * Get table headers configuration based on user roles
   */
  private getTableHeaders(userRoles: string[] = []) {
    let headers = [
      {
        key: 'seekerName',
        label: 'Seeker Name',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '1B'
      },
      {
        key: 'profileUrl',
        label: 'Profile URL',
        sortable: false,
        filterable: false,
        type: 'string',
        order: '1A'
      },
      {
        key: 'age',
        label: 'Age',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '2A'
      },
      {
        key: 'gender',
        label: 'Gender',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '2B'
      },
      {
        key: 'location',
        label: 'Location',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '2C'
      },
      {
        key: 'averageRating',
        label: 'RM Rating',
        sortable: true,
        filterable: false,
        type: 'number',
        order : '5A'
      },
      {
        key: 'blessedWith',
        label: 'Blessed With',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '3'
      },
      {
        key: 'numberOfHDBs',
        label: 'No. of HDBs',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '4'
      },
      {
        key: 'rmComments',
        label: 'RM Comments',
        sortable: false,
        filterable: true,
        type: 'string',
        order: '5B'
      },
      {
        key: 'paymentStatus',
        label: 'Payment Status',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '6'
      },
      {
        key: 'travelPlanStatus',
        label: 'Travel Status',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '7'
      },
      {
        key: 'mobileNumber',
        label: 'Mobile Number',
        sortable: false,
        filterable: true,
        type: 'string',
        order: '8'
      }
    ];

    if (userRoles.includes('finance_manager')) {
      headers.push({
        key: 'invoiceStatus',
        label: 'Invoice Status',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '6A',
      });
    }

    // Remove headers for RM and operational manager roles
    const isUserRoleRMorOM = userRoles.includes('finance_manager') || userRoles.includes('operational_manger');
    if (isUserRoleRMorOM) {
      headers = headers.filter(header => 
        !['rmComments', 'averageRating'].includes(header.key)
      );
    }

    // Sort headers by order
    return headers.sort((a, b) => a.order.localeCompare(b.order));
  }

  /**
   * Flatten KPIs for easier frontend consumption
   */
  private flattenKPIs(kpis: any) {
      const flatKPIs: Array<{ 
          category: string; 
          categoryLabel: string; 
          [key: string]: any; 
      }> = [];

    // Registration KPIs
    if (kpis.registrations) {
      kpis.registrations.items.forEach(item => {
        flatKPIs.push({
          ...item,
          category: 'registrations',
          categoryLabel: kpis.registrations.categoryLabel
        });
      });
    }

    // Program KPIs (HDB1, HDB2, etc.)
    if (kpis.programs) {
      kpis.programs.items.forEach(item => {
        flatKPIs.push({
          ...item,
          category: 'programs',
          categoryLabel: kpis.programs.categoryLabel
        });
      });
    }

    // Payment KPIs
    if (kpis.payments) {
      kpis.payments.items.forEach(item => {
        flatKPIs.push({
          ...item,
          category: 'payments',
          categoryLabel: kpis.payments.categoryLabel
        });
      });
    }

    // Invoice KPIs
    if (kpis.invoices) {
      kpis.invoices.items.forEach(item => {
        flatKPIs.push({
          ...item,
          category: 'invoices',
          categoryLabel: kpis.invoices.categoryLabel
        });
      });
    }

    // Travel KPIs
    if (kpis.travelAndLogistics) {
      kpis.travelAndLogistics.items.forEach(item => {
        flatKPIs.push({
          ...item,
          category: 'travelAndLogistics',
          categoryLabel: kpis.travelAndLogistics.categoryLabel
        });
      });
    }
    const flatKPIsLength = flatKPIs.length + 1
    const orderedKPIs = flatKPIs.map((kpi) => {kpi.order = kpi.order <0 ? flatKPIsLength + kpi.order : kpi.order; return kpi;}).sort((a, b) => a.order - b.order);
    return orderedKPIs;
  }

   private getDraftRegTableHeaders(userRoles: string[] = []) {
    let headers = [
      {
        key: 'profileUrl',
        label: 'Profile URL',
        sortKey: 'profileUrl',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '1'
      },
      {
        key: 'seekerName',
        label: 'Seeker Name',
        sortKey: 'fullName',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '2'
      },
      {
        key: 'age',
        label: 'Age',
        sortKey: 'dob',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '3'
      },
      {
        key: 'gender',
        label: 'Gender',
        sortKey: 'gender',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '4'
      },
      {
        // email Address
        key: 'emailAddress',
        label: 'Email Address',
        sortKey: 'emailAddress',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '5'
      },
      {
        key: 'location',
        label: 'Location',
        sortKey: 'city',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '6'
      },
      {
        key: 'numberOfHDBs',
        label: 'No. of HDBs',
        sortKey: 'noOfHDBs',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '7'
      },
      {
        key: 'mobileNumber',
        label: 'Mobile Number',
        sortKey: 'mobileNumber',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '8'
      },
      {
        // rm Contact 
        key: 'rmContact',
        label: 'RM Contact',
        sortKey: '',
        sortable: false,
        filterable: false,
        type: 'string',
        order: '9'
      }
    ];

    // if user role is relational_manager then remove rmContact
    if (userRoles.includes('relational_manager')) {
      headers = headers.filter(header => header.key !== 'rmContact');
    }
    // Sort headers by order
    return headers.sort((a, b) => a.order.localeCompare(b.order));
  }

  /**
   * Calculate age from date of birth
   */
  private calculateAge(dateOfBirth: Date): number {
    if (!dateOfBirth) return 0;
    const today = new Date();
    const birthDate = new Date(dateOfBirth);
    let age = today.getFullYear() - birthDate.getFullYear();
    const monthDiff = today.getMonth() - birthDate.getMonth();
    if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
      age--;
    }
    return age;
  }

  private getFilterConfiguration(userRoles: string[]): any[] {
    const filters = [
      {
        key: 'gender',
        label: 'Gender',
        type: 'checkbox',
        options: [
          { label: 'Male', value: 'male' },
          { label: 'Female', value: 'female' }
        ]
      },
      {
        key: 'age',
        label: 'Age',
        type: 'checkbox',
        options: [
          { label: '<=20 years', value: '0-20' },
          { label: '21 - 30 years', value: '21-30' },
          { label: '31 - 50 years', value: '31-50' },
          { label: '51 - 65 years', value: '51-65' },
          { label: '> 65 years', value: '>65' }
        ]
      },
      {
        key: 'numberOfHdbs',
        label: 'Number of HDBs',
        type: 'checkbox',
        options: [
          {
            label: '1st Timer',
            value: '0',
          },
          {
            label: '1 - 10 HDBs/MSDs',
            value: '1-10',
          },
          {
            label: '>10 HDBs/MSDs',
            value: '>10',
          },
        ]
      },
      {
        key: 'location',
        label: 'Location',
        type: 'dropdown',
        options: [
          { label: 'Hyderabad', value: 'Hyderabad' },
          { label: 'Mumbai', value: 'Mumbai' },
          { label: 'Delhi', value: 'Delhi' },
          { label: 'Bangalore', value: 'Bangalore' },
          { label: 'Chennai', value: 'Chennai' },
          { label: 'Kolkata', value: 'Kolkata' },
          { label: 'Pune', value: 'Pune' },
          { label: 'Ahmedabad', value: 'Ahmedabad' },
          { label: 'Jaipur', value: 'Jaipur' },
          { label: 'Surat', value: 'Surat' },
          { label: 'Lucknow', value: 'Lucknow' },
          { label: 'Kanpur', value: 'Kanpur' },
          { label: 'Nagpur', value: 'Nagpur' },
          { label: 'Patna', value: 'Patna' },
          { label: 'Indore', value: 'Indore' },
          { label: 'Bhopal', value: 'Bhopal' },
          { label: 'Vadodara', value: 'Vadodara' },
          { label: 'Ludhiana', value: 'Ludhiana' },
          { label: 'Agra', value: 'Agra' },
          { label: 'Varanasi', value: 'Varanasi' },
          { label: 'Amritsar', value: 'Amritsar' },
          { label: 'Coimbatore', value: 'Coimbatore' },
          { label: 'Thiruvananthapuram', value: 'Thiruvananthapuram' },
          { label: 'Guwahati', value: 'Guwahati' },
          { label: 'Ranchi', value: 'Ranchi' },
          { label: 'Other', value: 'Other' },
        ]
      },
      {
        key: 'registrationStatus',
        label: 'Registration status',
        type: 'checkbox',
        options: [
          { label: 'Pending', value: 'pending' },
          { label: 'Completed', value: 'completed' },
          // { label: 'Cancelled', value: 'cancelled' },
          // { label: 'Waitlisted', value: 'waitlisted' }
        ]
      },
      {
        key: 'approvalStatus',
        label: 'Approval status', 
        type: 'checkbox',
        options: [
          { label: 'Pending', value: 'pending' },
          { label: 'Approved', value: 'approved' },
          { label: 'Rejected', value: 'rejected' },
          { label: 'On Hold', value: 'on_hold' }
        ]
      }
    ];

    filters.push({
        key: 'paymentMode',
        label: 'Payment mode',
        type: 'checkbox',
        options: [
          { label: 'Online', value: 'online' },
          { label: 'Offline', value: 'offline' },
        ]
      });
    // Add role-based filters
      filters.push({
        key: 'paymentStatus',
        label: 'Payment status',
        type: 'checkbox',
        options: [
          { label: 'Completed', value: 'payment_completed' },
          { label: 'Pending', value: 'payment_pending' },
        ]
      });
     

      filters.push({
        key: 'invoiceStatus',
        label: 'Invoice status',
        type: 'checkbox',
        options: [
          { label: 'Completed', value: 'invoice_completed' },
          { label: 'Pending', value: 'invoice_pending' },
        ]
      });

      filters.push({
        key: 'travelStatus',
        label: 'Travel status',
        type: 'checkbox',
        options: [
          { label: 'Completed', value: 'travel_completed' },
          { label: 'Pending', value: 'travel_pending' }
        ]
      });

      if (userRoles.includes('admin') || userRoles.includes('mahatria') || userRoles.includes('shoba')) {
          filters.push({
            key: 'dateRange',
            label: 'Seeker registration start & end date',
            type: 'date',
            options: [
              { label: 'Yes', value: 'yes' },
              { label: 'No', value: 'no' }
            ]
          });
      }

      // add preferredRoomMate
      filters.push({
        key: 'preferredRoomMate',
        label: 'Preferred Roommate',
        type: 'checkbox',
        options: [
          { label: 'Yes', value: 'yes' },
          { label: 'No', value: 'no' }
        ]
      });


    if (userRoles.includes('admin') || userRoles.includes('mahatria') || userRoles.includes('shoba')) {
      filters.push({
        key: 'swapRequests',
        label: 'Swap requests',
        type: 'checkbox',
        options: [
          { label: 'Can shift', value: 'can_shift' },
          { label: 'Wants swap', value: 'wants_swap' }
        ]
      });
    }

      filters.push(
        {
          key: 'organisation',
          label: 'From Organization',
          type: 'checkbox',
          options: [
            { label : 'Yes', value: 'yes' },
            { label : 'No', value: 'no' }
          ]
        }
      );


      if (
        !userRoles.includes('finance_manager') &&
        !userRoles.includes('operational_manger')
      ) {
        filters.push({
          key: 'rmRating',
          label: 'RM rating',
          type: 'checkbox',
          options: [
            { label: '1 & up', value: '1up' },
            { label: '2 & up', value: '2up' },
            { label: '3 & up', value: '3up' },
            { label: '4 & up', value: '4up' },
            { label: 'Only 5', value: 'only5' },
            { label: 'Yet to review', value: 'unrated' },
          ],
        });
      }


    return filters;
  }

  async getSeatAllocationFiltersConfiguration(): Promise<any[]> {
    const filters = [
      {
        key: 'age',
        label: 'Age',
        type: 'checkbox',
        options: [
          { label: '<=20 years', value: '0-20' },
          { label: '21 - 30 years', value: '21-30' },
          { label: '31 - 50 years', value: '31-50' },
          { label: '51 - 65 years', value: '51-65' },
          { label: '> 65 years', value: '>65' }
        ]
      },
      {
        key: 'numberOfHdbs',
        label: 'Number of HDBs',
        type: 'checkbox',
        options: [
          {
            label: '1st Timer',
            value: '0',
          },
          {
            label: '1 - 10 HDBs/MSDs',
            value: '1-10',
          },
          {
            label: '>10 HDBs/MSDs',
            value: '>10',
          },
        ]
      },
      {
        key: 'location',
        label: 'Location',
        type: 'dropdown',
        options: [
          { label: 'Hyderabad', value: 'Hyderabad' },
          { label: 'Mumbai', value: 'Mumbai' },
          { label: 'Delhi', value: 'Delhi' },
          { label: 'Bangalore', value: 'Bangalore' },
          { label: 'Chennai', value: 'Chennai' },
          { label: 'Kolkata', value: 'Kolkata' },
          { label: 'Pune', value: 'Pune' },
          { label: 'Ahmedabad', value: 'Ahmedabad' },
          { label: 'Jaipur', value: 'Jaipur' },
          { label: 'Surat', value: 'Surat' },
          { label: 'Lucknow', value: 'Lucknow' },
          { label: 'Kanpur', value: 'Kanpur' },
          { label: 'Nagpur', value: 'Nagpur' },
          { label: 'Patna', value: 'Patna' },
          { label: 'Indore', value: 'Indore' },
          { label: 'Bhopal', value: 'Bhopal' },
          { label: 'Vadodara', value: 'Vadodara' },
          { label: 'Ludhiana', value: 'Ludhiana' },
          { label: 'Agra', value: 'Agra' },
          { label: 'Varanasi', value: 'Varanasi' },
          { label: 'Amritsar', value: 'Amritsar' },
          { label: 'Coimbatore', value: 'Coimbatore' },
          { label: 'Thiruvananthapuram', value: 'Thiruvananthapuram' },
          { label: 'Guwahati', value: 'Guwahati' },
          { label: 'Ranchi', value: 'Ranchi' },
          { label: 'Other', value: 'Other' },
        ]
      },
      {
        key: 'organisation',
        label: 'Organisation',
        type: 'checkbox',
        options: [
          { label : 'Yes', value: 'yes' },
          { label : 'No', value: 'no' }
        ]
      }
    ];

        filters.push({
          key: 'rmRating',
          label: 'RM rating',
          type: 'checkbox',
          options: [
            { label: '1 & up', value: '1up' },
            { label: '2 & up', value: '2up' },
            { label: '3 & up', value: '3up' },
            { label: '4 & up', value: '4up' },
            { label: 'Only 5', value: 'only5' },
            { label: 'Yet to review', value: 'unrated' },
          ],
        });
    return filters;
  }


  private extractKPIFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const kpiFilters: Record<string, any> = {};

    if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
      const kpiFilterCondition = this.getKPIFilterCondition(
        parsedFilters.kpiCategory,
        parsedFilters.kpiFilter,
      );

      if (kpiFilterCondition) {
        Object.assign(kpiFilters, kpiFilterCondition);
      }
    }

    return kpiFilters;
  }

  /**
   * Extract data filters (non-KPI filters)
   */
  private extractDataFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const dataFilters = { ...parsedFilters };

    // Apply KPI filters to data if they exist
    if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
      const kpiFilterCondition = this.getKPIFilterCondition(
        parsedFilters.kpiCategory,
        parsedFilters.kpiFilter,
      );

      if (kpiFilterCondition) {
        Object.assign(dataFilters, kpiFilterCondition);
      }
    }

    return dataFilters;
  }

  /**
   * Remove KPI filter parameters from filters object
   */
  private removeKPIFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const cleanFilters = { ...parsedFilters };
    delete cleanFilters.kpiCategory;
    delete cleanFilters.kpiFilter;
    return cleanFilters;
  }

  /**
   * Get Mahatria-specific KPIs for program allocation management with registration data
   */
  async getMahatriaKPIs(
    limit: number,
    offset: number,
    programId: number,
    searchText: string = '',
    parsedFilters: Record<string, any> = {},
    user: any,
  ): Promise<{
    data: any[];
    pagination: any;
    kpis: {
      mahatria: any;
    };
    filters: any;
  }> {
    this.logger.log('Getting Mahatria KPIs with registration data', {
      limit,
      offset,
      programId,
      searchText,
      parsedFilters,
    });

    try {
      // Validate that the program requires approval for mahatria users
      const program = await this.repository.findProgramWithType(programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }
      const seatAllocationFilters = await this.getSeatAllocationFiltersConfiguration();
      if (!program.requiresApproval) {
        return {
          data: [],
          pagination: {
            totalPages: 0,
            pageNumber: 0,
            pageSize: limit,
            totalRecords: 0,
            numberOfRecords: 0,
          },
          kpis: {
            mahatria: {
              groupedProgramMetrics: {
                isGroupedProgram: false,
                totalRegistrations: 0,
                message: 'No approval required for this program',
              },
            },
          },
          filters: seatAllocationFilters,
        };
      }

      // Apply KPI filter condition if provided (for Mahatria-specific filters)
      let allKpiCounts = {};
      if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
         if (parsedFilters?.kpiCategory == 'all') {
        allKpiCounts = await this.repository.getAllRegistrationMetrics(programId) ?? {}
      }
        const kpiFilterCondition = this.getMahatriaKPIFilterCondition(
          parsedFilters.kpiCategory,
          parsedFilters.kpiFilter,
        );
        if (kpiFilterCondition) {
          Object.assign(parsedFilters, kpiFilterCondition);
        }
      }
      let allocatedProgramId = parsedFilters?.allocatedProgramId;
      let allocatedProgramSwaps
      if (parsedFilters?.allocatedProgramId) {
        const where = {
          currentProgramId: parsedFilters?.allocatedProgramId,
          status: SwapRequestStatus.ACTIVE,
          type: SwapType.WantsSwap
        }
        allocatedProgramSwaps = await this.programRegistrationSwapRepository.findAll(where);
      }

      // Get registration data WITH search text (for table filtering)
      const registrationResult = await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        null, // programSessionId - not used for Mahatria KPIs
        searchText, // ✅ Search applies to data table
        parsedFilters,
        [], // userRoles - empty for Mahatria context
        undefined, // filterRegistrationsByUserId - not used for Mahatria
      );

      // ✅ CRITICAL FIX: Generate Mahatria-specific KPIs WITHOUT search text
      const mahatriaKPIResult = await this.generateMahatriaKPIs({
        programId,
        searchText: '', // ✅ Empty search text for KPIs
        parsedFilters,
      });

      // Format the KPIs similar to default KPIs structure
      const formattedMahatriaKPIs = this.formatMahatriaKPIs(mahatriaKPIResult);
      // Add all KPIs if available
      formattedMahatriaKPIs['allKpis'] = allKpiCounts;
      // Return in the expected format with only Mahatria KPIs
      if (allocatedProgramId) {
        let formattedAllocatedSubPrograms = formattedMahatriaKPIs?.allocated?.programs || [];

        formattedAllocatedSubPrograms = formattedAllocatedSubPrograms.map((program) => {
          program['swapRequestsCount'] = program.programId === allocatedProgramId ? allocatedProgramSwaps.length : 0
          return program
        });
      }
      const swapData = await this.programRegistrationService.getSwapRequestKPI({programId});
      formattedMahatriaKPIs['swapRequests'] = swapData?.kpis || [];
      return {
        data: registrationResult.data,
        pagination: registrationResult.pagination,
        kpis: {
          mahatria: {
            groupedProgramMetrics: formattedMahatriaKPIs,
          },
        },
        filters: seatAllocationFilters,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationMessages.GET_BY_ID);
    try {
      return await this.repository.findRegistrationById(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async getQuestionResponses(id: number) {
    this.logger.log(registrationMessages.GET_QA_REQUEST_RECEIVED, { id });
    try {
      return await this.repository.findQuestionResponsesByRegistrationId(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_QUESTION_RESPONSE_GET_FAILED, error);
    }
  }

  async getRegistrationStatuses(id: number) {
    this.logger.log(registrationMessages.GET_STATUSES, { id });
    try {
      const registration = await this.repository.findRegistrationById(id);
      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }

      const sortLatest = (arr?: any[]) => {
        if (!arr || arr.length === 0) return null;
        return arr.sort((a, b) => {
          const dateA = new Date(a.updatedAt ?? a.createdAt ?? 0).getTime();
          const dateB = new Date(b.updatedAt ?? b.createdAt ?? 0).getTime();
          return dateB - dateA;
        })[0];
      };

      const latestTravelPlanStatus = sortLatest(registration.travelPlans)?.travelPlanStatus ?? null;
      const latestTravelInfoStatus = sortLatest(registration.travelInfo)?.travelInfoStatus ?? null;

      let travelStatus: string | null = null;
      if (latestTravelPlanStatus === 'completed' && latestTravelInfoStatus === 'completed') {
        travelStatus = 'completed';
      } else if (latestTravelPlanStatus === 'pending' || latestTravelInfoStatus === 'pending') {
        travelStatus = 'pending';
      }

      return {
        registrationStatus: registration.registrationStatus ?? null,
        basicDetailsRegistrationStatus: registration.basicDetailsStatus ?? null,
        approvalStatus: sortLatest(registration.approvals)?.approvalStatus ?? null,
        paymentStatus: sortLatest(registration.paymentDetails)?.paymentStatus ?? null,
        invoiceStatus: sortLatest(registration.invoiceDetails)?.invoiceStatus ?? null,
        travelInfoStatus: latestTravelInfoStatus,
        travelPlanStatus: latestTravelPlanStatus,
        travelStatus,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  // ===== DEFAULT KPI GENERATION METHODS =====

  /**
   * Generate default KPIs for list view based on user roles
   */
  private async generateDefaultKPIs(parameters: {
    programId?: number | null;
    programSessionId?: number | null;
    searchText?: string; // ✅ This should be empty string when called
    parsedFilters?: Record<string, any>;
    userRoles: string[];
    filterRegistrationsByUserId?: number | null;
    activeKpiCategory?: string;
  }): Promise<DefaultKPIResult> {
    try {
      const {
        programId,
        programSessionId,
        searchText, // ✅ This will be empty string
        parsedFilters,
        userRoles,
        filterRegistrationsByUserId,
        activeKpiCategory,
      } = parameters;

      const kpis: DefaultKPIResult = {
        registrations: {
          categoryLabel: 'Registration Metrics',
          items: [],
        },
      };

      // ✅ FIXED: Get seats metrics WITHOUT search text
      const seatsMetrics = await this.repository.getSeatsMetrics(
        programId,
        programSessionId,
        parsedFilters,
        activeKpiCategory === 'registrations' ? searchText : '',
        filterRegistrationsByUserId,
      );


      // Build registration KPIs
      kpis.registrations.items = [
        {
          label: 'All',
          value: parseInt(seatsMetrics?.all ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'all',
          order: 1,
        },
        {
          label: 'Unassigned',
          value: parseInt(seatsMetrics?.pending ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'newSeekersPending',
          order: 2,
        },
        {
          label: 'Blessed',
          value: parseInt(seatsMetrics?.approved ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'approvedSeekers',
          order: 3,
        },
        {
          label: 'Yet to Decide',
          value: parseInt(seatsMetrics?.onHold ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'onHold',
          order: -2,
        },
        // {
        //   label: 'Cancelled',
        //   value: parseInt(seatsMetrics?.cancelled ?? '0'),
        //   kpiCategory: 'registrations',
        //   kpiFilter: 'cancelled'
        // },
        {
          label: 'On Hold',
          value: parseInt(seatsMetrics?.rejected ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'rejected',
          order: -1,
        },
      ];

      // Add program-specific KPIs if looking at a specific program
      if (programId) {
        // ✅ FIXED: Call without search text
        const programKPIs = await this.generateProgramKPIs(
          programId,
          parsedFilters,
          filterRegistrationsByUserId,
          activeKpiCategory === 'programs' ? searchText : '',
          4,
        );
        if (programKPIs.length > 0) {
          kpis.programs = {
            categoryLabel: 'Program Allocation',
            items: programKPIs,
          };
        }
      }

      // // Add payment KPIs for relevant roles
      // if (this.shouldShowPaymentKPIs(userRoles)) {
      //   // ✅ FIXED: Remove search text
      //   const paymentMetrics = await this.repository.getPaymentsMetrics(
      //     programId,
      //     programSessionId,
      //     parsedFilters,
      //     activeKpiCategory === 'payments' ? searchText : '',
      //     filterRegistrationsByUserId
      //   );

      //   kpis.payments = {
      //     categoryLabel: 'Payment Status',
      //     items: [
      //       {
      //         label: 'All Payments',
      //         value: parseInt(paymentMetrics?.all ?? '0'),
      //         kpiCategory: 'payments',
      //         kpiFilter: 'all'
      //       },
      //       {
      //         label: 'Payment Pending',
      //         value: parseInt(paymentMetrics?.paymentPending ?? '0'),
      //         kpiCategory: 'payments',
      //         kpiFilter: 'paymentPending'
      //       },
      //       {
      //         label: 'Payment Completed',
      //         value: parseInt(paymentMetrics?.paymentCompleted ?? '0'),
      //         kpiCategory: 'payments',
      //         kpiFilter: 'paymentCompleted'
      //       }
      //     ]
      //   };
      // }

      // // Add invoice KPIs for relevant roles
      // if (this.shouldShowInvoiceKPIs(userRoles)) {
      //   // ✅ FIXED: Remove search text
      //   const invoiceMetrics = await this.repository.getInvoicesMetrics(
      //     programId,
      //     programSessionId,
      //     parsedFilters,
      //     activeKpiCategory === 'invoices' ? searchText : '',
      //     filterRegistrationsByUserId
      //   );

      //   kpis.invoices = {
      //     categoryLabel: 'Invoice Status',
      //     items: [
      //       {
      //         label: 'All Invoices',
      //         value: parseInt(invoiceMetrics?.all ?? '0'),
      //         kpiCategory: 'invoices',
      //         kpiFilter: 'all'
      //       },
      //       {
      //         label: 'Invoice Pending',
      //         value: parseInt(invoiceMetrics?.invoicePending ?? '0'),
      //         kpiCategory: 'invoices',
      //         kpiFilter: 'invoicePending'
      //       },
      //       {
      //         label: 'Invoice Completed',
      //         value: parseInt(invoiceMetrics?.invoiceCompleted ?? '0'),
      //         kpiCategory: 'invoices',
      //         kpiFilter: 'invoiceCompleted'
      //       }
      //     ]
      //   };
      // }

      // // Add travel KPIs for relevant roles
      // if (this.shouldShowTravelKPIs(userRoles)) {
      //   // ✅ FIXED: Remove search text
      //   const travelMetrics = await this.repository.getTravelMetrics(
      //     programId,
      //     programSessionId,
      //     parsedFilters,
      //     activeKpiCategory === 'travelAndLogistics' ? searchText : '',
      //     filterRegistrationsByUserId
      //   );

      //   kpis.travelAndLogistics = {
      //     categoryLabel: 'Travel & Logistics',
      //     items: [
      //       {
      //         label: 'All Travel',
      //         value: parseInt(travelMetrics?.all ?? '0'),
      //         kpiCategory: 'travelAndLogistics',
      //         kpiFilter: 'all'
      //       },
      //       {
      //         label: 'Travel Pending',
      //         value: parseInt(travelMetrics?.travelPending ?? '0'),
      //         kpiCategory: 'travelAndLogistics',
      //         kpiFilter: 'travelPending'
      //       },
      //       {
      //         label: 'Travel Completed',
      //         value: parseInt(travelMetrics?.travelCompleted ?? '0'),
      //         kpiCategory: 'travelAndLogistics',
      //         kpiFilter: 'travelCompleted'
      //       },
      //       {
      //         label: 'Logistics Pending',
      //         value: parseInt(travelMetrics?.logisticsPending ?? '0'),
      //         kpiCategory: 'travelAndLogistics',
      //         kpiFilter: 'logisticsPending'
      //       },
      //       {
      //         label: 'Logistics Completed',
      //         value: parseInt(travelMetrics?.logisticsCompleted ?? '0'),
      //         kpiCategory: 'travelAndLogistics',
      //         kpiFilter: 'logisticsCompleted'
      //       }
      //     ]
      //   };
      // }

      return kpis;
    } catch (error) {
      this.logger.error('Error generating default KPIs', error);
      throw error;
    }
  }

  /**
   * Generate program-specific KPIs (HDB1, HDB2, MSD1, etc.)
   */
  private async generateProgramKPIs(
    programId: number,
    parsedFilters?: Record<string, any>,
    filterRegistrationsByUserId?: number | null,
    searchText: string = '',
    orderStartNumber: number = 0,
  ): Promise<KPIItem[]> {
    try {
      // Check if this is a grouped program
      const mainProgram = await this.repository.findProgramWithType(programId);
      if (!mainProgram?.isGroupedProgram) {
        return [];
      }

      // Get all programs in this group
      const groupedPrograms = await this.repository.findGroupedPrograms(programId);
      const programKPIs: KPIItem[] = [];
      for (const [index, program] of groupedPrograms.entries()) {
        // ✅ FIXED: Get registration count WITHOUT search text
        const count = await this.repository.getTotalAllocatedRegistrations(
          mainProgram.id,
          program.id,
          parsedFilters,
          searchText,
          filterRegistrationsByUserId,
        );


        programKPIs.push({
          label: program.name,
          value: count,
          kpiCategory: 'programs',
          kpiFilter: `program_${program.id}`,
          order: orderStartNumber + index || 0,
        });
      }

      return programKPIs;
    } catch (error) {
      this.logger.error('Error generating program KPIs', error);
      return [];
    }
  }

  /**
   * Role-based visibility methods
   */
  private shouldShowPaymentKPIs(userRoles: string[]): boolean {
    const paymentRoles = ['admin', 'finance_manager'];
    return userRoles.some((role) => paymentRoles.includes(role));
  }

  private shouldShowInvoiceKPIs(userRoles: string[]): boolean {
    const invoiceRoles = ['admin', 'finance_manager'];
    return userRoles.some((role) => invoiceRoles.includes(role));
  }

  private shouldShowTravelKPIs(userRoles: string[]): boolean {
    const travelRoles = ['admin', 'operational_manager'];
    return userRoles.some((role) => travelRoles.includes(role));
  }

  // ===== PRIVATE BUSINESS LOGIC METHODS =====

  private async validateUserNotAlreadyRegistered(
    userId: number,
    programId: number,
    programSessionId?: number,
  ): Promise<void> {
    const existingRegistration = await this.repository.findExistingUserRegistration(
      userId,
      programId,
      programSessionId,
    );
    if (existingRegistration) {
      throw new InifniBadRequestException(
        ERROR_CODES.USER_ALREADY_REGISTERED,
        null,
        null,
        `User ${userId} is already registered for program ${programId}${programSessionId ? ` session ${programSessionId}` : ''}`,
      );
    }
  }

  private async validateAndGetProgram(programId: number): Promise<Program> {
    const program = await this.repository.findProgramWithType(programId);
    if (!program) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        programId.toString(),
      );
    }
    return program;
  }

  private getRegistrationLevel(program: Program): string {
    const registrationLevel = program.type?.registrationLevel;
    if (!registrationLevel) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
        null,
        null,
        'Program type must have a registration level defined',
      );
    }
    return registrationLevel;
  }

  private async validateSessionIfRequired(
    dto: CreateRegistrationDto,
    program: Program,
    registrationLevel: string,
  ): Promise<ProgramSession | null> {
    if (registrationLevel !== 'session') {
      return null;
    }

    if (!dto.programSessionId) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
        null,
        null,
        'Program session ID is required for session-level registration',
      );
    }

    const session = await this.repository.findSessionForProgram(dto.programSessionId, program.id);
    if (!session) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_MISMATCH,
        null,
        null,
        dto.programSessionId.toString(),
      );
    }

    return session;
  }

  async findAll(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
  ) {
    this.logger.log(registrationMessages.GET_ALL, {
      limit,
      offset,
      programId,
      programSessionId,
      searchText,
      parsedFilters,
      userRoles,
    });
    try {
      if (userRoles.includes('mahatria')) {
        if (programId) {
          const program = await this.repository.findProgramWithType(programId);
          if (!program) {
            throw new InifniNotFoundException(
              ERROR_CODES.PROGRAM_NOTFOUND,
              null,
              null,
              programId.toString(),
            );
          }

          if (!program.requiresApproval) {
            return {
              message: 'No approval required for this program',
              data: [],
              pagination: {
                totalPages: 0,
                pageNumber: 1,
                pageSize: limit,
                totalRecords: 0,
                numberOfRecords: 0,
              },
            };
          }
        }
      }

      // Check if user is RM
      const isRMUser = userRoles.includes('relational_manager');
      return await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText,
        parsedFilters,
        userRoles,
        isRMUser ? filterRegistrationsByUserId : undefined,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  private async validateRegistrationDates(
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
  ): Promise<void> {
    const currentTimeUTC = new Date().getTime(); // current UTC timestamp

    let registrationStartDateTime: Date | null = null;
    let registrationEndDateTime: Date | null = null;

    if (registrationLevel === 'session' && session) {
      registrationStartDateTime = session.registrationStartsAt;
      registrationEndDateTime = session.registrationEndsAt;
    } else {
      registrationStartDateTime = program.registrationStartsAt;
      registrationEndDateTime = program.registrationEndsAt;
    }

    if (registrationStartDateTime && currentTimeUTC < registrationStartDateTime.getTime()) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_NOT_STARTED,
        null,
        null,
        `Registration will start on ${registrationStartDateTime.toLocaleString('en-IN', { timeZone: 'Asia/Kolkata' })}`,
      );
    }

    if (registrationEndDateTime && currentTimeUTC > registrationEndDateTime.getTime()) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_CLOSED,
        null,
        null,
        `Registration closed on ${registrationEndDateTime.toLocaleString('en-IN', { timeZone: 'Asia/Kolkata' })}`,
      );
    }
  }

  async checkSeatAvailability(
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
  ) {
    const seatsInfo = registrationLevel === 'session' ? session : program;
    const isLimitedSeats = seatsInfo?.limitedSeats === true;
    const isWaitlistEnabled = seatsInfo?.waitlistApplicable === true;

    // if program or session has requiresApproval, we don't need to check seats
    if (program.requiresApproval || (session && session.requiresApproval)) {
      return {
        isWaitlistTriggered: false,
        canRegister: true,
        shouldWaitlist: false,
      };
    }

    if (!isLimitedSeats) {
      return {
        isWaitlistTriggered: false,
        canRegister: true,
        shouldWaitlist: false,
      };
    } else if (!isWaitlistEnabled) {
      return {
        isWaitlistTriggered: false,
        canRegister:
          registrationLevel === 'session'
            ? (session?.reservedSeats ?? 0) < (session?.totalSeats ?? 0)
            : (program?.filledSeats || 0) < (seatsInfo?.totalSeats || 0),
        shouldWaitlist: false,
      };
    }
    const totalSeats = seatsInfo?.totalSeats || 0;
    const waitlistTriggerCount = seatsInfo?.waitlistTriggerCount || 0;
    // Only Program entity has filledSeats, ProgramSession doesn't
    const filledSeats = registrationLevel === 'session' ? 0 : program?.filledSeats || 0;

    const isWaitlistTriggered = filledSeats >= waitlistTriggerCount;
    const canDirectRegister = filledSeats < waitlistTriggerCount && !isWaitlistTriggered;
    const canRegister = filledSeats < totalSeats;
    const shouldWaitlist = !canDirectRegister && canRegister;

    return {
      totalSeats,
      waitlistTriggerCount,
      filledSeats,
      isWaitlistTriggered,
      canRegister,
      shouldWaitlist,
    };
  }

  private async validateQuestionAnswers(
    dto: CreateRegistrationDto,
    programId: number,
    sessionId?: number,
  ) {
    const questionMappings = await this.repository.findProgramQuestions(programId, sessionId);
    const validQuestionIds = questionMappings.map((m) => m.question.id);
    const invalid = dto.answers.filter((a) => !validQuestionIds.includes(a.questionId));

    if (invalid.length) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_INVALID_QUESTION,
        null,
        null,
        `Invalid question IDs: ${invalid.map((i) => i.questionId).join(', ')}`,
      );
    }

    return questionMappings;
  }

  private processQuestionAnswers(answers: any[], questionMappings: any[], userId: number) {
    const questionMap = new Map<number, Question>();
    questionMappings.forEach((m) => questionMap.set(m.question.id, m.question));

    const customResponses: RegistrationCustomResponse[] = [];
    const groupedDirectLocationAnswers: Map<
      string,
      { columnName: string; value: any; question: Question }[]
    > = new Map();
    const qaMappings: { questionId: number; answerValue: any }[] = [];

    for (const ans of answers) {
      const question = questionMap.get(ans.questionId);
      const location = question?.answerLocation;

      if (location) {
        const [tableName, columnName] = location.split('.');
        if (!groupedDirectLocationAnswers.has(tableName)) {
          groupedDirectLocationAnswers.set(tableName, []);
        }
        groupedDirectLocationAnswers
          .get(tableName)!
          .push({ columnName, value: ans.answer, question });
      } else {
        customResponses.push({
          questionId: ans.questionId,
          responseValue: ans.answer,
          createdBy: { id: userId } as any,
          updatedBy: { id: userId } as any,
        } as RegistrationCustomResponse);
      }
      qaMappings.push({ questionId: ans.questionId, answerValue: ans.answer });
    }

    return {
      customResponses,
      groupedDirectLocationAnswers,
      qaMappings,
    };
  }

  private determineRegistrationOutcome(seatInfo: any, program: Program) {
    const requiresPayment = program.requiresPayment === true;
    const approvalRequired = program.requiresApproval === true;

    // Determine registration path
    if (seatInfo.isWaitlistTriggered && seatInfo.shouldWaitlist) {
      return { type: 'waitlist', seatInfo };
    }

    if (!seatInfo.canRegister) {
      return { type: 'no_space', seatInfo };
    }

    if (approvalRequired) {
      return { type: 'pending_approval', requiresPayment };
    }

    return {
      type: 'direct_registration',
      requiresPayment,
      finalStatus: requiresPayment
        ? RegistrationStatusEnum.PENDING
        : RegistrationStatusEnum.COMPLETED,
    };
  }

  private async validateRegistrationForUpdate(registrationId: number, user: User,updateDto): Promise<ProgramRegistration> {
    this.logger.log(`Validating registration for update: ${registrationId}`);
    const registration = await this.repository.findRegistrationWithRelations(registrationId);
    if (!registration) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }

    // Check if user is a viewer
    const isViewer = user.userRoleMaps.some((urm) => urm.role.name === 'viewer');
    console.log('isViewer:', isViewer);
    if (isViewer) {
      const formSection = updateDto?.formSection;
      await this.validateViewerRegistrationUpdate(registration,formSection);
    }

    return registration;
  }

  /**
   * Validates if a viewer can update the registration
   * @param registration - The registration to validate
   * @throws InifniBadRequestException if validation fails
   */
  private async validateViewerRegistrationUpdate(
    registration: ProgramRegistration,
    formSection: FormSection | null,
  ): Promise<void> {
    // Check for cancelled status
    if (registration.registrationStatus === RegistrationStatusEnum.CANCELLED) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_INVALID_STATUS,
        null,
        null,
        'Cannot update cancelled registration'
      );
    }

    // Check for completed status
    if (registration.registrationStatus === RegistrationStatusEnum.COMPLETED) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_INVALID_STATUS,
        null,
        null,
        'Cannot update completed registration'
      );
    }

    // Check for approved status
    const approval = await this.repository.findApprovalStatus(registration.id);
    console.log('Approval status:', approval?.approvalStatus);
    const formSectionKey = formSection?.key || '';
    if (
      (formSectionKey === 'FS_BASICDETAILS' || formSectionKey === 'FS_MAHATRIAQUESTIONS') &&
      approval?.approvalStatus === ApprovalStatusEnum.APPROVED
    ) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_ALREADY_APPROVED,
        null,
        null
      );
    }
  }

  private isCancellationRequest(dto: UpdateRegistrationDto): boolean {
    return !!(dto.cancellationDate && dto.cancelledBy);
  }

  private async validateApprovalStatus(registration: any): Promise<void> {
    if (!registration.program?.requiresApproval) {
      return;
    }

    const pendingApproval = await this.repository.findPendingApproval(registration.id);
    if (pendingApproval) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_PENDING_APPROVAL,
        null,
        null,
        registration.id.toString(),
      );
    }
  }

  private async validateFormSection(formSectionId: number) {
    const formSection = await this.repository.findFormSection(formSectionId);
    if (!formSection) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
        null,
        null,
        `Form section with ID ${formSectionId} not found`,
      );
    }
    return formSection;
  }

  private async validateProgramForUpdate(programId: number, registration: any): Promise<Program> {
    if (!registration.program || registration.program.id !== programId) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        programId.toString(),
      );
    }
    return registration.program;
  }

  private async validateSessionForUpdate(
    dto: UpdateRegistrationDto,
    registration: any,
    program: Program,
  ): Promise<ProgramSession | null> {
    const registrationLevel = program.type?.registrationLevel;

    if (registrationLevel !== 'session') {
      return null;
    }

    const sessionId = dto.programSessionId ?? registration.programSession?.id;
    if (!sessionId) {
      throw new InifniBadRequestException(ERROR_CODES.PROGRAM_SESSION_NOTFOUND);
    }

    const session = await this.repository.findSessionForProgram(sessionId, program.id);
    if (!session) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_MISMATCH,
        null,
        null,
        sessionId.toString(),
      );
    }

    return session;
  }

  private async validateSectionQuestions(
    dto: UpdateRegistrationDto,
    programId: number,
    sessionId: number | undefined,
    formSection: FormSection | null,
  ) {
    const questionMappings = await this.repository.findProgramQuestions(programId, sessionId);
    let sectionQuestions: ProgramQuestion[] = questionMappings;
    if (
      formSection &&
      formSection.key !== 'FS_BASICDETAILS' &&
      formSection.key !== 'FS_MAHATRIAQUESTIONS'
    ) {
      sectionQuestions = questionMappings.filter(
        (m) => m.question.formSection?.id === formSection.id,
      );

      if (sectionQuestions.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_INVALID_QUESTION,
          null,
          null,
          `No questions found for form section ID: ${formSection.id}`,
        );
      }

      const validQuestionIds = sectionQuestions.map((m) => m.question.id);
      const invalid = dto.answers.filter((a) => !validQuestionIds.includes(a.questionId));
      if (invalid.length) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_INVALID_QUESTION,
          null,
          null,
          invalid.map((i) => i.questionId).join(','),
        );
      }
    }

    return sectionQuestions;
  }

  /**
   * Format Mahatria KPIs to match the structure of default KPIs
   */
  private formatMahatriaKPIs(kpis: MahatriaKPIResult): any {
    if (!kpis.isGroupedProgram) {
      return {
        isGroupedProgram: false,
        categoryLabel: 'Program Registrations',
        totalRegistrations: kpis.totalRegistrations || 0,
      };
    }

    const formatted = {
      isGroupedProgram: true,
      primaryProgramId: kpis.primaryProgramId,
      primaryProgramName: kpis.primaryProgramName,
      allocated: {
        categoryLabel: 'Allocated Programs',
        programs:
          kpis.allocated?.programs?.map((program) => ({
            programId: program.programId,
            programName: program.programName,
            allocatedCount: program.allocatedCount,
            maleCount: program.maleCount,
            femaleCount: program.femaleCount,
            organisationUserCount: program.organisationUserCount,
            totalSeatsCount: program.totalSeatsCount,
            maleOrganisationUserCount: program.maleOrganisationUserCount,
            femaleOrganisationUserCount: program.femaleOrganisationUserCount,
            roomMatePreferenceCount: program.roomMatePreferenceCount,
            label: `${program.programName} (${program.allocatedCount})`,
            kpiCategory: 'allocated',
            kpiFilter: `program_${program.programId}`,
          })) || [],
        totals: {
          holdCount: {
            label: 'On Hold',
            value: Number(kpis.allocated?.holdCount || 0),
            kpiCategory: 'allocated',
            kpiFilter: 'hold',
            femaleCount: Number(kpis.allocated?.holdFemaleCount || 0),
            maleCount: Number(kpis.allocated?.holdMaleCount || 0),
            maleOrganisationUserCount: Number(kpis.allocated?.holdMaleOrganisationUserCount || 0),
            femaleOrganisationUserCount: Number(kpis.allocated?.holdFemaleOrganisationUserCount || 0),
            organisationUserCount: Number(kpis.allocated?.holdOrganisationUserCount || 0),
          },
          rejectCount: {
            label: 'Rejected',
            value: Number(kpis.allocated?.rejectCount || 0),
            kpiCategory: 'allocated',
            kpiFilter: 'reject',
            femaleCount: Number(kpis.allocated?.rejectFemaleCount || 0),
            maleCount: Number(kpis.allocated?.rejectMaleCount || 0),
            maleOrganisationUserCount: Number(kpis.allocated?.rejectMaleOrganisationUserCount || 0),
            femaleOrganisationUserCount: Number(kpis.allocated?.rejectFemaleOrganisationUserCount || 0),
            organisationUserCount: Number(kpis.allocated?.rejectOrganisationUserCount || 0),
          },
        },
      },
      unallocated: {
        categoryLabel: 'Unallocated Programs',
        programs:
          kpis.unallocated?.programs?.map((program) => ({
            programId: program.programId,
            programName: program.programName,
            count: program.count,
            maleCount: program.maleCount,
            femaleCount: program.femaleCount,
            organisationUserCount: program.organisationUserCount,
            maleOrganisationUserCount: program.maleOrganisationUserCount,
            femaleOrganisationUserCount: program.femaleOrganisationUserCount,
            totalSeatsCount: program.totalSeatsCount,
            label: `${program.programName} (${program.count})`,
            kpiCategory: 'unallocated',
            kpiFilter: `program_${program.programId}`,
          })) || [],
        totals: {
          totalMaleCount: {
            label: 'Total Male',
            value: kpis.unallocated?.totalMaleCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'male',
            organisationUserCount: kpis.unallocated?.totalMaleOrganisationUserCount || 0,
          },
          totalFemaleCount: {
            label: 'Total Female',
            value: kpis.unallocated?.totalFemaleCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'female',
            organisationUserCount: kpis.unallocated?.totalFemaleOrganisationUserCount || 0,
          },
          totalUnallocatedCount: {
            label: 'Total Unallocated',
            value: kpis.unallocated?.totalUnallocatedCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'total_unallocated',
            organisationUserCount: kpis.unallocated?.totalOrganisationUserCount || 0,
          },
          totalMahatriaChoiceCount: {
            label: 'Mahatria Choice',
            value: kpis.unallocated?.totalMahatriaChoiceCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'mahatria_choice',
            organisationUserCount: kpis.unallocated?.totalMahatriaChoiceOrganisationUserCount || 0,
          },
        },
      },
    };

    return formatted;
  }

  /**
   * Generate Mahatria-specific KPIs for program allocation management
   */
  private async generateMahatriaKPIs(parameters: {
    programId: number;
    searchText?: string; // ✅ This should be empty string when called
    parsedFilters?: Record<string, any>;
  }): Promise<MahatriaKPIResult> {
    try {
      const { programId, parsedFilters, searchText } = parameters;

      // Get the main program to check if it's a grouped program
      const mainProgram = await this.repository.findProgramWithType(programId);

      if (!mainProgram) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }

      // Check if this is a grouped program
      const isGroupedProgram = mainProgram.isGroupedProgram || false;

      if (isGroupedProgram) {
        // Step 1: Get ALL programs first
        const allPrograms = await this.repository.findGroupedPrograms(programId);

        // Initialize metrics objects
        const allocated: AllocatedProgramMetrics = {
          programs: [],
          holdCount: 0,
          rejectCount: 0,
          holdMaleCount: 0,
          holdMaleOrganisationUserCount: 0,
          holdFemaleOrganisationUserCount: 0,
          holdFemaleCount: 0,
          rejectFemaleCount: 0,
          rejectMaleOrganisationUserCount: 0,
          rejectFemaleOrganisationUserCount: 0,
          rejectMaleCount: 0,
          holdOrganisationUserCount: 0,
          rejectOrganisationUserCount: 0,
        };

        const unallocated: UnallocatedProgramMetrics = {
          programs: [],
          totalMaleCount: 0,
          totalFemaleCount: 0,
          totalUnallocatedCount: 0,
          totalMahatriaChoiceCount: 0,
          totalOrganisationUserCount: 0,
          totalMaleOrganisationUserCount: 0,
          totalFemaleOrganisationUserCount: 0,
          totalMahatriaChoiceOrganisationUserCount: 0,
        };

        // Add Mahatria Choice metrics
        const mahatriaChoice = {
          count: 0,
          maleCount: 0,
          femaleCount: 0,
          organisationUserCount: 0,
        };

        let totalUnallocatedMaleCount = 0;
        let totalUnallocatedFemaleCount = 0;
        let totalUnalocatedCount = 0;
        let totalUnallocatedOrganisationCount = 0;
        let totalUnallocatedMaleOrganisationCount = 0;
        let totalUnallocatedFemaleOrganisationCount = 0;
        let totalMahatriaChoiceOrganisationUserCount = 0;

        const unallocatedPrograms: {
          programId: number;
          programName: string;
          count: number;
          maleCount: number;
          femaleCount: number;
          organisationUserCount: number;
          maleOrganisationUserCount: number;
          femaleOrganisationUserCount: number;
          totalSeatsCount: number;
        }[] = [];
        const allocatedPrograms: {
          programId: number;
          programName: string;
          allocatedCount: number;
          maleCount: number;
          femaleCount: number;
          organisationUserCount: number;
          totalSeatsCount: number;
          maleOrganisationUserCount: number;
          femaleOrganisationUserCount: number;
          roomMatePreferenceCount: number;
        }[] = [];

        // ✅ FIXED: Build base query for approvals WITHOUT search text
        const approvalResult = await this.repository.getApprovalMetrics(
          programId,
          parsedFilters,
          '',
        );

        allocated.holdMaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.MALE)[0]?.holdCount || 0);
        allocated.holdFemaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.FEMALE)[0]?.holdCount || 0);
        allocated.holdOrganisationUserCount =
          Number(approvalResult.find((record) => record.gender === GenderEnum.MALE)?.holdOrgCount || 0) +
          Number(approvalResult.find((record) => record.gender === GenderEnum.FEMALE)?.holdOrgCount || 0);
        allocated.holdCount = Number(allocated.holdMaleCount || 0) + Number(allocated.holdFemaleCount || 0) || 0;
        allocated.rejectMaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.MALE)[0]?.rejectedCount || 0);
        allocated.rejectFemaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.FEMALE)[0]?.rejectedCount || 0);
        allocated.rejectOrganisationUserCount =
          Number(approvalResult.find((record) => record.gender === GenderEnum.MALE)?.rejectedOrgCount || 0) +
          Number(approvalResult.find((record) => record.gender === GenderEnum.FEMALE)?.rejectedOrgCount || 0);
        allocated.rejectCount = Number(allocated.rejectMaleCount || 0) + Number(allocated.rejectFemaleCount || 0) || 0;
        allocated.holdMaleOrganisationUserCount =
          Number(approvalResult.find((record) =>
            record.gender === GenderEnum.MALE && record.approvalStatus === ApprovalStatusEnum.ON_HOLD,
          )?.holdOrgCount || 0);
        allocated.holdFemaleOrganisationUserCount =
          Number(approvalResult.find((record) =>
            record.gender === GenderEnum.FEMALE && record.approvalStatus === ApprovalStatusEnum.ON_HOLD,
          )?.holdOrgCount || 0);
        allocated.rejectMaleOrganisationUserCount =
          Number(approvalResult.find((record) => 
            record.gender === GenderEnum.MALE && record.approvalStatus === ApprovalStatusEnum.REJECTED,
          )?.rejectedOrgCount || 0);
        allocated.rejectFemaleOrganisationUserCount =
          Number(approvalResult.find((record) => 
            record.gender === GenderEnum.FEMALE && record.approvalStatus === ApprovalStatusEnum.REJECTED,
          )?.rejectedOrgCount || 0);
        // ✅ FIXED: Query for Mahatria Choice WITHOUT search text
        const mahatriaChoiceUnallocatedResult = await this.repository.getMahatriaChoiceMetrics(
          programId,
          parsedFilters,
          '',
        );
        mahatriaChoice.count = parseInt(mahatriaChoiceUnallocatedResult?.count ?? '0');
        mahatriaChoice.maleCount = parseInt(mahatriaChoiceUnallocatedResult?.maleCount ?? '0');
        mahatriaChoice.femaleCount = parseInt(mahatriaChoiceUnallocatedResult?.femaleCount ?? '0');
        mahatriaChoice.organisationUserCount = parseInt(mahatriaChoiceUnallocatedResult?.orgCount ?? '0');
        const mahatriaChoiceMaleOrgCount = parseInt(
          mahatriaChoiceUnallocatedResult?.maleOrganizationUserCount ?? '0',
        );
        const mahatriaChoiceFemaleOrgCount = parseInt(
          mahatriaChoiceUnallocatedResult?.femaleOrganizationUserCount ?? '0',
        );

        // Add to totals
        totalUnallocatedMaleCount += mahatriaChoice.maleCount;
        totalUnallocatedFemaleCount += mahatriaChoice.femaleCount;
        totalUnalocatedCount += mahatriaChoice.count;
        totalUnallocatedOrganisationCount += mahatriaChoice.organisationUserCount;
        totalUnallocatedMaleOrganisationCount += mahatriaChoiceMaleOrgCount;
        totalUnallocatedFemaleOrganisationCount += mahatriaChoiceFemaleOrgCount;
        totalMahatriaChoiceOrganisationUserCount = mahatriaChoice.organisationUserCount;

        for (const program of allPrograms) {
          // ✅ FIXED: Build allocated query WITHOUT search text
          const allocatedResult = await this.repository.getAllocatedMetrics(
            programId,
            program.id,
            parsedFilters,
            '',
          );
          const allocatedCount = parseInt(allocatedResult?.allocatedCount ?? '0');
          const allocatedMaleCount = parseInt(allocatedResult?.maleCount ?? '0');
          const allocatedFemaleCount = parseInt(allocatedResult?.femaleCount ?? '0');
          const allocatedOrgCount = parseInt(allocatedResult?.orgCount ?? '0');
          const allocatedMaleOrgCount = parseInt(allocatedResult?.maleOrganizationUserCount ?? '0');
          const allocatedFemaleOrgCount = parseInt(allocatedResult?.femaleOrganizationUserCount ?? '0');
          const roomMatePreferenceCount = parseInt(allocatedResult?.roomMatePreferenceCount ?? '0');

          // ✅ FIXED: Build non-allocated preferences query WITHOUT search text
          const nonAllocatedResult = await this.repository.getNonAllocatedMetrics(
            program.id,
            parsedFilters,
            '',
          );
          const nonAllocatedCount = parseInt(nonAllocatedResult?.count ?? '0');
          const nonAllocatedMaleCount = parseInt(nonAllocatedResult?.maleCount ?? '0');
          const nonAllocatedFemaleCount = parseInt(nonAllocatedResult?.femaleCount ?? '0');
          const nonAllocatedOrgCount = parseInt(nonAllocatedResult?.orgCount ?? '0');
          const nonAllocatedMaleOrgCount = parseInt(
            nonAllocatedResult?.maleOrganizationUserCount ?? '0',
          );
          const nonAllocatedFemaleOrgCount = parseInt(
            nonAllocatedResult?.femaleOrganizationUserCount ?? '0',
          );

          totalUnallocatedMaleCount += nonAllocatedMaleCount;
          totalUnallocatedFemaleCount += nonAllocatedFemaleCount;
          totalUnalocatedCount += nonAllocatedCount;
          totalUnallocatedOrganisationCount += nonAllocatedOrgCount;
          totalUnallocatedMaleOrganisationCount += nonAllocatedMaleOrgCount;
          totalUnallocatedFemaleOrganisationCount += nonAllocatedFemaleOrgCount;

          unallocatedPrograms.push({
            programId: program.id,
            programName: program.name,
            count: nonAllocatedCount,
            maleCount: nonAllocatedMaleCount,
            femaleCount: nonAllocatedFemaleCount,
            organisationUserCount: nonAllocatedOrgCount,
            maleOrganisationUserCount: nonAllocatedMaleOrgCount,
            femaleOrganisationUserCount: nonAllocatedFemaleOrgCount,
            totalSeatsCount: program.totalSeats || 0,
          });

          // Right side: Allocated programs
          allocatedPrograms.push({
            programId: program.id,
            programName: program.name,
            allocatedCount,
            maleCount: allocatedMaleCount,
            femaleCount: allocatedFemaleCount,
            organisationUserCount: allocatedOrgCount,
            totalSeatsCount: program.totalSeats || 0,
            maleOrganisationUserCount: allocatedMaleOrgCount || 0,
            femaleOrganisationUserCount: allocatedFemaleOrgCount || 0,
            roomMatePreferenceCount: roomMatePreferenceCount || 0
          });
        }

        allocated.programs = allocatedPrograms ?? [];
        unallocated.programs = unallocatedPrograms ?? [];
        unallocated.totalMaleCount = totalUnallocatedMaleCount;
        unallocated.totalFemaleCount = totalUnallocatedFemaleCount;
        unallocated.totalUnallocatedCount = totalUnalocatedCount;
        unallocated.totalMahatriaChoiceCount = mahatriaChoice.count;
        unallocated.totalOrganisationUserCount = totalUnallocatedOrganisationCount;
        unallocated.totalMaleOrganisationUserCount = totalUnallocatedMaleOrganisationCount;
        unallocated.totalFemaleOrganisationUserCount = totalUnallocatedFemaleOrganisationCount;
        unallocated.totalMahatriaChoiceOrganisationUserCount = totalMahatriaChoiceOrganisationUserCount;

        return {
          isGroupedProgram: true,
          primaryProgramId: programId,
          primaryProgramName: mainProgram.name,
          allocated,
          unallocated,
        };
      } else {
        // ✅ FIXED: Build total registrations query WITHOUT search text
        const totalRegistrations = await this.repository.getTotalRegistrations(
          programId,
          parsedFilters,
          '',
        );

        return {
          isGroupedProgram: false,
          totalRegistrations,
        };
      }
    } catch (error) {
      this.logger.error('Error generating Mahatria KPIs', error);
      throw error;
    }
  }

  private getMahatriaKPIFilterCondition(
    category: string,
    filter: string,
  ): Record<string, any> | null {
    switch (category) {
      case 'allocated':
        switch (filter) {
          case 'hold':
            return { approvalStatus: ApprovalStatusEnum.ON_HOLD };
          case 'reject':
            return { approvalStatus: ApprovalStatusEnum.REJECTED };
          default:
            // Handle program-specific filters like 'program_123'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return { allocatedProgramId: programId, approvalStatus: ApprovalStatusEnum.APPROVED };
            }
            return null;
        }

      case 'unallocated':
        switch (filter) {
          case 'male':
            return {
              gender: GenderEnum.MALE,
              approvalStatus: ApprovalStatusEnum.PENDING,
            };
          case 'female':
            return {
              gender: GenderEnum.FEMALE,
              approvalStatus: ApprovalStatusEnum.PENDING,
            };
          case 'total_unallocated':
            return {
              approvalStatus: ApprovalStatusEnum.PENDING,
              allocatedProgram: { id: IsNull() },
            };
          case 'mahatria_choice':
            return {
              approvalStatus: ApprovalStatusEnum.PENDING,
              // Assuming registrations without program preferences are "Mahatria Choice"
              preferences: {
                id: IsNull(),
              },
              allocatedProgram: {
                id: IsNull(),
              },
            };
          default:
            // Handle program-specific filters like 'program_123'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return {
                preferredProgramId: programId,
                approvalStatus: ApprovalStatusEnum.PENDING,
              };
            }
            return null;
        }

      case 'all':
        switch (filter) {
          case 'all':
            return {
              approvalStatus: [
                ApprovalStatusEnum.PENDING,
                ApprovalStatusEnum.ON_HOLD,
                ApprovalStatusEnum.REJECTED,
                ApprovalStatusEnum.APPROVED,
              ],
            };
          case 'male':
            return {
              gender: GenderEnum.MALE,
              approvalStatus: [
                ApprovalStatusEnum.PENDING,
                ApprovalStatusEnum.ON_HOLD,
                ApprovalStatusEnum.REJECTED,
                ApprovalStatusEnum.APPROVED,
              ],
            };
          case 'female':
            return {
              gender: GenderEnum.FEMALE,
              approvalStatus: [
                ApprovalStatusEnum.PENDING,
                ApprovalStatusEnum.ON_HOLD,
                ApprovalStatusEnum.REJECTED,
                ApprovalStatusEnum.APPROVED,
              ],
            };
          case 'mahatria-choice':
            return {
              mahatriaChoice: true
            }
          case 'swap-requests':
            return {
              swapRequests:[ SwapType.WantsSwap]
            }
          default:
            // Handle program-specific filters like 'program_123'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return {
                preferredProgramId: programId,
                approvalStatus: [
                  ApprovalStatusEnum.PENDING,
                  ApprovalStatusEnum.ON_HOLD,
                  ApprovalStatusEnum.REJECTED,
                ],
              };
            }
        }
      default:
        return null;
    }
  }

  /**
   * Fix the getKPIFilterCondition method to return proper values instead of TypeORM operators
   */
  private getKPIFilterCondition(category: string, filter: string): Record<string, any> | null {
    switch (category) {
      case 'registrations':
        switch (filter) {
          case 'all':
            return { registrationStatus: Not(In([RegistrationStatusEnum.DRAFT, RegistrationStatusEnum.SAVE_AS_DRAFT])) };
          case 'cancelled':
            return { registrationStatus: RegistrationStatusEnum.CANCELLED };
          case 'newSeekersPending':
            return {
              approvalStatus: In([ApprovalStatusEnum.PENDING]),
            };
          case 'rejected':
            return { approvalStatus: ApprovalStatusEnum.REJECTED };
          case 'onHold':
            return { approvalStatus: ApprovalStatusEnum.ON_HOLD };
          case 'approvedSeekers':
            return { approvalStatus: ApprovalStatusEnum.APPROVED };
          case 'rejectedSeekers':
            return { approvalStatus: ApprovalStatusEnum.REJECTED };
          default:
            return null;
        }

      case 'programs':
        if (filter.startsWith('program_')) {
          const programId = parseInt(filter.replace('program_', ''));
          return {
            approvalStatus: ApprovalStatusEnum.APPROVED,
            allocatedProgramId: programId,
          };
        }
        return null;

      case 'payments':
        switch (filter) {
          case 'all':
            return {
              paymentDetails: {
                paymentStatus: Not(PaymentStatusEnum.DRAFT),
              },
            };
          case 'paymentPending':
            return {
              paymentDetails: {
                paymentStatus: In([
                  PaymentStatusEnum.ONLINE_PENDING,
                  PaymentStatusEnum.OFFLINE_PENDING,
                ]),
              },
            };
          case 'paymentCompleted':
            return {
              paymentDetails: {
                paymentStatus: In([
                  PaymentStatusEnum.ONLINE_COMPLETED,
                  PaymentStatusEnum.OFFLINE_COMPLETED,
                ]),
              },
            };
          default:
            return null;
        }

      case 'invoices':
        switch (filter) {
          case 'all':
            return {
              invoiceDetails: {
                invoiceStatus: Not(InvoiceStatusEnum.DRAFT),
              },
            };
          case 'invoicePending':
            return {
              invoiceDetails: {
                invoiceStatus: InvoiceStatusEnum.INVOICE_PENDING,
              },
            };
          case 'invoiceCompleted':
            return {
              invoiceDetails: {
                invoiceStatus: InvoiceStatusEnum.INVOICE_COMPLETED,
              },
            };
          default:
            return null;
        }

      case 'travelAndLogistics':
        switch (filter) {
          case 'all':
            return {
              travelInfo: {
                travelInfoStatus: In([TravelStatusEnum.PENDING, TravelStatusEnum.COMPLETED]),
              },
            };
          case 'travelPending':
            return {
              travelInfo: {
                travelInfoStatus: TravelStatusEnum.PENDING,
              },
            };
          case 'travelCompleted':
            return {
              travelInfo: {
                travelInfoStatus: TravelStatusEnum.COMPLETED,
              },
            };
          case 'logisticsPending':
            return {
              travelPlans: {
                travelPlanStatus: TravelStatusEnum.PENDING,
              },
            };
          case 'logisticsCompleted':
            return {
              travelPlans: {
                travelPlanStatus: TravelStatusEnum.COMPLETED,
              },
            };
          default:
            return null;
        }

      default:
        return null;
    }
  }

  async getProgramDashboardData({
    programId,
    mainProgramDimensions = [],
    subProgramId,
    subProgramDimensions = [],
    rmId,
    userRole,
  }: {
    programId: number;
    mainProgramDimensions: string[];
    subProgramId?: number;
    subProgramDimensions?: string[];
    rmId?: number;
    userRole: string;
  }) {
    // this.logger.log(registrationMessages.GET_PROGRAM_DASHBOARD_DATA, { programId });
    try {
      const program = await this.repository.findProgramWithType(programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }
      const subPrograms = await this.repository.getSubPrograms(programId);
      const subProgramIds = subProgramId ? [subProgramId] : subPrograms.map((item) => item.id);
      const dimensions = subProgramId ? subProgramDimensions : mainProgramDimensions;
      let data: Array<{
        displayName: string;
        key: string;
        columnDefinition?: any;
        count?: number;
        values: {
          displayName: string;
          key: string;
          count: any;
          id?: number;
          filters?: {};
          order: number;
        }[];
      }> = [];

      const result = dimensions.map(async (dimension) => {
        const resultData = await this.repository.getRegistrationCountsOfDimension({
          programId: programId,
          dimension,
          subProgramId: subProgramId ? subProgramId : undefined,
          rmId: rmId ? rmId : undefined,
        });
        const ageGroups = [
        { key: 'greaterThanEqual0', displayName : '<=20', min: 0, max: 20, filters : {age : ['0-20']} },
        { key: 'greaterThanEqual20', displayName : '21 - 30', min: 21, max: 30, filters : {age : ['21-30']} },
        { key: 'greaterThanEqual30', displayName : '31 - 50', min: 31, max: 50, filters : {age : ['31-50']} },
        { key: 'greaterThanEqual50', displayName : '51 - 65', min: 51, max: 65, filters : {age : ['51-65']} },
        { key: 'greaterThanEqual65', displayName : '> 65', min: 66, max: 200, filters : {age : ['>65']} },
      ];
        switch (dimension) {
          case 'programRegistration':
            let programValues: any = [
              {
                displayName: 'Total registrations',
                key: 'totalRegistrations',
                count: Number(resultData?.totalRegistrations || 0),
                filters: {},
                order: 1,
              },
              {
                displayName: 'Unassigned',
                key: 'newRegistrations',
                count: Number(resultData?.newRegistrations || 0),
                filters: {
                  kpiCategory: 'registrations',
                  kpiFilter: 'newSeekersPending',
                },
                order: 2,
              },
              {
                displayName: 'Blessed seekers',
                key: 'blessedSeekers',
                count: Number(resultData?.blessedSeekers || 0),
                filters: {
                  kpiCategory: 'registrations',
                  kpiFilter: 'approvedSeekers',
                },
                order: 3,
              },
            ];
            if (userRole === 'rm') {
              const rmProgramValues = [
                {
                  displayName: 'Pending details',
                  key: 'pendingDetails',
                  count: Number(resultData?.pendingDetails || 0),
                  filters: {
                    kpiCategory: 'registrations',
                    kpiFilter: 'approvedSeekers',
                    approvalStatus: 'pending',
                    registrationStatus: 'pending',
                    order: 4,
                  },
                },
                {
                  displayName: 'Swap request',
                  key: 'swapRequest',
                  count: Number(resultData?.swapRequest || 0),
                  filters: {
                    swapRequests: ['can_shift', 'wants_swap'],
                    order: 5,
                  },
                },
              ];
              programValues = [...programValues, ...rmProgramValues];
            }
            if (
              [
                'admin',
                'viewer',
                'mahatria',
                'finance_manager',
                'relational_manager',
                'shoba',
                'operational_manger',
              ].includes(userRole)
            ) {
              programValues.push({
                displayName: 'Payment pending',
                key: 'paymentPending',
                count: Number(resultData?.paymentPending || 0),
                filters: {
                  paymentStatus: 'payment_pending',
                },
                order: 4,
              });
              if (userRole === 'finance_manager') {
                programValues.push({
                  displayName: 'Invoice pending',
                  key: 'invoicePending',
                  count: Number(resultData?.invoicePending || 0),
                  filters: {
                    invoiceStatus: 'invoice_pending',
                  },
                  order: 5,
                });
              } else if (
                [
                  'admin',
                  'viewer',
                  'mahatria',
                  'relational_manager',
                  'shoba',
                  'operational_manger',
                ].includes(userRole)
              ) {
                programValues.push({
                  displayName: 'Travel pending',
                  key: 'travelPending',
                  count: Number(resultData?.travelPending || 0),
                  filters: {
                    travelStatus: 'travel_pending',
                  },
                  order: 6,
                });
              }
            }

            return {
              displayName: 'Program registrations',
              key: 'programRegistrations',
              values: programValues,
              count: programValues.reduce((sum, item) =>
                sum + Number(item.count || 0), 0),
            };
          
          
          case 'rating':
            let ratingColumnDefinition = [
              {
                displayName: 'User profile',
                key: 'userProfileUrl',
              },
              {
                displayName: 'Full name',
                key: 'fullName',
              },
              {
                displayName: 'No Of HDBs',
                key: 'noOfHdbs',
              }
            ];
            let ratingValues = resultData?.unratedRegistrations || [];
            return {
              displayName: 'Yet to review',
              key: 'ratings',
              columnDefinition: ratingColumnDefinition,
              values: ratingValues,
              filters: {"rmRating":["unrated"]},
              count: Number(resultData?.totalUnRatedRegistrations) || 0,
            };

          case 'programBlessedSeekers':
            let blessedSeekersValues = subProgramIds.map((item) => {
            const subProgram = subPrograms.filter((sp) => sp.id === item);
            const filteredItems = (resultData.filter(r => r.id == item))
              const itemData = filteredItems.length > 0 ? filteredItems[0] : {
                programId: item,
                programName: null,
                count: 0,
              };
              return {
                id: subProgram[0].id,
                displayName: subProgram[0].name,
                key: `program_${itemData.id}`,
                count: Number(itemData.count) || 0,
                filters: {
                  kpiCategory: 'programs',
                  kpiFilter: `program_${itemData.id}`,
                },
              };
            });
            const totalBlessedSeekers = blessedSeekersValues.reduce(
              (sum, item) => sum + Number(item.count),
              0,
            );
            return {
              displayName: 'Blessed Seekers - Snapshot',
              key: 'blessedSeekers',
              values: blessedSeekersValues,
              count: totalBlessedSeekers,
            };

          case 'programApprovalStatus':
            const registrationStatuses = [
              {
                displayName: 'Blessed',
                key: 'blessed',
                count: 0,
                approvalStatus: 'approved',
                filters: { kpiCategory: 'registrations', kpiFilter: 'approvedSeekers' },
              },
              {
                displayName: 'Yet to assign',
                key: 'yetToAssign',
                count: 0,
                approvalStatus: 'pending',
                filters: { kpiCategory: 'registrations', kpiFilter: 'newSeekersPending' },
              },
              {
                displayName: 'On hold',
                key: 'onHold',
                count: 0,
                approvalStatus: 'rejected',
                filters: { kpiCategory: 'registrations', kpiFilter: 'rejected' },
              },
              {
                displayName: 'Yet to decide',
                key: 'yetToDecide',
                count: 0,
                approvalStatus: 'on_hold',
                filters: { kpiCategory: 'registrations', kpiFilter: 'onHold' },
              },
              {
                displayName: 'Swap request',
                key: 'swapRequest',
                count: 0,
                approvalStatus: 'swap_request',
                filters: { kpiCategory: 'registrations', kpiFilter: 'swap_request' },
              },
            ];

            let values = registrationStatuses.map((item) => {
              const filterItem = resultData.find(
                (status) =>
                  status.approvalStatus.toLowerCase() === item.approvalStatus.toLowerCase(),
              );
              return {
                displayName: item.displayName,
                key: item.key,
                count: Number(filterItem?.count) || 0,
                filters: item.filters,
              };
            });
            const totalSeekers = values.reduce((sum, item) => sum + Number(item.count), 0);

           return {
             displayName: 'Registrations - Snapshot',
             key: 'programApprovalStatus',
             values: values,
             count: totalSeekers,
           };
          
          case  'detailedData':
            let detailedDataColumnDefinition  = [
                  {
                      displayName: 'HDBs/MSDs',
                      key: 'hdbsMsds'
                  },
                  {
                      displayName: '1st Timer',
                      key: 'firstTimers'
                  },
                  {
                      displayName: '1 - 10 HDBs/MSDs',
                      key: 'oneToTenHdbsMsds'
                  },
                  {
                      displayName: '>10 HDBs/MSDs',
                      key: 'greaterThanTenHdbsMsds'
                  },
                  {
                      displayName: 'Organisation',
                      key: 'organisation'
                  },
                  {
                      displayName: 'Total',
                      key: 'totalSeekers'
                  },
                  {
                      displayName: 'Male',
                      key: 'male'
                  },
                  {
                      displayName: 'Female',
                      key: 'female'
                  }

              ]
            const detailedDataValues = subProgramIds.map(subProgramId => {
              const subProgram = subPrograms.filter((sp) => sp.id === subProgramId);
              const filteredItems = resultData.filter((r) => r.programId == subProgramId);
              const item =
                filteredItems.length > 0
                  ? filteredItems[0]
                  : {
                      programId: subProgramId,
                      hdbsMsds: subProgram[0].name,
                      totalSeekers: 0,
                      firstTimers: 0,
                      oneToTenHdbsMsds: 0,
                      greaterThanTenHdbsMsds: 0,
                      organisation: 0,
                      male: 0,
                      female: 0,
                    };
              return {
                id: item.programId,
                hdbsMsds: subProgram[0].name,
                totalSeekers: Number(item.totalSeekers) || 0,
                firstTimers: Number(item.firstTimers) || 0,
                oneToTenHdbsMsds: Number(item.oneToTenHdbsMsds) || 0,
                greaterThanTenHdbsMsds: Number(item.greaterThanTenHdbsMsds) || 0,
                organisation: Number(item.organisation) || 0,
                male: Number(item.male) || 0,
                female: Number(item.female) || 0,
                filters: {
                  totalSeekers: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}` },
                  firstTimers: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    numberOfHdbs: '0',
                  },
                  oneToTenHdbsMsds: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    numberOfHdbs: '1-10',
                  },
                  greaterThanTenHdbsMsds: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    numberOfHdbs: '>10',
                  },
                  organisation: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    organisation: 'yes'
                  },
                  male: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`, 
                    gender: ['male'],
                  },
                  female: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,   
                    gender: ['female']
                  }
                },
              };
            })
              return {
                displayName: "Blessed seekers by no. of HDBs",
                key: "detailedData",
                columnDefinition: detailedDataColumnDefinition,
                values: detailedDataValues,
                count: detailedDataValues.reduce(
                  (sum, item) => sum + Number(item.totalSeekers || 0),
                  0,
                ),
              };
          
          case 'demographics':
            const ageGenderGroupCounts = ageGroups.map((group) => {
              const genders = ['male', 'female'];
              const genderCounts = genders.map((gender) => {
                const count = resultData
                  .filter((row) => {
                    const age = Number(row.age);
                    return (
                      age >= group.min && age <= group.max && row.gender?.toLowerCase() === gender
                    );
                  })
                  .reduce((sum, row) => sum + Number(row.count), 0);
                return { gender, count };
              });
              return {
                key: group.key,
                displayName: group.displayName,
                genders: {
                  [genderCounts[0].gender.toLowerCase()]: genderCounts[0].count,
                  [genderCounts[1].gender.toLowerCase()]: genderCounts[1].count,
                },
                filters: group.filters,
              };
            });
            const demographicsValues = ageGenderGroupCounts.map((item) => {
              const subProgramFilters = subProgramId
                ? { kpiCategory: 'programs', kpiFilter: `program_${subProgramId}` }
                : {};
              return {
                displayName: item.displayName,
                key: item.key,
                values: [
                  {
                    displayName: 'Male',
                    key: 'male',
                    count: item.genders.male || 0,
                    filters: {
                      ...item.filters,
                      gender: ['male'],
                      ...subProgramFilters,
                    },
                  },
                  {
                    displayName: 'Female',
                    key: 'female',
                    count: item.genders.female || 0,
                    filters: {
                      ...item.filters,
                      gender: ['female'],
                      ...subProgramFilters,
                    },
                  },
                ],
              };
            });
            return {
              displayName: subProgramId ? 'Demographics': 'All Seekers - Demographics' ,
              key: 'demographics',
              values: demographicsValues,
              count: demographicsValues.reduce((sum, item) => {
                const male = item.values.find(v => v.key === 'male');
                const female = item.values.find(v => v.key === 'female');
                return sum + Number(male?.count || 0) + Number(female?.count || 0);
              }, 0),
            };

          case 'preferredVsBlessed':
            const preferredVsBlessedValues = resultData.map((item) => {
              return {
                id: item.programId,
                displayName: item.displayName,
                key: item.displayName.toLowerCase().replace(/\s+/g, '_'),
                values: [
                  {
                    displayName: 'Total preferences',
                    key: 'totalPreferences',
                    count: Number(item.totalPreferences) || 0,
                    filters: {
                      kpiCategory: 'programs',
                      kpiFilter: `program_${item.programId}`,
                      preferredProgram: [item.programId],
                    },
                  },
                  {
                    displayName: 'Total blessed seekers',
                    key: 'totalBlessedSeekers',
                    count: Number(item.totalBlessedSeekers) || 0,
                    filters: {
                      kpiCategory: 'programs',
                      kpiFilter: `program_${item.programId}`,
                    },
                  },
                ],
              };
            });

            return {
              displayName: 'Preference against blessings',
              key: 'preferredVsBlessed',
              values: preferredVsBlessedValues,
              count: this.calculateNestedCount(preferredVsBlessedValues),
            };

          case 'payment':
            const onlinePending = resultData.find(
              (item) => item.paymentStatus === 'online_pending',
            );
            const onlineCompleted = resultData.find(
              (item) => item.paymentStatus === 'online_completed',
            );
            const onlineFailed = resultData.find((item) => item.paymentStatus === 'online_failed');
            const offlinePending = resultData.find(
              (item) => item.paymentStatus === 'offline_pending',
            );
            const offlineCompleted = resultData.find(
              (item) => item.paymentStatus === 'offline_completed',
            );
            const offlineFailed = resultData.find(
              (item) => item.paymentStatus === 'offline_failed',
            );
            const paymentValues = [
              {
                displayName: 'Completed',
                key: 'completed',
                values: [
                  {
                    displayName: 'Online',
                    key: 'online',
                    count: onlineCompleted ? Number(onlineCompleted?.count) : 0,
                    filters : {
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                      paymentStatus : ['payment_completed'],
                      paymentMode : ['online']
                    }
                  },
                  {
                    displayName: 'Offline',
                    key: 'offline',
                    count: offlineCompleted ? Number(offlineCompleted?.count) : 0,
                    filters : {
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                      paymentStatus : ['payment_completed'],
                      paymentMode : ['offline']
                    }
                  },
                ],
                count:
                  (onlineCompleted ? Number(onlineCompleted?.count) : 0) +
                  (offlineCompleted ? Number(offlineCompleted?.count) : 0),
              },
              {
                displayName: 'Pending',
                key: 'pending',
                values: [
                  {
                    displayName: 'Online',
                    key: 'online',
                    count: onlinePending ? Number(onlinePending?.count) : 0,
                    filters : {
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                      paymentStatus : ['payment_pending'],
                      paymentMode : ['online']
                    }
                  },
                  {
                    displayName: 'Offline',
                    key: 'offline',
                    count: offlinePending ? Number(offlinePending?.count) : 0,
                    filters: {
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                      paymentStatus : ['payment_pending'],
                      paymentMode : ['offline']
                    }
                  },
                ],
                count:
                  (onlinePending ? Number(onlinePending?.count) : 0) +
                  (offlinePending ? Number(offlinePending?.count) : 0),
              },
              {
                displayName: 'Failed',
                key: 'failed',
                values: [
                  {
                    displayName: 'Online',
                    key: 'online',
                    count: onlineFailed ? Number(onlineFailed?.count) : 0,
                  },
                  {
                    displayName: 'Offline',
                    key: 'offline',
                    count: offlineFailed ? Number(offlineFailed?.count) : 0,
                  },
                ],
                count:
                  (onlineFailed ? Number(onlineFailed?.count) : 0) +
                  (offlineFailed ? Number(offlineFailed?.count) : 0),
              },
            ];

            return {
              displayName: 'Payment status',
              key: 'paymentStatus',
              values: paymentValues,
              count: paymentValues.reduce((sum, item) => {
                return sum + Number(item?.count);
              }, 0),
            };
          
          case 'invoice':
            const invoiceCompleted = resultData?.invoiceStatusCounts.find(
              (item) => item.invoiceStatus === 'invoice_completed',
            );
            const invoicePending = resultData?.invoiceStatusCounts.find(
              (item) => ['invoice_pending', 'draft'].includes(item.invoiceStatus),
            );
            const programCompletedFilters = {
              kpiCategory: 'registrations',
              kpiFilter: 'approvedSeekers',
              invoiceStatus: ['invoice_completed'],
            };
            const subProgramCompletedFilters = {
              kpiCategory: 'programs',
              kpiFilter: `program_${subProgramId}`,
              invoiceStatus: ['invoice_completed'],
            };
            const programPendingFilters = {
              kpiCategory: 'registrations',
              kpiFilter: 'approvedSeekers',
              invoiceStatus: ['invoice_pending'],
            };
            const subProgramPendingFilters = {
              kpiCategory: 'programs',
              kpiFilter: `program_${subProgramId}`,
              invoiceStatus: ['invoice_pending'],
            };
            const invoiceValues = [
              {
                displayName: 'Completed',
                key: 'completed',
                count: invoiceCompleted ? Number(invoiceCompleted?.count) : 0,
                filters: subProgramId ? subProgramCompletedFilters : programCompletedFilters,
              },
              {
                displayName: 'Pending',
                key: 'pending',
                count: invoicePending ? Number(invoicePending?.count) : 0,
                filters: subProgramId ? subProgramPendingFilters : programPendingFilters,
              },
            ];

            return {
              displayName: 'Invoice generated',
              key: 'invoiceStatus',
              values: invoiceValues,
              count: resultData?.totalInvoiceCount
            };
          
          case 'travelPlan':
            const flightCount = resultData.find(
              (item) => item.travelType === TravelTypeEnum.FLIGHT,
            );
            const ownTransportCount = resultData.find(
              (item) => item.travelType === TravelTypeEnum.OWN_TRANSPORT,
            );
            const cityPickupCount = resultData.find(
              (item) => item.travelType === TravelTypeEnum.CITY_PICK_UP,
            );

            const travelValues = [
              {
                displayName: 'Flight',
                key: 'flight',
                count: flightCount ? Number(flightCount?.count) : 0,
              },
              {
                displayName: 'Own transport',
                key: 'ownTransport',
                count: ownTransportCount ? Number(ownTransportCount?.count) : 0,
              },
              {
                displayName: 'City pickup',
                key: 'cityPickup',
                count: cityPickupCount ? Number(cityPickupCount?.count) : 0,
              },
            ];
            return {
              displayName: 'Completed travel plan',
              key: 'travelPlan',
              values: travelValues,
              count: travelValues.reduce((sum, item) => sum + Number(item.count), 0),
            };
          
          case 'travelStatus':
            const travelStatusValues = [
              {
                displayName: 'Completed',
                key: 'completed',
                count: resultData?.completedCount ? Number(resultData?.completedCount) : 0,
                filters: {
                  kpiCategory: 'programs',
                  kpiFilter: `program_${subProgramId}`,
                  travelStatus: 'travel_completed',
                },
              },
              {
                displayName: 'Pending',
                key: 'pending',
                count: resultData?.pendingCount ? Number(resultData?.pendingCount) : 0,
                filters: {
                  kpiCategory: 'programs',
                  kpiFilter: `program_${subProgramId}`,
                  travelStatus: 'travel_pending',
                },
              },
            ];

            return {
              displayName: 'Travel status',
              key: 'travelStatus',
              values: travelStatusValues,
              count: resultData?.totalRegistrations ? Number(resultData?.totalRegistrations) : 0,
            };
          
          case 'location':
          const cities = [...MajorCities, 'Others'];
            const locationValues = cities.map((city) => {
              const cityData = resultData.find((item) => item.location === city);
              return {
                displayName: city,
                key: city,
                count: cityData ? Number(cityData.count) : 0,
                filters: {
                  location: [city],
                },
              };
            });
            return {
              displayName: 'Location insights',
              key: 'location',
              values: locationValues,
              count: locationValues.reduce((sum, item) => sum + Number(item.count), 0),
            };
          default:
            console.warn(`Unknown dimension: ${dimension}`);
        }
      });

      const settledResults = await Promise.allSettled(result);
      data = settledResults
        .filter((res) => res.status === 'fulfilled' && res.value)
        .map((res) => (res as PromiseFulfilledResult<any>).value);

      if (subProgramId) {
        const paymentStatus = data.find(d => d.key === 'paymentStatus');
        const invoiceStatus = data.find(d => d.key === 'invoiceStatus');
        const travelStatus = data.find(d => d.key === 'travelStatus');
        const programRegistrations = data.find(d => d.key === 'programRegistrations');
      
        // Get total seekers count
        const totalSeekers = programRegistrations?.values?.find(v => v.key === 'totalRegistrations')?.count || 0;
      
        // Get completed counts
        const paymentCompleted = paymentStatus?.values?.find(v => v.key === 'completed')?.count || 0;
        const invoiceCompleted = invoiceStatus?.values?.find(v => v.key === 'completed')?.count || 0;
        const travelCompleted = travelStatus?.values?.find(v => v.key === 'completed')?.count || 0;
      
        // Calculate percentages (avoid division by zero)
        const paymentPercent = totalSeekers > 0 ? Math.round((paymentCompleted / totalSeekers) * 100) : 0;
        const invoicePercent = paymentCompleted > 0 ? Math.round((invoiceCompleted / paymentCompleted) * 100) : 0;
        const travelPercent = totalSeekers > 0 ? Math.round((travelCompleted / totalSeekers) * 100) : 0;
      
        var subProgramValues = {
          displayName: 'Sub program',
          key: 'subProgramRegistrations',
          values: [
            {
              displayName: 'Total seekers',
              key: 'totalSeekers',
              count: totalSeekers,
              order: 1,
            },
            {
              displayName: `Payment (${paymentPercent}% completed)`,
              key: 'paymentCompleted',
              count: paymentCompleted,
              totalCount: totalSeekers,
              percentCompleted: paymentPercent,
              order: 2,
            },
            {
              displayName: `Invoice generated (${invoicePercent}% completed)`,
              key: 'invoiceGenerated',
              count: invoiceCompleted,
              totalCount: paymentCompleted,
              percentCompleted: invoicePercent,
              order: 3,
            },
            {
              displayName: `Travel (${travelPercent}% completed)`,
              key: 'travelCompleted',
              count: travelCompleted,
              totalCount: totalSeekers,
              percentCompleted: travelPercent,
              order: 4,
            },
          ],
        };
        data = [subProgramValues, ...data];
      }
      return data;
    } catch (error) {
      console.error('Error fetching program dashboard data:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_DASHBOARD_DATA_GET_FAILED, error);
    }
  }

  public formatPaymentStatus(status: string): string {
    switch (status) {
      case 'online_pending':
        return 'Online Pending';
      case 'offline_pending':
        return 'Offline Pending';
      case 'online_completed':
        return 'Online Completed';
      case 'offline_completed':
        return 'Offline Completed';
      case 'draft':
        return 'Draft';
      case 'save_as_draft':
        return '-';
      case 'payment_pending':
        return 'Pending';
      default:
        return status;
    }
  }

  public formatInvoiceStatus(status: string): string {
    switch (status) {
      case 'invoice_pending':
        return 'Pending';
      case 'invoice_completed':
        return 'Completed';
      case 'draft':
        return 'Draft';
      case 'save_as_draft':
        return '-';
      default:
        return status;
    }
  }

  public formatTravelStatus(status: string): string {
    switch (status) { 
      case 'travel_pending':
        return 'Pending';
      case 'travel_completed':
        return 'Completed';
      case 'pending':
        return 'Pending';
      case 'completed':
        return 'Completed';
      case 'save_as_draft':
        return '-';
      case 'draft':
        return 'Draft';
      default:
        return status;
    }
  }

  async remove(id: number, user: User) {
    this.logger.log(registrationMessages.DELETE_REQUEST_RECEIVED, { id });
    try {
      await this.repository.softDeleteRegistration(id, user.id);
      return { registrationId: id };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    }
  }

  /**
   * Calculate the total count from a single level of nested KPI structures.
   * Each item contains a `values` array where the actual counts reside.
   * @param items - Array of KPI items with nested `values`
   * @returns Aggregated total count of all nested values
   */
  private calculateNestedCount(items: any[]): number {
    if (!Array.isArray(items)) {
      return 0;
    }

    return items.reduce((sum, item) => {
      const nestedValues = Array.isArray(item.values)
        ? item.values.reduce(
            (nestedSum, nestedItem) =>
              nestedSum + Number(nestedItem.count || 0),
            0,
          )
        : 0;

      return sum + nestedValues;
    }, 0);
  }

  /**
   * 
   * @param string - The input string to capitalize
   * @description Capitalizes the first letter of the given string.
   * If the string is empty or not a string, it returns an empty string.
   * If the string is only whitespace, it returns an empty string.
   * @returns The string with the first letter capitalized
   */
  public capitalizeFirstLetter(string: string): string {
    if (!string || typeof string !== 'string') return '';
    const trimmed = string.trim();
    if (trimmed === '') return '';
    return string.charAt(0).toUpperCase() + string.slice(1);
  }

  /**
   * Capitalizes the first letter of every word in a string
   * @param str - The input string to capitalize
   * @returns The string with first letter of each word capitalized
   */
  public capitalizeWords(str: string): string {
    if (!str || typeof str !== 'string') return '';
    const trimmed = str.trim();
    if (trimmed === '') return '';
    return str
      .split(' ')
      .map((word) => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
      .join(' ');
  }

  /**
   * Store or update question answers for a registration using existing repository logic
   * @param queryRunner - Active query runner managing the transaction
   * @param registrationId - Program registration ID
   * @param answers - Array of question/answer mappings
   * @param userId - User performing the update
   */
  public async storeQuestionAnswers(
    queryRunner: any,
    registrationId: number,
    answers: RegistrationAnswerDto[],
    userId: number,
  ): Promise<void> {
    const qaMappings = answers.map((a) => ({
      questionId: a.questionId,
      answerValue: a.answer,
    }));
    
    // fetch the program from the registration and check the travel info and travel plan for the registration if travel any of them is complete then make the registrationstatus as comepletd
    const registration = await this.repository.findRegistrationById(registrationId);
    let travelCompleted = false;
    const travelRequired = registration?.program?.involvesTravel || false;

    if (registration && travelRequired) {
      if (
        Array.isArray(registration.travelInfo) &&
        registration.travelInfo.some((info) => info.travelInfoStatus === TravelStatusEnum.COMPLETED)
      ) {
        travelCompleted = true;
        registration.registrationStatus = RegistrationStatusEnum.COMPLETED;
      } else if (
        Array.isArray(registration.travelPlans) &&
        registration.travelPlans.some((plan) => plan.travelPlanStatus === TravelStatusEnum.COMPLETED)
      ) {
        travelCompleted = true;
        registration.registrationStatus = RegistrationStatusEnum.COMPLETED;
      }
    } else if (registration) {
      travelCompleted = true;
      registration.registrationStatus = RegistrationStatusEnum.COMPLETED;
    }

    await this.repository.upsertQuestionAnswers(
      queryRunner,
      registrationId,
      qaMappings,
      userId,
    );
    this.logger.log(
      `Stored question answers for registration ${registrationId}: ${JSON.stringify(qaMappings)}`
    );

    if (travelCompleted) {
      await this.repository.updateRegistrationStatus(
        queryRunner,
        registrationId,
        registration.registrationStatus,
        userId,
      );
    }
  }

  /**
   * Get save-as-draft registrations with optional search and filters
   */
  async getSaveDraftRegistrations(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
    sortKey: string = 'id',
    sortOrder: 'ASC' | 'DESC' = 'ASC',
  ) {
    try {
      const isRMUser = userRoles.includes('relational_manager');
      return await this.repository.findSaveDraftRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText,
        parsedFilters,
        isRMUser ? filterRegistrationsByUserId : undefined,
        sortKey,
        sortOrder,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }
  registrationEditLink(programId:Number, userId:number | undefined): string {
    var url = `${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programId}${SEEKER_FE_EDIT_REGISTRATION_PATH}`;
    if (userId) {
      url += `&userId=${userId}`;
    }
    return url;
  }
}