import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, IsNull, Not, DataSource, ILike, EntityManager, In, MoreThanOrEqual, MoreThan, LessThan, LessThanOrEqual, Equal, FindOperator, Between } from 'typeorm';
import { CreateRegistrationDto } from './dto/create-registration.dto';
import {
  Program,
  ProgramSession,
  ProgramQuestion,
  Question,
  ProgramRegistration,
  RegistrationApproval,
  RegistrationCustomResponse,
  RegistrationQuestionAnswer,
  ProgramRegistrationFailure,
  FormSection,
  RegistrationTravelInfo,
  RegistrationTravelPlan,
  RegistrationInvoiceDetail,
  RegistrationPaymentDetail,
  Preference,
  ProgramType,
  ProgramRegistrationSwap,
  ProgramRegistrationRmRating,
} from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { RegistrationBasicStatusEnum } from 'src/common/enum/registration-basic-status.enum';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { decrementSeatCounts, incrementSeatCounts } from '../common/utils/seat-count.util';
import { TravelStatusEnum } from 'src/common/enum/travel-status.enum';
import { PreferenceService } from 'src/preference/preference.service';
import { UpdatePreferenceDto } from 'src/preference/dto/update-preference.dto';
import { GenderEnum } from 'src/common/enum/gender.enum';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { toSnakeCase } from 'src/common/utils/common.util';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { MajorCities } from 'src/common/constants/constants';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';

@Injectable()
export class RegistrationRepository {
  constructor(
    @InjectRepository(Program) private readonly programRepo: Repository<Program>,
    @InjectRepository(ProgramSession) private readonly sessionRepo: Repository<ProgramSession>,
    @InjectRepository(ProgramQuestion) private readonly programQuestionRepo: Repository<ProgramQuestion>,
    @InjectRepository(Question) private readonly questionRepo: Repository<Question>,
    @InjectRepository(ProgramRegistration) private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(RegistrationApproval) private readonly approvalRepo: Repository<RegistrationApproval>,
    @InjectRepository(RegistrationCustomResponse) private readonly customResponseRepo: Repository<RegistrationCustomResponse>,
    @InjectRepository(RegistrationQuestionAnswer) private readonly questionAnswerRepo: Repository<RegistrationQuestionAnswer>,
    @InjectRepository(ProgramRegistrationFailure) private readonly failureRepo: Repository<ProgramRegistrationFailure>,
    @InjectRepository(Preference) private readonly preferenceRepo: Repository<Preference>,
    @InjectRepository(ProgramRegistrationSwap) private readonly swapRepo: Repository<ProgramRegistrationSwap>,
    @InjectRepository(ProgramRegistrationRmRating) private readonly rmRatingRepo: Repository<ProgramRegistrationRmRating>,
    @InjectRepository(RegistrationTravelInfo) private readonly travelInfoRepo: Repository<RegistrationTravelInfo>,
    @InjectRepository(RegistrationTravelPlan) private readonly travelPlanRepo: Repository<RegistrationTravelPlan>,
    private readonly preferenceService: PreferenceService,
    private readonly dataSource: DataSource,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
    private readonly programRegistrationRepository: ProgramRegistrationRepository,
  ) {}

  /**
   * Get registration counts for each sub-program of a given program.
   * @param programId - The primary program ID.
   * @returns Array of objects: [{ programId, programName, registrationCount }]
   */
  async getSubProgramRegistrationCounts(
    programId: number,
  ): Promise<{ programId: number; programName: string; registrationCount: number }[]> {
    // Use a single query to get counts for all sub-programs
    const query = this.programRepo
      .createQueryBuilder('sub_program')
      .select('sub_program.id', 'programId')
      .leftJoin(
        'program_registrations',
        'registration',
        'registration.program_id = sub_program.id AND registration.deleted_at IS NULL',
      )
      .addSelect('sub_program.name', 'programName')
      .addSelect('COUNT(registration.id)', 'regisstrationCount')
      .where('sub_program.primary_program_id = :programId', { programId })
      .andWhere('sub_program.deleted_at IS NULL')
      .groupBy('sub_program.id')
      .addGroupBy('sub_program.name');

    const result = await query.getRawMany();

    return result.map((row) => ({
      programId: Number(row.programId),
      programName: row.programName,
      registrationCount: Number(row.registrationCount),
    }));
  }

/**
 * Get registration counts for a specific dimension
 * @param programId The ID of the program
 * @param dimension The dimension to get registration counts for
 * @param subProgramId The ID of the sub-program (optional)
 * @returns The registration counts for the specified dimension
 */
  async getRegistrationCountsOfDimension({programId, dimension, subProgramId, rmId}: {programId: number, dimension: string, subProgramId?: number, rmId?: number}){
    try{

      const subProgramsIds = subProgramId ? [subProgramId] : (await this.getSubPrograms(programId)).map(sp => sp.id);

      switch (dimension) {
        case 'programRegistration':
         const registrationQuery = this.registrationRepo
         .createQueryBuilder('registration')
         .innerJoin('hdb_registration_approval', 'approval', 'approval.registration_id = registration.id')
         .select([
          `COUNT(DISTINCT CASE WHEN approval.approval_status = 'approved' THEN registration.id END) AS "blessedSeekers"`,
          `COUNT(DISTINCT CASE WHEN approval.approval_status = 'approved' AND registration.registration_status = 'pending' THEN registration.id END) AS "pendingDetails"`,
          `COUNT(DISTINCT registration.id) AS "totalRegistrations"`,
          `COUNT(DISTINCT CASE WHEN approval.approval_status = 'pending' THEN registration.id END) AS "newRegistrations"`,          
        ])
        .addSelect(`COUNT(DISTINCT CASE WHEN (payment.payment_status IN ('online_pending', 'offline_pending')) OR (payment.id IS NULL AND registration.allocated_program_id IS NOT NULL) THEN registration.id END) AS "paymentPending"`,)
        .addSelect(`COUNT(DISTINCT CASE WHEN (travelInfo.travel_info_status != 'completed' OR travelPlan.travel_plan_status != 'completed' OR travelInfo.travel_info_status IS NULL OR travelPlan.travel_plan_status IS NULL) AND registration.allocated_program_id IS NOT NULL THEN registration.id END)`, 'travelPending')
        .addSelect(`COUNT(DISTINCT CASE WHEN invoice.invoice_status = 'invoice_pending' THEN registration.id END)`, 'invoicePending') 
        .leftJoin('registration.paymentDetails', 'payment')
        .leftJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registrationId = registration.id')
        .leftJoin('hdb_registration_travel_info', 'travelInfo', 'travelInfo.registrationId = registration.id')
        .leftJoin('registration.invoiceDetails', 'invoice')
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT })
        .andWhere('travelInfo.deleted_at IS NULL')
        .andWhere('travelPlan.deleted_at IS NULL')
        .andWhere('registration.deleted_at IS NULL');
        if (subProgramId) {
          registrationQuery.andWhere('registration.allocated_program_id = :subProgramId', { subProgramId });
        }
        if(rmId){
          registrationQuery.andWhere('registration.rm_contact = :rmId', { rmId });
        }
        const registrationCounts = await registrationQuery.getRawOne()

        let swapRequestQuery = this.swapRepo
        .createQueryBuilder('swapRequest')
        .select('COUNT(swapRequest.id)', 'totalSwapRequests')
        .leftJoin('hdb_program_registration', 'registration', 'registration.id = swapRequest.program_registration_id')
        .andWhere('registration.deleted_at IS NULL')

        if(subProgramId){
          swapRequestQuery.where('registration.allocated_program_id = :subProgramId', { subProgramId })
          .andWhere('swapRequest.status = :status', { status: 'active' });
         } else {      
            swapRequestQuery.where('registration.program_id = :programId', { programId })
            .andWhere('swapRequest.status = :status', { status: 'active' })
        }
    
        if(rmId){
          swapRequestQuery.andWhere('registration.rm_contact = :rmId', { rmId });
        }
        const swapRequestCounts = await swapRequestQuery.getRawOne();
        registrationCounts.swapRequest = swapRequestCounts.totalSwapRequests;
        return registrationCounts;

        case 'rating':
       let ratingQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('sub_registration.full_name', 'fullName')
          .addSelect('sub_registration.id', 'programRegistrationId')
          .addSelect('sub_registration.no_of_hdbs', 'noOfHdbs')
          .addSelect('sub_registration.user_profile_url', 'userProfileUrl')
          .addSelect('sub_registration.gender', 'gender')
          .addSelect('sub_registration.city', 'city')
          .addSelect('sub_registration.created_at', 'createdAt')
          .addSelect('EXTRACT(YEAR FROM AGE(sub_registration.dob))', 'age')
          .leftJoin('sub_registration.ratings', 'rating')
          .where('sub_registration.program_id = :programId', { programId })
          .andWhere('sub_registration.deleted_at IS NULL')
          .andWhere('rating.id IS NULL')

          if(rmId){
            ratingQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const totalUnRatedRegistrations = await ratingQuery.getCount();
          ratingQuery.take(5);
          const unratedRegistrations = await ratingQuery.getRawMany();
          return { unratedRegistrations, totalUnRatedRegistrations }
        
        case 'programBlessedSeekers':
          const subProgramQuery = this.programRepo
          .createQueryBuilder('sub_program')
          .select('sub_program.name', 'displayName')
          .addSelect('sub_program.id', 'id')
          .addSelect('COUNT(DISTINCT registration.id)', 'count')
          .leftJoin('hdb_program_registration', 'registration', 'registration.allocated_program_id = sub_program.id')
          .where('sub_program.primary_program_id = :programId', { programId })
          .andWhere('sub_program.deleted_at IS NULL')
          .andWhere('registration.deleted_at IS NULL')
          .groupBy('sub_program.id')
          .addGroupBy('sub_program.name')

          if(rmId){
            subProgramQuery.andWhere('registration.rm_contact = :rmId', { rmId });
          }
          const subProgramCounts = await subProgramQuery.getRawMany();
          return subProgramCounts;

        case 'programApprovalStatus':
          const programRegistrationStatusQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('approval.approval_status', 'approvalStatus')
          .innerJoin('hdb_registration_approval', 'approval', 'approval.registration_id = sub_registration.id')
          .where('sub_registration.program_id = :programId', { programId: programId })
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('approval.approval_status')

          if(rmId){
            programRegistrationStatusQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const programRegistrationStatusCounts = await programRegistrationStatusQuery.getRawMany()
        return programRegistrationStatusCounts;
        
        case 'preferredVsBlessed':

        const preferencesQuery = this.programRepo
          .createQueryBuilder('sub_program')
          .select('sub_program.name', 'displayName')
          .addSelect('sub_program.id', 'programId')
          .addSelect('COUNT(DISTINCT preference.id)', 'totalPreferences')
          .addSelect('COUNT(CASE WHEN preference.preferred_program_id = registration.allocated_program_id THEN registration.id END)', 'totalBlessedSeekers')
          .innerJoin('hdb_preference', 'preference', 'preference.preferred_program_id = sub_program.id AND preference.priority_order = 1')
          .innerJoin('hdb_program_registration', 'registration', 'registration.id = preference.registration_id')
          .where('sub_program.primary_program_id = :programId', { programId })
          .andWhere('sub_program.deleted_at IS NULL')
          .andWhere('registration.deleted_at IS NULL')
          .groupBy('sub_program.id')
          .addGroupBy('sub_program.name')
        const preferencesCounts = await preferencesQuery.getRawMany();
        return preferencesCounts;

        case 'detailedData':
        const detailedDataQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('sub_program.name', 'hdbsMsds')
          .addSelect('sub_program.id', 'programId')
          .addSelect('COUNT(*)', 'totalSeekers')
          .addSelect('COUNT(CASE WHEN sub_registration.no_of_hdbs = 0 THEN 1 END)', 'firstTimers')
          .addSelect('COUNT(CASE WHEN sub_registration.no_of_hdbs BETWEEN 1 AND 10 THEN 1 END)', 'oneToTenHdbsMsds')
          .addSelect('COUNT(CASE WHEN sub_registration.no_of_hdbs > 10 THEN 1 END)', 'greaterThanTenHdbsMsds')
          .addSelect(`COUNT(CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN 1 END)`, 'organisation')
          .addSelect(`COUNT(CASE WHEN sub_registration.gender = '${GenderEnum.MALE}' THEN 1 END)`, 'male')
          .addSelect(`COUNT(CASE WHEN sub_registration.gender = '${GenderEnum.FEMALE}' THEN 1 END)`, 'female')
          .leftJoin('sub_registration.allocatedProgram', 'sub_program')
          .leftJoin('sub_registration.user', 'user')
          .where('sub_registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('sub_registration.allocated_program_id')
          .addGroupBy('sub_program.name')
          .addGroupBy('sub_program.id');
          if(rmId){
            detailedDataQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }

          const result  = await detailedDataQuery.getRawMany();

        return result;
        
        case 'gender':
          let genderQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('sub_registration.gender', 'gender')
          .innerJoin('hdb_registration_approval', 'approval', 'approval.registration_id = sub_registration.id')
          .where('sub_registration.program_id = :programId', { programId: programId })
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('sub_registration.gender');
          const genderCounts = await genderQuery.getRawMany();
          return genderCounts;

        case 'demographics':
        const ageGenderQuery = this.registrationRepo.createQueryBuilder('sub_registration')
          .select('EXTRACT(YEAR FROM AGE(sub_registration.dob))', 'age')
          .addSelect('sub_registration.gender', 'gender')
          .addSelect('COUNT(DISTINCT sub_registration.id)', 'count')

          if(subProgramId){
           // Demographics for a specific sub-program
            ageGenderQuery.where('sub_registration.allocated_program_id = :subProgramId', { subProgramId: subProgramId });
          }
          else {
            // Demographics for all sub-programs under the main program
            ageGenderQuery.where('sub_registration.program_id = :programId', { programId: programId });
            ageGenderQuery.andWhere('sub_registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT });
          }
          if(rmId){
            ageGenderQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          ageGenderQuery
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('age')
          .addGroupBy('sub_registration.gender');

        const rawAgesGender = await ageGenderQuery.getRawMany();

        return rawAgesGender;
              
        case 'payment':
          let paymentQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('payment.payment_status', 'paymentStatus')
          .leftJoin('sub_registration.paymentDetails', 'payment')

          if(subProgramId){
            // Payment counts for a specific sub-program
            paymentQuery.where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId });
          }
          else {
            // Payment counts for all sub-programs under the main program
            paymentQuery.where('sub_registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds });
          }
          if(rmId){
            paymentQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          paymentQuery
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('payment.payment_status');

          const paymentCounts = await paymentQuery.getRawMany();
          return paymentCounts;

        case 'invoice':
          const invoiceQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('invoice.invoice_status', 'invoiceStatus')
          .innerJoin('sub_registration.invoiceDetails', 'invoice')

          if(subProgramId){
          // Invoice counts for a specific sub-program
          invoiceQuery.where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId });
          }
          else {
          // Invoice counts for all sub-programs under the main program
          invoiceQuery.where('sub_registration.program_id = :programId', { programId: programId });
          }
          if(rmId){
          invoiceQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          invoiceQuery
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('invoice.invoice_status');

          let totalCountQuery = invoiceQuery
          totalCountQuery.andWhere('invoice.invoice_status != :invoiceStatus', { invoiceStatus: InvoiceStatusEnum.SAVE_AS_DRAFT });
          const totalInvoiceCount = await totalCountQuery.getCount();
          
          const invoiceStatusCounts = await invoiceQuery.getRawMany();
          return { invoiceStatusCounts : invoiceStatusCounts, totalInvoiceCount :  totalInvoiceCount};
        case 'travelPlan':
          const travelPlanQuery = this.registrationRepo
            .createQueryBuilder('sub_registration')
            .select('COUNT(DISTINCT sub_registration.id)', 'count')
            .addSelect('travelPlan.travel_type', 'travelType')
            .leftJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registration_id = sub_registration.id')
            .andWhere('sub_registration.deleted_at IS NULL')
            .groupBy('travelPlan.travel_type');

          if (subProgramId) {
            travelPlanQuery.where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId });
          }

          if (rmId) {
            travelPlanQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const travelPlanCounts = await travelPlanQuery.getRawMany();
          return travelPlanCounts;
        
        case 'travelStatus':
          const travelStatusQuery =  this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'totalRegistrations')
          .addSelect(`COUNT(CASE WHEN travelInfo.travel_info_status = 'completed' AND travelPlan.travel_plan_status = 'completed' THEN travelInfo.id END)`, 'completedCount')
          .addSelect(
            `COUNT(CASE WHEN travelInfo.travel_info_status != 'completed' OR travelPlan.travel_plan_status != 'completed' OR travelInfo.travel_info_status IS NULL OR travelPlan.travel_plan_status IS NULL THEN sub_registration.id END)`,
            'pendingCount'
          )
          .leftJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registrationId = sub_registration.id')
          .leftJoin('hdb_registration_travel_info', 'travelInfo', 'travelInfo.registrationId = sub_registration.id')
          .where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId })
          .andWhere('sub_registration.deleted_at IS NULL')
          // .groupBy('sub_registration.id');

          if(rmId){
            travelStatusQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const travelStatusCounts = await travelStatusQuery.getRawOne();
          return travelStatusCounts;
          
        case 'location':
          const cities = MajorCities
          const locationQuery = this.registrationRepo
            .createQueryBuilder('sub_registration')
            .select(`CASE 
              WHEN sub_registration.city IN (:...majorCities) THEN sub_registration.city
              ELSE 'Others'
            END`, 'location')
            .addSelect('COUNT(DISTINCT sub_registration.id)', 'count')
            .where('sub_registration.deleted_at IS NULL')
            .andWhere('sub_registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT })

            if(programId){
              locationQuery.andWhere('sub_registration.program_id = :programId', { programId });
            }else{
              locationQuery.andWhere('sub_registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
            }
            locationQuery.groupBy('location');

          if(rmId){
            locationQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const locationCounts = await locationQuery.setParameter('majorCities', cities).getRawMany();
          return locationCounts;
        }
    }catch(error){
      console.error('Error fetching registration statistics:', error);
      throw new Error('Failed to fetch registration statistics');
    }
  }

  async getSubPrograms(programId : number): Promise<{ id: number; name: string; code: string; totalSeats: number }[]> {
    try {
      const subPrograms = await this.programRepo.find({
        where: { primaryProgramId: programId, deletedAt: IsNull() },
        select: ['id', 'name', 'code', 'totalSeats'],
        order: { groupDisplayOrder: 'ASC' },
      });
      return subPrograms;
    } catch (error) {
      this.logger.error('Error fetching sub-programs', error);
      throw new InifniNotFoundException(ERROR_CODES.PROGRAM_NOTFOUND, null, null, programId.toString());
    }
  }
  
  // ===== USER REGISTRATION QUERIES =====

  async findExistingUserRegistration(userId: number, programId: number, programSessionId?: number): Promise<ProgramRegistration | null> {
    return await this.registrationRepo.findOne({
      where: { 
        userId, 
        program: { id: programId }, 
        programSession: programSessionId ? { id: programSessionId } : undefined,
        deletedAt: IsNull(),
      },
    });
  }

  // ===== PROGRAM AND SESSION QUERIES =====

  async findProgramWithType(programId: number): Promise<Program | null> {
    return await this.programRepo.findOne({
      where: { id: programId },
      relations: ['type'],
    });
  }

  async findSessionForProgram(sessionId: number, programId: number): Promise<ProgramSession | null> {
    return await this.sessionRepo.findOne({
      where: { id: sessionId },
      relations: ['program'],
    });
  }

  // ===== QUESTION QUERIES =====

  async findProgramQuestions(programId: number, sessionId?: number) {
    return await this.programQuestionRepo.find({
      where: {
        program: { id: programId },
        programSession: sessionId ? { id: sessionId } : IsNull(),
        deletedAt: IsNull(),
      },
      relations: ['question', 'question.formSection'],
    });
  }

  async findFormSection(formSectionId: number): Promise<FormSection | null> {
    return await this.dataSource.getRepository(FormSection).findOne({
      where: { id: formSectionId },
    });
  }

  // ===== REGISTRATION QUERIES =====

  async findRegistrationWithRelations(registrationId: number) {
    return await this.registrationRepo.findOne({
      where: { id: registrationId, deletedAt: IsNull() },
      relations: ['program', 'program.type', 'programSession', 'createdBy', 'updatedBy'],
    });
  }

  async findPendingApproval(registrationId: number): Promise<RegistrationApproval | null> {
    return await this.approvalRepo.findOne({
      where: { 
        registrationId: registrationId,
        approvalStatus: ApprovalStatusEnum.PENDING,
        deletedAt: IsNull(),
      },
      order: { createdAt: 'DESC' },
    });
  }

  async findLastWaitlistRegistration(programId: number, sessionId?: number) {
    const whereCondition: any = {
      waitingListSeqNumber: Not(IsNull()),
    };
    
    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }
    
    return await this.registrationRepo.findOne({
      where: whereCondition,
      order: { waitingListSeqNumber: 'DESC' },
    });
  }

  async findLastDirectRegistration(programId: number, sessionId?: number) {
    const whereCondition: any = {
      registrationSeqNumber: Not(IsNull()),
    };

    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }

    return await this.registrationRepo.findOne({
      where: whereCondition,
      order: { registrationSeqNumber: 'DESC' },
    });
  }

  /**
   * Get the next registration sequence number based on existing registrationSeqNumber values
   * Extracts numeric part from registrationSeqNumber and increments it
   */
  private async getNextRegistrationSeqNumber(programId: number, sessionId?: number): Promise<number> {
    const whereCondition: any = {
      registrationSeqNumber: Not(IsNull()),
    };

    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }

    const lastRegistration = await this.registrationRepo.findOne({
      where: whereCondition,
      order: { registrationSeqNumber: 'DESC' },
    });

    if (!lastRegistration?.registrationSeqNumber) {
      return 1;
    }

    // Extract numeric part from registrationSeqNumber (e.g., "ABC123" -> 123)
    const numericPart = await this.extractNumericFromSeqNumber(lastRegistration.registrationSeqNumber, programId);
    return numericPart + 1;
  }

  /**
   * Get the next waitlist sequence number based on existing waitListRegistrationSeqNumber values
   * Extracts numeric part from waitListRegistrationSeqNumber and increments it
   */
  private async getNextWaitlistSeqNumber(programId: number, sessionId?: number): Promise<number> {
    const whereCondition: any = {
      waitListRegistrationSeqNumber: Not(IsNull()),
    };

    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }

    const lastWaitlistRegistration = await this.registrationRepo.findOne({
      where: whereCondition,
      order: { waitListRegistrationSeqNumber: 'DESC' },
    });

    if (!lastWaitlistRegistration?.waitListRegistrationSeqNumber) {
      return 1;
    }

    // Extract numeric part from waitListRegistrationSeqNumber
    const numericPart = await this.extractNumericFromSeqNumber(
      lastWaitlistRegistration.waitListRegistrationSeqNumber,
      programId,
      true,
    );
    return numericPart + 1;
  }

  /**
   * Helper method to extract numeric part from sequence number
   * e.g., "ABC123" -> 123, "XYZ001" -> 1
   */
  private async extractNumericFromSeqNumber(
    seqNumber: string,
    programId: number,
    isWaitlist = false,
  ): Promise<number> {
    if (programId) {
      const program = await this.programRepo.findOne({ where: { id: programId } });
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }
      const prefix = isWaitlist ? `WL${program.code}` : program.code;
      const escapedPrefix = prefix.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      seqNumber = seqNumber.replace(new RegExp(`^${escapedPrefix}`), '');
    }
    const match = seqNumber.match(/\d+$/);
    return match ? parseInt(match[0], 10) : 0;
  }

  /**
   * Generate formatted sequence number with program code prefix
   */
  private generateFormattedSeqNumber(program: Program, numericSeq: number, isWaitlist: boolean = false): string {
    const prefix = isWaitlist ? `WL${program.code}` : program.code;
    return `${prefix}${numericSeq.toString().padStart(3, '0')}`;
  }

  // ===== REGISTRATION CREATION =====

  async createRegistration(params: {
    dto: CreateRegistrationDto,
    userId: number,
    isSelfRegister: boolean,
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
    seatInfo: any,
    processedAnswers: any,
    registrationOutcome: any,
    registrationStatus?: RegistrationStatusEnum,
    allocatedProgram?: Program | null,
    allocatedSession?: ProgramSession | null
  }) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      await queryRunner.startTransaction();
      transactionStarted = true;
      this.logger.log('Creating new registration...', params.dto);

      // Create base registration record
      const registrationData = this.buildRegistrationData(params);
      const registration = queryRunner.manager.create(ProgramRegistration, registrationData);
      await queryRunner.manager.save(ProgramRegistration, registration);
      const regId = registration.id;

      // Save custom responses
      await this.saveCustomResponses(queryRunner, params.processedAnswers.customResponses, regId);

      // Store grouped answers to specific table locations
      if (params.processedAnswers.groupedDirectLocationAnswers.size > 0) {
        await this.storeQuestionResponsesGrouped(params.processedAnswers.groupedDirectLocationAnswers, regId, queryRunner, registration, params.registrationStatus);
      }

      // Store question-answer mappings
      await this.upsertQuestionAnswers(queryRunner, regId, params.processedAnswers.qaMappings, params.userId);

      // Handle registration outcome
      const result = await this.handleRegistrationOutcome(queryRunner, regId, params);

      await queryRunner.commitTransaction();
      transactionCommitted = true;
      return result;
      
    } catch (error) {
      if (transactionStarted && !transactionCommitted) {
        await queryRunner.rollbackTransaction();
      }
      this.logger.error('Registration transaction failed', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_CREATION_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  // ===== REGISTRATION UPDATE =====

  async updateRegistrationData(params: {
    registration: any;
    processedAnswers: any;
    userId: number;
    registrationStatus?: RegistrationStatusEnum;
  }) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;
    
    try {
      await queryRunner.startTransaction();
      transactionStarted = true;

      // Update or create custom responses
      await this.updateCustomResponses(queryRunner, params.processedAnswers.customResponses, params.registration.id, params.userId);

      // Store answers in designated table locations
      if (params.processedAnswers.groupedDirectLocationAnswers.size > 0) {
        await this.storeQuestionResponsesGrouped(params.processedAnswers.groupedDirectLocationAnswers, params.registration.id, queryRunner, params.registration, params.registrationStatus);
      }

      // Update question-answer mappings
      await this.upsertQuestionAnswers(queryRunner, params.registration.id, params.processedAnswers.qaMappings, params.userId);

      await queryRunner.commitTransaction();
      transactionCommitted = true;
      
      return { 
        registrationId: params.registration.id,
        updated: true,
      };
      
    } catch (error) {
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Registration update transaction failed', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_CREATION_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  async cancelRegistration(registrationId: number, cancellationDate: Date, cancelledBy: number) {
    if (!registrationId) {
      throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_ID_REQUIRED);
    }
    const registration = await this.registrationRepo.findOne({
      where: { id: registrationId, deletedAt: IsNull() }
    });
    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,   
        registrationId.toString(),
      );
    }
    if (registration.registrationStatus === RegistrationStatusEnum.CANCELLED) {
      throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_ALREADY_CANCELLED);
    }
    if (registration.registrationStatus === RegistrationStatusEnum.COMPLETED) {
      // decrement seat counts if registration is completed
      const queryRunner = this.dataSource.createQueryRunner();
      await queryRunner.connect();
      try {
        await queryRunner.startTransaction();
        await decrementSeatCounts(queryRunner.manager, registration.program.id, registration.programSession?.id);
        await queryRunner.commitTransaction();
      } catch (error) {
        await queryRunner.rollbackTransaction();
        this.logger.error('Failed to decrement seat counts during cancellation', error);
        throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_CANCEL_FAILED, error);
      } finally {
        await queryRunner.release();
      }
    }

    await this.registrationRepo.update(registrationId, {
      cancellationDate,
      cancelledBy,
      registrationStatus: RegistrationStatusEnum.CANCELLED,
    });
    return { registrationId, cancelled: true };
  }

  // ===== FAILURE TRACKING =====

  async createRegistrationFailure(program: Program, session: ProgramSession | null, dto: CreateRegistrationDto, userId: number, reason: string) {
    const failure = this.failureRepo.create({
      programId: dto.programId,
      programSessionId: session?.id || null,
      payload: dto as any,
      failureReason: reason,
      createdBy: { id: userId } as any,
      updatedBy: { id: userId } as any,
      program,
      programSession: session,
    });
    return await this.failureRepo.save(failure);
  }

  // ===== SEARCH AND LISTING =====
  async findRegistrations(
    limit: number | 'all',
    offset: number | 'all',
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles?: string[],
    filterRegistrationsByUserId?: number | null,
    sortKey: string = 'createdAt',
    sortOrder: 'ASC' | 'DESC' = 'DESC',
  ) {
    try {
      const whereClause: any = { deletedAt: IsNull() };
      // if (!(userRoles?.includes('shoba') || userRoles?.includes('admin'))) {
      if (!parsedFilters?.createdBy) {
        whereClause.registrationStatus = Not(RegistrationStatusEnum.SAVE_AS_DRAFT)
      }
      // }
      // Apply program/session filters
      if (programId) {
        const program = await this.programRepo.findOne({ where: { id: programId } });
        if (program?.registrationLevel === 'program') {
          whereClause.program = { id: programId };
          whereClause.programSession = IsNull();
        } else if (program?.registrationLevel === 'session') {
          if (programSessionId) {
            whereClause.programSession = { id: programSessionId };
          } else {
            whereClause.program = { id: programId };
          }
        } else {
          whereClause.program = { id: programId };
        }
      } else if (programSessionId) {
        whereClause.programSession = { id: programSessionId };
      }
  
      if (parsedFilters?.registrationStatus) {
        if (Array.isArray(parsedFilters.registrationStatus)) {
          whereClause.registrationStatus = In(parsedFilters.registrationStatus);
        } else {
          whereClause.registrationStatus = parsedFilters.registrationStatus;
        }
      }
  
      if (parsedFilters?.createdBy) {
        whereClause.createdBy = { id: parsedFilters.createdBy };
      }
  
      if (parsedFilters?.organisation) {
        const orgValues = Array.isArray(parsedFilters.organisation)
          ? parsedFilters.organisation
          : [parsedFilters.organisation];

        // If both 'yes' and 'no' are present, do not filter by userType (include all)
        if (orgValues.includes('yes') && orgValues.includes('no')) {
          // No filter needed, include all user types
        } else if (orgValues.includes('yes')) {
          whereClause.user = { ...whereClause.user, userType: UserTypeEnum.ORG };
        } else if (orgValues.includes('no')) {
          whereClause.user = { ...whereClause.user, userType: Not(UserTypeEnum.ORG) };
        }
      }
      if (parsedFilters?.approvalStatus) {
        whereClause.approvals = {
          approvalStatus: Array.isArray(parsedFilters.approvalStatus) 
            ? In(parsedFilters.approvalStatus)
            : parsedFilters.approvalStatus,
          deletedAt: IsNull(),
        };
      }
  
      if (parsedFilters?.mahatriaChoice) {
        whereClause.preferences = {
          preferredProgram: IsNull(),
          preferredSession: IsNull(),
        }
        whereClause.allocatedProgram = IsNull();
        whereClause.allocatedSession = IsNull();
      }
  
      // Existing preferences-based filters
      if (parsedFilters?.preferredProgramId || parsedFilters?.preferredSessionId) {
        const preferenceSubQuery = this.preferenceRepo
          .createQueryBuilder('preference')
          .select('preference.registration_id')
          .where('preference.priority_order = 1')
          .andWhere('preference.deleted_at IS NULL');
  
        if (parsedFilters.preferredProgramId) {
          preferenceSubQuery.andWhere('preference.preferred_program_id = :preferredProgramId', {
            preferredProgramId: parsedFilters.preferredProgramId,
          });
        }
        if (parsedFilters.preferredSessionId) {
          preferenceSubQuery.andWhere('preference.preferred_session_id = :preferredSessionId', {
            preferredSessionId: parsedFilters.preferredSessionId,
          });

          // In Seat Allocation for subPrograms KPIs blessed and registration save-as-draft seekers should be excluded.
          if (parsedFilters.kpiCategory == 'all') {
            preferenceSubQuery
              .leftJoin('preference.registration', 'registration')
              .leftJoin('registration.approvals', 'approval');

            if (Array.isArray(parsedFilters.approvalStatus))
            {
              preferenceSubQuery.andWhere('approval.approval_status IN (:...approvalStatuses)', {
                approvalStatuses: parsedFilters.approvalStatus,
              });
            }
            else if (parsedFilters.approvalStatus) {
              preferenceSubQuery.andWhere('approval.approval_status = :approvalStatus', {
                approvalStatus: parsedFilters.approvalStatus,
              });
            }
          }

        }
        const preferenceRegistrationIds = await preferenceSubQuery.getRawMany().then(rows => rows.map(r => r.registration_id))
        const allRegistrationIds = [...new Set([...preferenceRegistrationIds || []])];
        whereClause.id = In(allRegistrationIds);
      }
  
      // Existing allocated filters
      if (parsedFilters.allocatedProgramId) {
        whereClause.allocatedProgram = { id: parsedFilters.allocatedProgramId };
      }
  
      if (parsedFilters.allocatedSessionId) {
        whereClause.allocatedSession = { id: parsedFilters.allocatedSessionId };
      }
  
      if (filterRegistrationsByUserId) {
        whereClause.rmContactUser = { id: filterRegistrationsByUserId };
      }
    
      if (parsedFilters?.gender) {
        if (Array.isArray(parsedFilters.gender)) {
          whereClause.gender = In(parsedFilters.gender);
        } else {
          whereClause.gender = parsedFilters.gender;
        }
      }
  
      if (parsedFilters?.age) {
        const ageRanges = Array.isArray(parsedFilters.age) ? parsedFilters.age : [parsedFilters.age];
        const ageRegistrationIds: number[] = [];
        
        for (const range of ageRanges) {
          const currentDate = new Date();
          let minAge: number = 0, maxAge: number = 0;
  
          switch (range) {
            case '0-20':
              minAge = 0;
              maxAge = 20;
              break;
            case '21-30':
              minAge = 21;
              maxAge = 30;
              break;
            case '31-50':
              minAge = 31;
              maxAge = 50;
              break;
            case '51-65':
              minAge = 51;
              maxAge = 65;
              break;
            case '>65':
              minAge = 66;
              maxAge = 200; // Arbitrary large number for upper limit
              break;
            default:
              continue;
          }
  
          const maxBirthDate = new Date(currentDate.getFullYear() - minAge, currentDate.getMonth(), currentDate.getDate());
          const minBirthDate = new Date(currentDate.getFullYear() - maxAge - 1, currentDate.getMonth(), currentDate.getDate());
          

          const ageSubQuery = this.registrationRepo
            .createQueryBuilder('reg')
            .select('reg.id')
            .where('reg.dob BETWEEN :minBirthDate AND :maxBirthDate', {
              minBirthDate: minBirthDate.toISOString().split('T')[0],
              maxBirthDate: maxBirthDate.toISOString().split('T')[0]
            })
            .andWhere('reg.deleted_at IS NULL');
  
          const ageResults = await ageSubQuery.getRawMany();
          ageRegistrationIds.push(...ageResults.map(r => r.reg_id));
        }

        // Always set a restrictive filter if no matches found
        if (ageRegistrationIds.length > 0) {
          const uniqueIds = [...new Set(ageRegistrationIds)];
          whereClause.id = In(uniqueIds);
        } else {
          whereClause.id = In([-1]);
        }
      }
  
      if (parsedFilters?.numberOfHdbs) {
      //   const hdbFilters = Array.isArray(parsedFilters.numberOfHdbs)
      //     ? parsedFilters.numberOfHdbs
      //     : [parsedFilters.numberOfHdbs];
      
      //   // Support format: { operator: '>=', value: 5 }
      //   const typeormOperators: FindOperator<number>[] = [];

      //   for (const condition of hdbFilters) {
      //     if (!condition || typeof condition !== 'object') continue;
      //     const operator = condition.operator;
      //     const value = parseInt(condition.value);
      //     if (isNaN(value)) continue;
      
      //     switch (operator) {
      //       case '>':
      //         typeormOperators.push(MoreThan(value));
      //         break;
      //       case '>=':
      //         typeormOperators.push(MoreThanOrEqual(value));
      //         break;
      //       case '<':
      //         typeormOperators.push(LessThan(value));
      //         break;
      //       case '<=':
      //         typeormOperators.push(LessThanOrEqual(value));
      //         break;
      //       case '==':
      //         typeormOperators.push(Equal(value));
      //         break;
      //     }
      //   }
      //  console.log('TypeORM operators for HDBs:', typeormOperators);
      //   if (typeormOperators.length === 1) {
      //     whereClause.noOfHDBs = typeormOperators[0];
      //   } else if (typeormOperators.length > 1) {
      //     // If multiple conditions (e.g., >=5 and <=10), fallback to a custom range
      //     // This only works when combining range-style logic
      //     const min = typeormOperators.find(op => op instanceof MoreThan || op instanceof MoreThanOrEqual);
      //     const max = typeormOperators.find(op => op instanceof LessThan || op instanceof LessThanOrEqual);
      
      //     if (min && max) {
      //       whereClause.noOfHDBs = {
      //         ...min,
      //         ...max,
      //       };
      //     }
      //   }
      const hdbRanges = Array.isArray(parsedFilters.numberOfHdbs) ? parsedFilters.numberOfHdbs : [parsedFilters.numberOfHdbs];
      const hdbRegistrationIds: number[] = [];

      for (const range of hdbRanges) {
        let minHdb: number = 0, maxHdb: number = 0;
        let isRange = false;

        if (typeof range === 'string') {
          if (range.includes('-')) {
            // e.g. '0-5'
            const [min, max] = range.split('-').map(Number);
            if (!isNaN(min) && !isNaN(max)) {
              minHdb = min;
              maxHdb = max;
              isRange = true;
            }
          } else if (range.startsWith('>')) {
            minHdb = parseInt(range.replace('>', ''), 10) + 1;
            maxHdb = 1000; // Arbitrary upper limit
            isRange = true;
          } else if (range.startsWith('<')) {
            minHdb = 0;
            maxHdb = parseInt(range.replace('<', ''), 10) - 1;
            isRange = true;
          } else if (!isNaN(Number(range))) {
            minHdb = maxHdb = Number(range);
            isRange = true;
          }
        } else if (typeof range === 'number') {
          minHdb = maxHdb = range;
          isRange = true;
        }

        if (!isRange) continue;

        const hdbSubQuery = this.registrationRepo
          .createQueryBuilder('reg')
          .select('reg.id')
          .where('reg.no_of_hdbs BETWEEN :minHdb AND :maxHdb', {
            minHdb,
            maxHdb,
          })
          .andWhere('reg.deleted_at IS NULL');

        const hdbResults = await hdbSubQuery.getRawMany();
        hdbRegistrationIds.push(...hdbResults.map(r => r.reg_id));
      }

      // Always set a restrictive filter if no matches found
      if (hdbRegistrationIds.length > 0) {
        const uniqueIds = [...new Set(hdbRegistrationIds)];
        whereClause.id = In(uniqueIds);
      } else {
        whereClause.id = In([-1]);
      }
      
      }
      
      if (parsedFilters?.location) {
        if (Array.isArray(parsedFilters.location)) {
          whereClause.city = In(parsedFilters.location);
        } else {
          whereClause.city = parsedFilters.location;
        }
      }
  
      if (parsedFilters?.paymentStatus) {
        const paymentStatuses = Array.isArray(parsedFilters.paymentStatus)
          ? parsedFilters.paymentStatus
          : [parsedFilters.paymentStatus];

        const includesPending = paymentStatuses.includes('payment_pending');
        const includesCompleted = paymentStatuses.includes('payment_completed');

        if (includesPending && includesCompleted) {
          // No filtering needed if both pending and completed are requested
        } else if (includesCompleted) {
          whereClause.paymentDetails = {
            ...whereClause.paymentDetails,
            paymentStatus: In([
              PaymentStatusEnum.ONLINE_COMPLETED,
              PaymentStatusEnum.OFFLINE_COMPLETED,
            ]),
            deletedAt: IsNull(),
          };
        } else if (includesPending) {
          const pendingIdsRaw = await this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin(
              'registration.paymentDetails',
              'payment',
              'payment.deleted_at IS NULL'
            )
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere(
              '(payment.payment_status IN (:...pending) OR (payment.id IS NULL AND registration.allocated_program_id IS NOT NULL))',
              { pending: [PaymentStatusEnum.ONLINE_PENDING, PaymentStatusEnum.OFFLINE_PENDING] }
            )
            .getRawMany();

          const pendingIds = pendingIdsRaw.map((row: any) => Number(row.id));

          const existingIds =
            whereClause.id && (whereClause.id as any)._value
              ? (whereClause.id as any)._value
              : null;

          const finalIds = existingIds
            ? existingIds.filter((id: number) => pendingIds.includes(id))
            : pendingIds;

          whereClause.id = In(finalIds.length > 0 ? finalIds : [-1]);
        }
      }

      if (parsedFilters?.paymentMode) {
        const paymentModes = Array.isArray(parsedFilters.paymentMode)
          ? parsedFilters.paymentMode
          : [parsedFilters.paymentMode];

        if (paymentModes.length > 0) {
          whereClause.paymentDetails = {
            ...whereClause.paymentDetails,
            paymentMode: In(paymentModes),
            deletedAt: IsNull(),
          };
        }
      }
  
      if (parsedFilters?.invoiceStatus) {
        const invoiceStatuses = Array.isArray(parsedFilters.invoiceStatus) ? parsedFilters.invoiceStatus : [parsedFilters.invoiceStatus];
        const mappedStatuses: string[] = [];
  
        invoiceStatuses.forEach(status => {
          switch (status) {
            case 'invoice_completed':
              mappedStatuses.push(InvoiceStatusEnum.INVOICE_COMPLETED);
              break;
            case 'invoice_pending':
              mappedStatuses.push(InvoiceStatusEnum.INVOICE_PENDING);
              break;
          }
        });
  
        if (mappedStatuses.length > 0) {
          whereClause.invoiceDetails = {
            invoiceStatus: In(mappedStatuses),
            deletedAt: IsNull(),
          };
        }
      }
  
      if (parsedFilters?.travelPlan) {
        const travelStatuses = Array.isArray(parsedFilters.travelPlan)
          ? parsedFilters.travelPlan
          : [parsedFilters.travelPlan];
        const mappedStatuses: string[] = [];

        travelStatuses.forEach(status => {
          switch (status) {
            case 'travel_completed':
              mappedStatuses.push(TravelStatusEnum.COMPLETED);
              break;
            case 'travel_pending':
              mappedStatuses.push(TravelStatusEnum.PENDING);
              break;
          }
        });

        if (mappedStatuses.length > 0) {
          whereClause.travelPlans = {
            travelPlanStatus: In(mappedStatuses),
            deletedAt: IsNull(),
          };
        }
      }

      if (parsedFilters?.travelStatus) {
        const statusValues = Array.isArray(parsedFilters.travelStatus)
          ? parsedFilters.travelStatus
          : [parsedFilters.travelStatus];

        const includesPending = statusValues.includes('travel_pending');
        const includesCompleted = statusValues.includes('travel_completed');

        // If both pending and completed are selected, no filtering is needed
        if (includesPending && includesCompleted) {
          // do nothing
        } else if (includesPending) {
          // Fetch ids where either travel plan or info is not completed or missing
          const pendingIdsRaw = await this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.travelPlans', 'travelPlan', 'travelPlan.deleted_at IS NULL')
            .leftJoin('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere(
              '(travelInfo.travel_info_status != :completed ' +
                'OR travelPlan.travel_plan_status != :completed ' +
                'OR travelInfo.travel_info_status IS NULL ' +
                'OR travelPlan.travel_plan_status IS NULL)',
              { completed: TravelStatusEnum.COMPLETED },
            )
            .getRawMany();

          const pendingIds = pendingIdsRaw.map((row: any) => Number(row.id));

          const existingIds = (whereClause.id && (whereClause.id as any)._value)
            ? (whereClause.id as any)._value
            : null;

          const finalIds = existingIds
            ? existingIds.filter((id: number) => pendingIds.includes(id))
            : pendingIds;

          whereClause.id = In(finalIds.length > 0 ? finalIds : [-1]);
        } else if (includesCompleted) {
          // Only completed records
          whereClause.travelPlans = {
            travelPlanStatus: TravelStatusEnum.COMPLETED,
            deletedAt: IsNull(),
          };
          whereClause.travelInfo = {
            travelInfoStatus: TravelStatusEnum.COMPLETED,
            deletedAt: IsNull(),
          };
        }
      }
  
      if (parsedFilters?.swapRequests) {
        const swapTypes = Array.isArray(parsedFilters.swapRequests) ? parsedFilters.swapRequests : [parsedFilters.swapRequests];

        if (swapTypes.length > 0) {
          whereClause.swapsRequests = {
            type: In(swapTypes),
            status : SwapRequestStatus.ACTIVE,
          };
        }
      }

      if(parsedFilters?.swapPreferredProgramId) {
        whereClause.swapsRequests = {
          ...whereClause.swapsRequests,
          requestedPrograms: {
            id: parsedFilters.swapPreferredProgramId,
            deletedAt: IsNull(),
          },
        };
      }

      if (parsedFilters?.preferredRoomMate) {
        const roomMateValues = Array.isArray(parsedFilters.preferredRoomMate)
          ? parsedFilters.preferredRoomMate.map((v: string) => v.toLowerCase())
          : [parsedFilters.preferredRoomMate.toLowerCase()];

        if (roomMateValues.includes('yes') && roomMateValues.includes('no')) {
          // No filter needed, include all
        } else if (roomMateValues.includes('yes')) {
          whereClause.preferredRoomMate = Not(IsNull());
        } else if (roomMateValues.includes('no')) {
          whereClause.preferredRoomMate = IsNull();
        }
      }
      if (parsedFilters?.rmRating) {
        const ratingFilters = Array.isArray(parsedFilters.rmRating)
          ? parsedFilters.rmRating
          : [parsedFilters.rmRating];

        const ratingIdsSet = new Set<number>();

        for (const ratingFilter of ratingFilters) {
          const query = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.ratings', 'rating', 'rating.deleted_at IS NULL')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .groupBy('registration.id');

          if (ratingFilter === 'unrated') {
            query.having('COUNT(rating.id) = 0');
          } else if (ratingFilter === 'only5' || ratingFilter === '5') {
            query.having('AVG(rating.rating) = 5');
          } else {
            const threshold = parseInt(ratingFilter);
            if (!isNaN(threshold)) {
              query.having('AVG(rating.rating) >= :threshold', { threshold });
            }
          }

          const ids = await query.getRawMany();
          ids.forEach((r: any) => ratingIdsSet.add(Number(r.id)));
        }

        if (ratingIdsSet.size > 0) {
          whereClause.id = In([...ratingIdsSet]);
        } else {
          whereClause.id = In([-1]);
        }
      }
  
      if (parsedFilters?.preferredProgram) {
        console.log('Using preferredProgram filter:', parsedFilters.preferredProgram);
        const preferredPrograms = Array.isArray(parsedFilters.preferredProgram) ? parsedFilters.preferredProgram : [parsedFilters.preferredProgram];
        
        if (preferredPrograms.length > 0) {
          const preferenceSubQuery = this.preferenceRepo
            .createQueryBuilder('preference')
            .select('preference.registration_id')
            .where('preference.preferred_program_id IN (:...preferredPrograms)', {
              preferredPrograms: preferredPrograms
            })
            .andWhere('preference.priority_order = 1')
            .andWhere('preference.deleted_at IS NULL');
  
          const preferenceResults = await preferenceSubQuery.getRawMany();
          const registrationIds = preferenceResults.map(r => r.registration_id);
          
          if (registrationIds.length > 0) {
            whereClause.id = In(registrationIds);
          }
        }
      }

      if (parsedFilters?.programStatus) {
        whereClause.program = {
          ...whereClause.program,
          status: Array.isArray(parsedFilters.programStatus)
            ? In(parsedFilters.programStatus)
            : parsedFilters.programStatus,
          deletedAt: IsNull(),
        };
      }

      if (parsedFilters?.programType) {
        whereClause.program = {
          ...whereClause.program,
          type: {
            key: parsedFilters.programType,
            deletedAt: IsNull(),
          },
          deletedAt: IsNull(),
        };
      }

      if (parsedFilters?.dateRange?.startDate && parsedFilters?.dateRange?.endDate) {
        // Convert DD/MM/YYYY to YYYY-MM-DD if needed
        const parseDate = (dateStr: string) => {
          if (/^\d{2}\/\d{2}\/\d{4}$/.test(dateStr)) {
            const [day, month, year] = dateStr.split('/');
            return `${year}-${month}-${day}`;
          }
          return dateStr;
        };
        whereClause.createdAt = Between(
          parseDate(parsedFilters.dateRange.startDate),
          parseDate(parsedFilters.dateRange.endDate)
        );
      }

      // Handle any remaining unknown filters
      if (parsedFilters) {
        const handledFilters = [
          'registrationStatus',
          'createdBy',
          'approvalStatus',
          'mahatriaChoice',
          'preferredProgramId',
          'preferredSessionId',
          'allocatedProgramId',
          'allocatedSessionId',
          'gender',
          'age',
          'numberOfHdbs',
          'location',
          'paymentStatus',
          'paymentMode',
          'invoiceStatus',
          'travelPlan',
          'travelStatus',
          'swapRequests',
          'preferredProgram',
          'programStatus',
          'programType',
          'rmRating',
          'preferredRoomMate',
          'dateRange',
          'organisation',
          'swapPreferredProgramId'
        ];

        delete parsedFilters.kpiCategory;
        delete parsedFilters.kpiFilter;

        for (const [key, value] of Object.entries(parsedFilters)) {
          if (handledFilters.includes(key)) {
            continue;
          }
          this.logger.debug(`Processing unknown filter - ${key}:`, value);
          if (Array.isArray(value)) {
            whereClause[key] = In(value);
          }
          else if (typeof value === 'object' && value !== null) {
            whereClause[key] = value;
          }
          else {
            whereClause[key] = value;
          }
        }
      }

      // Apply search text
      let finalWhereClause: any = whereClause;
      if (searchText?.trim()) {
        const searchTerm = ILike(`%${searchText.trim()}%`);
        
        // Create search conditions that properly inherit all base filters
        const createSearchCondition = (searchField: string, searchTarget?: any) => {
          const condition = { ...whereClause };
          
          if (searchTarget) {
            // For nested searches (like program.name), merge with existing program filters
            if (searchField === 'program') {
              condition.program = {
                ...condition.program, // Keep existing program filters
                ...searchTarget, // Add search criteria
              };
            }
          } else {
            // For direct field searches
            condition[searchField] = searchTerm;
          }
          
          return condition;
        };

        // Only include program/programType search if parsedFilters.createdBy is present
        if (parsedFilters?.createdBy) {
          finalWhereClause = [
            createSearchCondition('fullName'),
            createSearchCondition('emailAddress'),
            createSearchCondition('mobileNumber'),
            createSearchCondition('program', {
              name: searchTerm,
              deletedAt: IsNull(),
            }),
            createSearchCondition('program', {
              type: {
                name: searchTerm,
                deletedAt: IsNull(),
              },
              deletedAt: IsNull(),
            }),
          ];
        } else {
          finalWhereClause = [
            createSearchCondition('fullName'),
            // createSearchCondition('emailAddress'),
            createSearchCondition('mobileNumber'),
          ];
        }
      }
  
      const relations = [
        'program',
        'program.type',
        'programSession',
        'invoiceDetails',
        'paymentDetails', 
        'travelInfo',
        'travelPlans',
        'user',
        'approvals',
        'allocatedProgram',
        'allocatedSession',
        'swapsRequests',
        'swapsRequests.requestedPrograms',
        'ratings',
        'rmContactUser',
        'preferences',
        'preferences.preferredProgram',
        'preferences.preferredSession',
      ];
      const data = await this.commonDataService.get(
        this.registrationRepo,
        undefined,
        finalWhereClause,
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        relations,
        { rmContactUser: ["id", "fullName"] }
      );
  
      // Add averageRating calculation
      const dataWithAvgRating = data.map((registration: any) => {
        if (registration.ratings && Array.isArray(registration.ratings) && registration.ratings.length > 0) {
          const total = registration.ratings.reduce((sum: number, r: any) => sum + (Number(r.rating) || 0), 0);
          registration.averageRating = (total / registration.ratings.length).toFixed(2);
        } else {
          registration.averageRating = null;
        }
        return registration;
      });
  
      // ✅ Handle pagination response based on limit/offset type
      let paginationInfo;
      
      if (limit === 'all' || offset === 'all') {
        // Unlimited query - return simplified pagination
        paginationInfo = {
          totalPages: 1,
          pageNumber: 1,
          pageSize: null,
          totalRecords: dataWithAvgRating.length,
          numberOfRecords: dataWithAvgRating.length,
          queryType: 'unlimited',
          isUnlimited: true,
        };
      } else {
        // Regular pagination - get total count
        const total = await this.registrationRepo.count({ where: finalWhereClause });
        paginationInfo = {
          totalPages: Math.ceil(total / (limit as number)),
          pageNumber: Math.floor((offset as number) / (limit as number)) + 1,
          pageSize: limit as number,
          totalRecords: total,
          numberOfRecords: dataWithAvgRating.length,
          queryType: 'paginated',
          isUnlimited: false,
        };
      }
  
      return {
        data: dataWithAvgRating,
        pagination: paginationInfo,
      };
  
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }
  async findRegistrationById(id: number) {
    try {
      const registration = await this.commonDataService.findOneById(
        this.registrationRepo,
        id,
        true,
        [
          'program',
          'program.type',
          'programSession',
          'invoiceDetails',
          'paymentDetails',
          'travelInfo',
          'travelPlans',
          'user',
          'approvals',
          'preferences',
          'preferences.preferredProgram',
          'preferences.preferredSession',
          'ratings',
          'allocatedProgram',
          'allocatedSession',
          'swapsRequests',
          'swapsRequests.requestedPrograms',
          'rmContactUser'
        ]
      );
      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }
      return registration;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async findQuestionResponsesByRegistrationId(registrationId: number) {
    try {
      const answers = await this.questionAnswerRepo.find({
        where: { registrationId },
        relations: [
          'question',
          'question.formSection',
          'registration',
          'registration.program',
        ],
        order: { id: 'ASC' },
      });

      const programId = answers[0].registration?.program?.id;
      let mapping: any = undefined;
      if (programId) {
        mapping = await this.programQuestionRepo.find({
          where: {
            program: { id: programId },
            deletedAt: IsNull(),
          },
          relations: ['question'],
        });
      }

      const responsePromises = answers.map(async a => {

        const section =  a.question?.formSection || null;
        const order = mapping?.find(m => m.question.id === a.questionId)?.displayOrder || 0;

        return {
          questionId: a.questionId,
          questionLabel: a.question?.label,
          config: a.question?.config,
          questionType: a.question?.type,
          sectionId: section?.id,
          sectionName: section?.name,
          sectionOrder: section?.displayOrder || 0,
          displayOrder: order,
          sectionKey: section?.key,
          answer: a.answerValue,
        };
      });

      const responses = await Promise.all(responsePromises);
      responses.sort((a, b) => {
        if (a.sectionOrder !== b.sectionOrder) {
          return (a.sectionOrder ?? 0) - (b.sectionOrder ?? 0);
        }
        return (a.displayOrder ?? 0) - (b.displayOrder ?? 0);
      });
      return responses;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_QUESTION_RESPONSE_GET_FAILED, error);
    }
  }

  // ===== KPI QUERY METHODS FOR SERVICE LAYER =====

  async findGroupedPrograms(primaryProgramId: number) {
    return await this.programRepo.find({
      where: { 
        primaryProgramId: primaryProgramId,
        deletedAt: IsNull()
      },
      select: ['id', 'name', 'totalSeats'],
      order: { groupDisplayOrder: 'ASC' }
    });
  }

  async getApprovalMetrics(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let approvalQuery = this.dataSource
      .getRepository(RegistrationApproval)
      .createQueryBuilder('approval')
      .leftJoin('approval.registration', 'registration')
      .leftJoin('registration.user', 'user')
      .select([
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' THEN 1 END) AS "holdCount"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' THEN 1 END) AS "rejectedCount"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "holdOrgCount"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "rejectedOrgCount"`
      ])
      .addSelect('registration.gender AS "gender"')
      .where('registration.program_id = :programId', { programId })
      .andWhere('approval.deleted_at IS NULL')
      .andWhere('registration.deleted_at IS NULL')
      .groupBy('registration.gender');

    return await approvalQuery.getRawMany();
  }

  async getMahatriaChoiceMetrics(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let mahatriaChoiceUnallocatedQuery = this.dataSource
      .getRepository(ProgramRegistration)
      .createQueryBuilder('registration')
      .leftJoin('registration.preferences', 'preference')
      .leftJoin('registration.user', 'user')
      .select([
        'COUNT(DISTINCT registration.id) AS "count"',
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN registration.id END) AS "maleCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN registration.id END) AS "femaleCount"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "orgCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "maleOrganizationUserCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "femaleOrganizationUserCount"`
      ])
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id IS NULL')
      .andWhere('preference.id IS NULL') // No preferences exist
      .andWhere('registration.deleted_at IS NULL');

    mahatriaChoiceUnallocatedQuery = this.applyFiltersToQuery(mahatriaChoiceUnallocatedQuery, {
      approvalStatus: ApprovalStatusEnum.PENDING,
    }, searchText || '', 'registration');

    return await mahatriaChoiceUnallocatedQuery.getRawOne();
  }

  async getAllocatedMetrics(programId: number, allocatedProgramId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let allocatedQuery = this.dataSource
      .getRepository(ProgramRegistration)
      .createQueryBuilder('registration')
      .leftJoin('registration.user', 'user')
      .select([
        'COUNT(registration.id) AS "allocatedCount"',
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN 1 END) AS "maleCount"`,
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN 1 END) AS "femaleCount"`,
        `COUNT(CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "orgCount"`,
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "maleOrganizationUserCount"`,
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "femaleOrganizationUserCount"`,
        `COUNT(CASE WHEN registration.preferredRoomMate IS NOT NULL THEN 1 END) AS "roomMatePreferenceCount"`,
      ])
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id = :allocatedProgramId', {
        allocatedProgramId: allocatedProgramId
      })
      .andWhere('registration.deleted_at IS NULL');

    allocatedQuery = this.applyFiltersToQuery(allocatedQuery, {
      approvalStatus: ApprovalStatusEnum.APPROVED,
    }, searchText, 'registration');

    return await allocatedQuery.getRawOne();
  }

  async getNonAllocatedMetrics(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let nonAllocatedQuery = this.dataSource
      .getRepository(Preference)
      .createQueryBuilder('preference')
      .leftJoin('preference.registration', 'registration')
      .leftJoin('registration.user', 'user')
      .select([
        'COUNT(DISTINCT preference.id) AS "count"',
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN preference.id END) AS "maleCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN preference.id END) AS "femaleCount"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN preference.id END) AS "orgCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN preference.id END) AS "maleOrganizationUserCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN preference.id END) AS "femaleOrganizationUserCount"`
      ])
      .where('preference.preferred_program_id = :programId', {
        programId: programId
      })
      .andWhere('preference.priority_order = 1')
      .andWhere('registration.allocated_program_id IS NULL')
      .andWhere('preference.deleted_at IS NULL')
      .andWhere('registration.deleted_at IS NULL');

    nonAllocatedQuery = this.applyFiltersToQuery(nonAllocatedQuery, {
      approvalStatus: ApprovalStatusEnum.PENDING,
    }, searchText, 'registration');

    return await nonAllocatedQuery.getRawOne();
  }

  async getTotalRegistrations(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let totalRegistrationsQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.deleted_at IS NULL');

    totalRegistrationsQuery = this.applyFiltersToQuery(totalRegistrationsQuery, parsedFilters, searchText, 'registration');

    return await totalRegistrationsQuery.getCount();
  }
  async getTotalAllocatedRegistrations(programId: number, allocatedProgramId: number, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let totalAllocatedRegistrationsQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId })
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED })
      .andWhere('registration.deleted_at IS NULL');

    totalAllocatedRegistrationsQuery = this.applyUserFilterToQuery(totalAllocatedRegistrationsQuery, filterRegistrationsByUserId ?? null);
    totalAllocatedRegistrationsQuery = this.applyFiltersToQuery(totalAllocatedRegistrationsQuery, parsedFilters, searchText, 'registration');

    const [query, parameters] = totalAllocatedRegistrationsQuery.getQueryAndParameters();
    console.log('Generated Query:', query);
    console.log('Query Parameters:', parameters);

    return await totalAllocatedRegistrationsQuery.getCount();
}

  async getSeatsMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .select([
        `COUNT(DISTINCT registration.id) AS "all"`,
        `COUNT(CASE WHEN registration.registration_status = '${RegistrationStatusEnum.WAITLISTED}' THEN 1 END) AS "waitlisted"`,
        `COUNT(CASE WHEN registration.registration_status = '${RegistrationStatusEnum.CANCELLED}' THEN 1 END) AS "cancelled"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.PENDING}' THEN 1 END) AS "pending"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' THEN 1 END) AS "rejected"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' THEN 1 END) AS "onHold"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.APPROVED}' THEN 1 END) AS "approved"`,
        `COUNT(CASE WHEN registration.registration_status = '${RegistrationStatusEnum.COMPLETED}' THEN 1 END) AS "completed"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status != :saveAsDraft', { saveAsDraft: RegistrationStatusEnum.SAVE_AS_DRAFT });

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

   
  async getPaymentsMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .select([
        `COUNT(CASE WHEN payment.payment_status NOT IN ('${PaymentStatusEnum.DRAFT}', '${PaymentStatusEnum.SAVE_AS_DRAFT}') THEN payment.id END) AS "all"`,
        `COUNT(CASE WHEN payment.payment_status IN ('${PaymentStatusEnum.ONLINE_PENDING}', '${PaymentStatusEnum.OFFLINE_PENDING}') THEN payment.id END) AS "paymentPending"`,
        `COUNT(CASE WHEN payment.payment_status IN ('${PaymentStatusEnum.ONLINE_COMPLETED}', '${PaymentStatusEnum.OFFLINE_COMPLETED}') THEN payment.id END) AS "paymentCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('payment.deleted_at IS NULL');

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  async getInvoicesMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .select([
        `COUNT(CASE WHEN invoice.invoice_status NOT IN ('${InvoiceStatusEnum.DRAFT}', '${InvoiceStatusEnum.SAVE_AS_DRAFT}') THEN invoice.id END) AS "all"`,
        `COUNT(CASE WHEN invoice.invoice_status = '${InvoiceStatusEnum.INVOICE_PENDING}' THEN invoice.id END) AS "invoicePending"`,
        `COUNT(CASE WHEN invoice.invoice_status = '${InvoiceStatusEnum.INVOICE_COMPLETED}' THEN invoice.id END) AS "invoiceCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('invoice.deleted_at IS NULL');

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  async getTravelMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .leftJoinAndSelect('registration.travelPlans', 'travelPlan', 'travelPlan.deleted_at IS NULL')
      .select([
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status IN ('${TravelStatusEnum.PENDING}', '${TravelStatusEnum.COMPLETED}') THEN registration.id END) AS "all"`,
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status = '${TravelStatusEnum.PENDING}' THEN registration.id END) AS "travelPending"`,
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status = '${TravelStatusEnum.COMPLETED}' THEN registration.id END) AS "travelCompleted"`,
        `COUNT(DISTINCT CASE WHEN travelPlan.travel_plan_status = '${TravelStatusEnum.PENDING}' THEN registration.id END) AS "logisticsPending"`,
        `COUNT(DISTINCT CASE WHEN travelPlan.travel_plan_status = '${TravelStatusEnum.COMPLETED}' THEN registration.id END) AS "logisticsCompleted"`
      ])
      .where('registration.deleted_at IS NULL');

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  // ===== HELPER METHODS FOR FILTERS =====

  /**
   * Apply filters and search criteria to a query builder
   */
  private applyFiltersToQuery(
    queryBuilder: any,
    parsedFilters: Record<string, any> = {},
    searchText: string = '',
    tableAlias: string = 'registration'
  ): any {
    // registrationStatus filter
    const regStatus = parsedFilters?.registrationStatus;
    if (regStatus) {
      if (Array.isArray(regStatus)) {
        const validStatuses = regStatus.filter((s) => typeof s === 'string');
        if (validStatuses.length > 0) {
          queryBuilder.andWhere(`${tableAlias}.registration_status IN (:...registrationStatuses)`, {
            registrationStatuses: validStatuses
          });
        }
      } else if (typeof regStatus === 'string') {
        queryBuilder.andWhere(`${tableAlias}.registration_status = :registrationStatus`, {
          registrationStatus: regStatus
        });
      }
    }
  
    // createdBy filter
    if (parsedFilters?.createdBy) {
      queryBuilder.andWhere(`${tableAlias}.created_by = :createdBy`, {
        createdBy: parsedFilters.createdBy
      });
    }
  
    // approvalStatus filter
    const approvalStatus = parsedFilters?.approvalStatus;
    if (approvalStatus) {
      queryBuilder.leftJoin(`${tableAlias}.approvals`, 'approval_filter');
  
      if (Array.isArray(approvalStatus)) {
        const validApprovalStatuses = approvalStatus.filter((s) => typeof s === 'string');
        if (validApprovalStatuses.length > 0) {
          queryBuilder.andWhere('approval_filter.approval_status IN (:...approvalStatuses)', {
            approvalStatuses: validApprovalStatuses
          });
        }
      } else if (typeof approvalStatus === 'string') {
        queryBuilder.andWhere('approval_filter.approval_status = :approvalStatus', {
          approvalStatus
        });
      }
  
      queryBuilder.andWhere('approval_filter.deleted_at IS NULL');
    }
  
    // preferred program/session filters
    if (parsedFilters?.preferredProgramId || parsedFilters?.preferredSessionId) {
      queryBuilder.leftJoin(`${tableAlias}.preferences`, 'preference_filter');
      queryBuilder.andWhere('preference_filter.priority_order = 1');
      queryBuilder.andWhere('preference_filter.deleted_at IS NULL');
  
      if (parsedFilters.preferredProgramId) {
        queryBuilder.andWhere('preference_filter.preferred_program_id = :preferredProgramId', {
          preferredProgramId: parsedFilters.preferredProgramId
        });
      }
  
      if (parsedFilters.preferredSessionId) {
        queryBuilder.andWhere('preference_filter.preferred_session_id = :preferredSessionId', {
          preferredSessionId: parsedFilters.preferredSessionId
        });
      }
    }
  
    // searchText
    if (searchText && searchText.trim() !== '') {
      const searchTerm = `%${searchText.trim()}%`;
      queryBuilder.andWhere(
        `(${tableAlias}.full_name ILIKE :searchTerm OR ${tableAlias}.email_address ILIKE :searchTerm OR ${tableAlias}.mobile_number ILIKE :searchTerm)`,
        { searchTerm }
      );
    }
  
    return queryBuilder;
  }  

  private applyUserFilterToQuery(queryBuilder: any, userId: number | null): any {
    if (userId) {
      queryBuilder.andWhere('registration.rm_contact = :rmContactUserId', {
        rmContactUserId: userId
      });
    }
    return queryBuilder;
  }

  // ===== USER HISTORY QUERIES =====

  async findLastRegistrationByUserId(userId: number) {
    try {
      const registration = await this.registrationRepo.findOne({
        where: { userId, deletedAt: IsNull() },
        order: { registrationDate: 'DESC' },
        relations: [
          'invoiceDetails',
          'paymentDetails',
          'travelInfo',
          'travelPlans',
        ],
      });
      return registration;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async findLastInvoiceDetailByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationInvoiceDetail)
        .createQueryBuilder('invoice')
        .innerJoin('invoice.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('invoice.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(invoice.updatedAt, invoice.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_GET_FAILED, error);
    }
  }

  async findLastPaymentDetailByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationPaymentDetail)
        .createQueryBuilder('payment')
        .innerJoin('payment.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('payment.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(payment.updatedAt, payment.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_GET_FAILED, error);
    }
  }

  async findLastTravelPlanByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationTravelPlan)
        .createQueryBuilder('plan')
        .innerJoin('plan.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('plan.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(plan.updatedAt, plan.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_GET_FAILED, error);
    }
  }

  async findLastTravelInfoByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationTravelInfo)
        .createQueryBuilder('info')
        .innerJoin('info.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('info.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(info.updatedAt, info.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_GET_FAILED, error);
    }
  }

  /**
   * Find the approval status of a registration
   * @param registrationId - The ID of the registration
   * @returns The latest approval status of the registration
   */
  async findApprovalStatus(registrationId: number): Promise<RegistrationApproval | null> {
    try {
      return await this.approvalRepo.findOne({
        where: { 
          registrationId: registrationId,
          deletedAt: IsNull()
        },
        order: { createdAt: 'DESC' },
      });
    } catch (error) {
      this.logger.error('Failed to find approval status', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  private buildRegistrationData(params: any): any {
    const {
      dto,
      userId,
      isSelfRegister,
      registrationLevel,
      session,
      registrationStatus,
    } = params;
    
    const registrationData: any = {
      createdBy: { id: userId } as any,
      updatedBy: { id: userId } as any,
      registrationStatus: registrationStatus ?? RegistrationStatusEnum.DRAFT,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
    };

    // Set the appropriate ID field based on registration level
    if (registrationLevel === 'program') {
      registrationData.programId = dto.programId;
      registrationData.program = { id: dto.programId } as any;
      registrationData.programSessionId = null;
      registrationData.programSession = null;
    } else if (registrationLevel === 'session') {
      registrationData.programSessionId = session.id;
      registrationData.programSession = { id: session.id } as any;
      registrationData.programId = null;
      registrationData.program = null;
    }

    // If self registration, set userId on registration
    if (isSelfRegister) {
      registrationData.userId = userId;
    }

    return registrationData;
  }

  private async saveCustomResponses(queryRunner: any, customResponses: RegistrationCustomResponse[], registrationId: number): Promise<void> {
    for (const cr of customResponses) {
      cr.registrationId = registrationId;
      cr.registration = { id: registrationId } as any;
      await queryRunner.manager.save(RegistrationCustomResponse, cr);
    }
  }

  private async updateCustomResponses(queryRunner: any, customResponses: RegistrationCustomResponse[], registrationId: number, userId: number): Promise<void> {
    for (const cr of customResponses) {
      const existing = await queryRunner.manager.findOne(RegistrationCustomResponse, {
        where: { registrationId: registrationId, questionId: cr.questionId },
      });
      if (existing) {
        await queryRunner.manager.update(RegistrationCustomResponse, existing.id, {
          responseValue: cr.responseValue,
          updatedBy: { id: userId } as any,
        });
      } else {
        cr.registrationId = registrationId;
        cr.registration = { id: registrationId } as any;
        await queryRunner.manager.save(RegistrationCustomResponse, cr);
      }
    }
  }

  private async handleRegistrationOutcome(queryRunner: any, registrationId: number, params: any) {
    const { registrationOutcome, seatInfo, dto, program, session, registrationLevel, userId, registrationStatus } = params;
    if (registrationStatus == RegistrationStatusEnum.SAVE_AS_DRAFT) {
      await queryRunner.manager.update(ProgramRegistration, registrationId, {
        registrationStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
        basicDetailsStatus: RegistrationBasicStatusEnum.DRAFT,
      });
      return {
        registrationId,   
        waitlisted: false,
        pendingApproval: false,
        registrationLevel,
        registrationStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
        basicDetailsStatus: RegistrationBasicStatusEnum.DRAFT,
      }
    }

    switch (registrationOutcome.type) {
      case 'waitlist':
        return await this.handleWaitlistRegistration(queryRunner, registrationId, program, session?.id, seatInfo, registrationLevel);
      
      case 'no_space':
        await this.createRegistrationFailure(program, session, dto, userId, 'NO_SPACE');
        throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_NO_SPACE);
      
      case 'pending_approval':
        return await this.handleApprovalRegistration(queryRunner, registrationId, program, session?.id, userId, registrationLevel);
      
      case 'direct_registration':
        return await this.handleDirectRegistration(queryRunner, registrationId, program, session, registrationLevel, registrationOutcome, seatInfo);
      
      default:
        throw new Error('Unknown registration outcome type');
    }
  }

  private async handleWaitlistRegistration(
    queryRunner: any,
    registrationId: number,
    program: Program,
    sessionId: number | undefined,
    seatInfo: any,
    registrationLevel: string,
  ) {
    // Get next waitlist sequence number
    const nextWaitlistSeqNumber = await this.getNextWaitlistSeqNumber(program.id, sessionId);
    
    // Get next registration sequence number
    const nextRegSeqNumber = await this.getNextRegistrationSeqNumber(program.id, sessionId);
    
    // Generate formatted sequence numbers
    const formattedRegNumber = this.generateFormattedSeqNumber(program, nextRegSeqNumber, false);
    const formattedWaitlistNumber = this.generateFormattedSeqNumber(program, nextWaitlistSeqNumber, true);

    await queryRunner.manager.update(ProgramRegistration, registrationId, {
      waitingListSeqNumber: nextWaitlistSeqNumber,
      registrationSeqNumber: formattedRegNumber,
      waitListRegistrationSeqNumber: formattedWaitlistNumber, // New column
      basicDetailsStatus: RegistrationBasicStatusEnum.WAITLISTED,
      registrationStatus: RegistrationStatusEnum.WAITLISTED,
    });

    return {
      registrationId,
      registrationSeqNumber: formattedRegNumber,
      waitListRegistrationSeqNumber: formattedWaitlistNumber,
      waitlisted: true,
      waitlistPosition: nextWaitlistSeqNumber,
      registrationLevel,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
      registrationStatus: RegistrationStatusEnum.WAITLISTED,
      participantRemainingSeats: seatInfo.totalSeats - seatInfo.filledSeats,
    };
  }

  private async handleApprovalRegistration(
    queryRunner: any,
    registrationId: number,
    program: Program,
    sessionId: number | undefined,
    userId: number,
    registrationLevel: string,
  ) {
    const nextRegSeqNumber = await this.getNextRegistrationSeqNumber(program.id, sessionId);
    const formattedNumber = this.generateFormattedSeqNumber(program, nextRegSeqNumber, false);

    await queryRunner.manager.update(ProgramRegistration, registrationId, {
      registrationSeqNumber: formattedNumber,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
      registrationStatus: RegistrationStatusEnum.PENDING,
    });
    
    const approvalType = ApprovalStatusEnum.PENDING;
    const approval = queryRunner.manager.create(RegistrationApproval, {
      registrationId: registrationId,
      approvalStatus: ApprovalStatusEnum.PENDING,
      approvalType: approvalType,
      createdBy: { id: userId } as any,
      updatedBy: { id: userId } as any,
      registration: { id: registrationId } as any,
    });
    await queryRunner.manager.save(RegistrationApproval, approval);

    return {
      registrationId,
      registrationSeqNumber: formattedNumber,
      pendingApproval: true,
      approvalType: approvalType,
      registrationLevel,
      registrationStatus: RegistrationStatusEnum.PENDING,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
    };
  }

  private async handleDirectRegistration(
    queryRunner: any,
    registrationId: number,
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
    registrationOutcome: any,
    seatInfo: any,
  ) {
    const nextRegSeqNumber = await this.getNextRegistrationSeqNumber(program.id, session?.id);
    const formattedNumber = this.generateFormattedSeqNumber(program, nextRegSeqNumber, false);

    await queryRunner.manager.update(ProgramRegistration, registrationId, {
      registrationSeqNumber: formattedNumber,
      registrationStatus: registrationOutcome.finalStatus,
    });

    if (!registrationOutcome.requiresPayment) {
      await incrementSeatCounts(queryRunner.manager, program.id, session?.id);
    }

    return {
      registrationId,
      registrationSeqNumber: formattedNumber,
      registrationLevel,
      registrationStatus: registrationOutcome.finalStatus,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
      remainingSeats: seatInfo.waitlistTriggerCount - seatInfo.filledSeats,
      participantRemainingSeats: seatInfo.totalSeats - seatInfo.filledSeats,
    };
  }
  private async storeQuestionResponsesGrouped(
    groupedAnswers: Map<string, { columnName: string; value: any; question: Question }[]>,
    registrationId: number,
    queryRunner: any,
    registration: ProgramRegistration,
    registrationStatus?: RegistrationStatusEnum
  ): Promise<void> {
    const regWithProgram = await queryRunner.manager.findOne(ProgramRegistration, {
      where: { id: registrationId },
      relations: ['program'],
    });

    const programId = regWithProgram?.program?.id ?? registration.program?.id;
    const programInvolvesTravel = regWithProgram?.program?.involvesTravel ?? registration.program?.involvesTravel;
    const programRequiresPayment = regWithProgram?.program?.requiresPayment ?? registration.program?.requiresPayment;
    this.logger.log(`Program ID: ${programId}, Involves Travel: ${programInvolvesTravel}, Requires Payment: ${programRequiresPayment}`);

    const travelInfoTable = this.dataSource.getMetadata(RegistrationTravelInfo).tableName;
    const travelPlanTable = this.dataSource.getMetadata(RegistrationTravelPlan).tableName;
    const paymentDetailTable = this.dataSource.getMetadata(RegistrationPaymentDetail).tableName;
    const invoiceDetailTable = this.dataSource.getMetadata(RegistrationInvoiceDetail).tableName;

    // Move completion check outside the loop and modify it to handle transaction context
    const checkAndUpdateRegistrationCompletion = async (currentTableName?: string) => {
      if (registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT) {
        return;
      }

      this.logger.log(`Checking completion for registration ${registrationId}...`);
      
      // Get current registration status - use fresh query to ensure we get latest data
      const currentRegistration = await queryRunner.manager.findOne(ProgramRegistration, {
        where: { id: registrationId },
      });
      this.logger.log(`Current registration status: ${currentRegistration?.registrationStatus}`);
      
      // Only check completion for pending registrations
      if (currentRegistration?.registrationStatus !== RegistrationStatusEnum.PENDING) {
        return;
      }
      
      this.logger.log(`Registration ${registrationId} is pending, checking completion criteria...`);
      let shouldComplete = false;

      // Helper function to check if current table indicates completion
      const isCurrentTableCompleted = (tableName: string) => {
        if (!currentTableName) return false;
        
        // If we're currently processing travel info or travel plan tables, consider travel as completed
        if (tableName === travelInfoTable || tableName === travelPlanTable) {
          return currentTableName === travelInfoTable || currentTableName === travelPlanTable;
        }
        
        // If we're currently processing payment table, consider payment as completed
        if (tableName === paymentDetailTable) {
          return currentTableName === paymentDetailTable;
        }
        
        return false;
      };

      if (programInvolvesTravel && programRequiresPayment) {
        this.logger.log(`Program ${programId} involves travel and requires payment, checking both...`);
        
        // Check payment completion - look for any valid payment status OR if we're currently processing payment
        let paymentCompleted = false;
        if (isCurrentTableCompleted(paymentDetailTable)) {
          paymentCompleted = true;
          this.logger.log(`Payment completion assumed from current table processing`);
        } else {
          const paymentRecord = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
            where: {
              registrationId,
              paymentStatus: In([
                PaymentStatusEnum.ONLINE_COMPLETED,
                PaymentStatusEnum.OFFLINE_COMPLETED,
                PaymentStatusEnum.ONLINE_PENDING,
                PaymentStatusEnum.OFFLINE_PENDING,
              ]),
            },
          });
          paymentCompleted = !!paymentRecord;
        }

        // Check travel completion - look for completed travel info/plan OR if we're currently processing travel
        let travelCompleted = false;
        if (isCurrentTableCompleted(travelInfoTable) || isCurrentTableCompleted(travelPlanTable)) {
          travelCompleted = true;
          this.logger.log(`Travel completion assumed from current table processing`);
        } else {
          const travelInfoCompleted = await queryRunner.manager.findOne(RegistrationTravelInfo, {
            where: { registrationId, travelInfoStatus: TravelStatusEnum.COMPLETED },
          });

          const travelPlanCompleted = await queryRunner.manager.findOne(RegistrationTravelPlan, {
            where: { registrationId, travelPlanStatus: TravelStatusEnum.COMPLETED },
          });

          travelCompleted = !!travelInfoCompleted || !!travelPlanCompleted;
        }

        this.logger.log(`Payment completed: ${paymentCompleted}, Travel completed: ${travelCompleted}`);
        shouldComplete = paymentCompleted && travelCompleted;

      } else if (programInvolvesTravel) {
        this.logger.log(`Program ${programId} involves travel, checking travel info and plan...`);
        
        // Check travel completion - look for completed travel info/plan OR if we're currently processing travel
        let travelCompleted = false;
        if (isCurrentTableCompleted(travelInfoTable) || isCurrentTableCompleted(travelPlanTable)) {
          travelCompleted = true;
          this.logger.log(`Travel completion assumed from current table processing`);
        } else {
          const travelInfoCompleted = await queryRunner.manager.findOne(RegistrationTravelInfo, {
            where: { registrationId, travelInfoStatus: TravelStatusEnum.COMPLETED },
          });

          const travelPlanCompleted = await queryRunner.manager.findOne(RegistrationTravelPlan, {
            where: { registrationId, travelPlanStatus: TravelStatusEnum.COMPLETED },
          });

          travelCompleted = !!travelInfoCompleted || !!travelPlanCompleted;
        }

        this.logger.log(`Travel completed: ${travelCompleted}`);
        shouldComplete = travelCompleted;

      } else if (programRequiresPayment) {
        this.logger.log(`Program ${programId} requires payment, checking payment details...`);
        
        // Check payment completion - look for any valid payment status OR if we're currently processing payment
        let paymentCompleted = false;
        if (isCurrentTableCompleted(paymentDetailTable)) {
          paymentCompleted = true;
          this.logger.log(`Payment completion assumed from current table processing`);
        } else {
          const paymentRecord = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
            where: {
              registrationId,
              paymentStatus: In([
                PaymentStatusEnum.ONLINE_COMPLETED,
                PaymentStatusEnum.OFFLINE_COMPLETED,
                PaymentStatusEnum.ONLINE_PENDING,
                PaymentStatusEnum.OFFLINE_PENDING,
              ]),
            },
          });
          paymentCompleted = !!paymentRecord;
        }
        
        this.logger.log(`Payment completed: ${paymentCompleted}`);
        shouldComplete = paymentCompleted;
      } else {
        shouldComplete = true;
      }
      
      this.logger.log(`Should complete registration ${registrationId}: ${shouldComplete}`);
      if (shouldComplete) {
        await queryRunner.manager.update(
          ProgramRegistration,
          { id: registrationId },
          { registrationStatus: RegistrationStatusEnum.COMPLETED },
        );
        this.logger.log(`Registration ${registrationId} marked as completed`);
      }
    };

    // Process all table updates first
    for (const [tableName, answers] of groupedAnswers.entries()) {
      // Handle preferences via service
      if (tableName === this.dataSource.getMetadata(Preference).tableName) {
        let parsedFilters;
        if (answers[0]?.value) {
          parsedFilters = decodeURIComponent(answers[0].value);
          parsedFilters = JSON.parse(parsedFilters);
        }
  
        await this.handlePreferencesWithService(
          queryRunner.manager,
          parsedFilters,
          registrationId,
          programId,
          registration.createdBy.id,
          registration.updatedBy.id
        );
        continue;
      }
  
      // Get the entity name from the table name
      const entityMetadata = await queryRunner.manager.connection.entityMetadatas.find(
        metadata => metadata.tableName === tableName || `hdb_${metadata.tableName}` === tableName
      );
      if (!entityMetadata) {
        throw new Error(`Entity metadata not found for table: ${tableName}`);
      }
  
      const repository = queryRunner.manager.getRepository(entityMetadata.target);
      if (!repository) {
        throw new Error(`Repository not found for table: ${tableName}`);
      }
  
      const isRegistrationTable = tableName === this.dataSource.getMetadata(ProgramRegistration).tableName;
      const existing = await queryRunner.manager.findOne(repository.target, {
        where: isRegistrationTable ? { id: registrationId } : { registrationId: registrationId },
      });

  

    // if existing record then check for table name and corresponding status if it's sttaus is draft or save_as_draft then only new status as save_as_draft will be set
    if (registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT) {
      // Map table names to their status fields and enums
      const tableStatusMap = {
        [this.dataSource.getMetadata(ProgramRegistration).tableName]: {
          field: 'registrationStatus',
          enums: RegistrationStatusEnum,
          error: 'Cannot update registration with status other than draft or save_as_draft',
        },
        [travelInfoTable]: {
          field: 'travelInfoStatus',
          enums: TravelStatusEnum,
          error: 'Cannot update travel info with status other than draft or save_as_draft',
        },
        [travelPlanTable]: {
          field: 'travelPlanStatus',
          enums: TravelStatusEnum,
          error: 'Cannot update travel plan with status other than draft or save_as_draft',
        },
        [paymentDetailTable]: {
          field: 'paymentStatus',
          enums: PaymentStatusEnum,
          error: 'Cannot update payment detail with status other than draft or save_as_draft',
        },
        [invoiceDetailTable]: {
          field: 'invoiceStatus',
          enums: InvoiceStatusEnum,
          error: 'Cannot update invoice detail with status other than draft or save_as_draft',
        },
      };

      // Determine the key for the current table
      const tableKey = isRegistrationTable
        ? this.dataSource.getMetadata(ProgramRegistration).tableName
        : tableName;

      const statusConfig = tableStatusMap[tableKey];

      if (statusConfig) {
        const currentStatus = existing?.[statusConfig.field];
        if (
          !currentStatus || (
          currentStatus === statusConfig.enums.DRAFT ||
          currentStatus === statusConfig.enums.SAVE_AS_DRAFT)
        ) {
          this.logger.log("here in save as draft");
        } else {
          throw new InifniBadRequestException(
            ERROR_CODES.REGISTRATION_INVALID_STATUS,
            null,
            null,
            statusConfig.error
          );
        }
      }
    }
    
      
      // Data transformation logic
      if (tableName === travelInfoTable || tableName === travelPlanTable) {
        answers.forEach(a => {
          if (a.columnName === 'travel_type' || a.columnName === 'return_travel_type') {
            if (typeof a.value === 'string') {
              a.value = toSnakeCase(a.value.trim());
            }
            if (a.value.trim() === '') {
              a.value = null;
            }
          } else if (a.value.trim() === '') {
            a.value = null; // Ensure empty strings are set to null
          }
        });
      } else {
        answers.forEach(a => {
          if (a.value.trim() === '') {
            a.value = null; // Ensure empty strings are set to null
          }
        });
      }
      // Payment mode transformation
      if (tableName === paymentDetailTable) {
        answers.forEach(a => {
          if (a.columnName === 'payment_mode') {
            if (typeof a.value === 'string') {
              a.value = a.value.toLowerCase().includes('online') ? 'online' : 'offline';
            }
          }
        });
      }

      // Determine appropriate status based on table and registrationStatus
      const getStatusForTable = async (tableName: string, registrationStatus?: RegistrationStatusEnum) => {
        if (!registrationStatus) {
          return null;
        }

        if ([RegistrationStatusEnum.SAVE_AS_DRAFT].includes(registrationStatus)) {
          if (tableName === travelInfoTable) {
            return TravelStatusEnum.SAVE_AS_DRAFT;
          } else if (tableName === travelPlanTable) {
            return TravelStatusEnum.SAVE_AS_DRAFT;
          } else if (tableName === paymentDetailTable) {
            return PaymentStatusEnum.SAVE_AS_DRAFT;
          } else if (tableName === invoiceDetailTable) {
            return InvoiceStatusEnum.SAVE_AS_DRAFT;
          }
          return null;
        }
        return null;
      };

      if (existing || isRegistrationTable) {
        // UPDATE path
        let setClause = answers.map((a, i) => `${a.columnName} = $${i + 1}`).join(', ');
        let values = answers.map(a => a.value);
        
        // Add status columns based on table type and registrationStatus
        if (tableName === travelInfoTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            setClause += `, travel_info_status = $${answers.length + 1}`;
            values.push(status);
          } else {
            setClause += `, travel_info_status = $${answers.length + 1}`;
            values.push(TravelStatusEnum.COMPLETED);
          }
        } else if (tableName === travelPlanTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            setClause += `, travel_plan_status = $${answers.length + 1}`;
            values.push(status);
          } else {
            // Fix: should be travel_plan_status, not travel_info_status
            setClause += `, travel_plan_status = $${answers.length + 1}`;
            values.push(TravelStatusEnum.COMPLETED);
          }
        } else if (tableName === paymentDetailTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            setClause += `, payment_status = $${answers.length + 1}`;
            values.push(status);
          }
        } else if (tableName === invoiceDetailTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            setClause += `, invoice_status = $${answers.length + 1}`;
            values.push(status);
          }
        }

        const searchId = isRegistrationTable ? 'id' : 'registration_id';
        const searchParamIndex = `$${values.length + 1}`;
        await queryRunner.query(
          `UPDATE ${tableName} SET ${setClause} WHERE ${searchId} = ${searchParamIndex}`,
          [...values, registrationId]
        );
        this.logger.log('Batch updated existing record', { 
          tableName, 
          registrationId, 
          columns: answers.map(a => a.columnName),
          registrationStatus 
        });

        if (isRegistrationTable && registrationStatus != RegistrationStatusEnum.SAVE_AS_DRAFT && (existing.registrationStatus != RegistrationStatusEnum.PENDING && existing.registrationStatus != RegistrationStatusEnum.COMPLETED && existing.registrationStatus != RegistrationStatusEnum.DRAFT)) {
          await this.handleApprovalRegistration(
            queryRunner,
            registrationId,
            registration.program,
            registration.programSession?.id,
            registration.updatedBy.id,
            registration.program.registrationLevel
          );
        }
      } else {
        // INSERT path
        const columns = ['registration_id', ...answers.map(a => a.columnName)];
        const values = [registrationId, ...answers.map(a => a.value)];
        let placeholders = columns.map((_, i) => `$${i + 1}`).join(', ');
        
        // Add status columns for new records
        if (tableName === travelInfoTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            columns.push('travel_info_status');
            placeholders += `, $${columns.length}`;
            values.push(status);
          } else {
            columns.push('travel_info_status');
            placeholders += `, $${columns.length}`;
            values.push(TravelStatusEnum.COMPLETED);
          }
        } else if (tableName === travelPlanTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            columns.push('travel_plan_status');
            placeholders += `, $${columns.length}`;
            values.push(status);
          } else {
            columns.push('travel_plan_status');
            placeholders += `, $${columns.length}`;
            values.push(TravelStatusEnum.COMPLETED);
          }
        } else if (tableName === paymentDetailTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            columns.push('payment_status');
            placeholders += `, $${columns.length}`;
            values.push(status);
          }
        } else if (tableName === invoiceDetailTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
            columns.push('invoice_status');
            placeholders += `, $${columns.length}`;
            values.push(status);
          }
        }
        
        await queryRunner.query(
          `INSERT INTO ${tableName} (${columns.join(', ')}) VALUES (${placeholders})`,
          values
        );
        this.logger.log('Batch created new record', { 
          tableName, 
          registrationId, 
          columns: columns,
          registrationStatus 
        });
      }
      await checkAndUpdateRegistrationCompletion(tableName);
    }

  }

/**
   * Handle preferences using the preference service within a transaction
   * This replaces the direct database manipulation with proper service layer usage
   */
  private async handlePreferencesWithService(
    manager: EntityManager,
    preferences: any[],
    registrationId: number,
    programId: number,
    createdBy: number,
    updatedBy: number
  ): Promise<void> {
    try {
      // Validate preferences data
      if (!preferences || !Array.isArray(preferences) || preferences.length === 0) {
        this.logger.warn('No preferences provided or invalid preferences format', { 
          registrationId, 
          programId 
        });
        // If no preferences, we can still update the registration with an empty preference set
        const updatePreferenceDto: UpdatePreferenceDto = {
          programId: programId,
          preferences: [],
          updatedBy: updatedBy,
          createdBy: createdBy,
        };

        await this.preferenceService.update(
          registrationId,
          updatePreferenceDto,
          manager,
        );

        this.logger.log('No preferences to update, cleared preferences for registration', { 
          registrationId, 
          programId 
        });
        return;
      }

      // Prepare UpdatePreferenceDto for the preference service
      const updatePreferenceDto: UpdatePreferenceDto = {
        programId: programId,
        preferences: preferences.map((p, index) => ({
          sessionId: parseInt(p.sessionId),
          priorityOrder: p.priorityOrder ?? index + 1,
        })),
        updatedBy: updatedBy,
        createdBy: createdBy,
      };

      // Use the preference service's update method with the transaction manager
      // This will handle soft deletion of existing preferences and creation of new ones
      const result = await this.preferenceService.update(
        registrationId, 
        updatePreferenceDto, 
        manager
      );

      this.logger.log('Preferences handled via preference service', { 
        registrationId, 
        programId, 
        preferencesCount: preferences.length,
        createdPreferences: result.length
      });

    } catch (error) {
      this.logger.error('Error handling preferences via preference service', error);
      throw error;
    }
  }

  async upsertQuestionAnswers(
    queryRunner: any,
    registrationId: number,
    answers: { questionId: number; answerValue: any }[],
    userId: number,
  ): Promise<void> {
    for (const ans of answers) {
      const existing = await queryRunner.manager.findOne(RegistrationQuestionAnswer, {
        where: { registrationId, questionId: ans.questionId },
      });
      if (existing) {
        await queryRunner.manager.update(
          RegistrationQuestionAnswer,
          existing.id,
          { answerValue: ans.answerValue, updatedBy: { id: userId } as any },
        );
      } else {
        const entity = queryRunner.manager.create(RegistrationQuestionAnswer, {
          registrationId,
          questionId: ans.questionId,
          answerValue: ans.answerValue,
          createdBy: { id: userId } as any,
          updatedBy: { id: userId } as any,
        });
        await queryRunner.manager.save(RegistrationQuestionAnswer, entity);
      }
    }
  }

  private async updatePreferenceFromAnswer(
    value: UpdatePreferenceDto,
    registrationId: number,
    manager: EntityManager,
  ): Promise<void> {
    try {
      const dto = value;
      await this.preferenceService.update(registrationId, dto, manager);
    } catch (error) {
      this.logger.error('Error updating preferences from registration', error);
      throw error;
    }
  }

  async softDeleteRegistration(id: number, userId: number) {
    if (!id) {
      throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_ID_REQUIRED);
    }

    const registration = await this.registrationRepo.findOne({
      where: { id, deletedAt: IsNull() },
      relations: [
        'approvals',
        'program',
        'programSession',
        'allocatedProgram',
        'allocatedSession',
      ],
    });

    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        id.toString(),
      );
    }

    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();

    try {
      await queryRunner.startTransaction();

      const isApproved = registration.approvals?.some(
        (app) =>
          app.approvalStatus === ApprovalStatusEnum.APPROVED && !app.deletedAt,
      );

      if (
        registration.registrationStatus === RegistrationStatusEnum.COMPLETED ||
        isApproved
      ) {
        await decrementSeatCounts(
          queryRunner.manager,
          registration.program.id,
          registration.programSession?.id,
        );

        if (registration.allocatedProgram) {
          await decrementSeatCounts(
            queryRunner.manager,
            registration.allocatedProgram.id,
            registration.allocatedSession?.id,
          );
        }
      }

      await queryRunner.manager.update(ProgramRegistration, id, {
        deletedAt: new Date(),
        updatedBy: { id: userId } as any,
      });

      await queryRunner.commitTransaction();
      return { registrationId: id };
    } catch (error) {
      await queryRunner.rollbackTransaction();
      this.logger.error('Registration soft delete transaction failed', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  async findTravelPlanByRegistrationId(registrationId: number): Promise<RegistrationTravelPlan | null> {
    try {
      return await this.travelPlanRepo.findOne({
        where: { registrationId, deletedAt: IsNull() }
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_GET_FAILED, error);
    }
  } 

  async updateTravelPlanStatus(
    planId: number,
    status: TravelStatusEnum,
    userId: number
     ): Promise<void> {
    try { 
      await this.travelPlanRepo.update(
        { id: planId, deletedAt: IsNull() },
        { travelPlanStatus: status, updatedAt: new Date() , updatedBy: { id: userId } as any }

      );
      this.logger.log(`Travel plan status updated for registration ${planId} to ${status}`);
    }
    catch (error) {
      this.logger.error('Failed to update travel plan status', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_UPDATE_FAILED, error);
    } 
  }

  async createTravelPlan(
    registrationId: number,
    travelPlanStatus: TravelStatusEnum,
    userId: number
  ): Promise<RegistrationTravelPlan> {
    try {
      const travelPlan = this.travelPlanRepo.create({
        registrationId,
        travelPlanStatus: travelPlanStatus,
        createdBy: { id: userId } as any,
        updatedBy: { id: userId } as any,
        travelType: null, // Default to null, can be set later
      });
      return await this.travelPlanRepo.save(travelPlan);
    } catch (error) {
      this.logger.error('Failed to create travel plan', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_CREATE_FAILED, error);
    }
  }

  async findTravelInfoByRegistrationId(registrationId: number): Promise<RegistrationTravelInfo | null> {
    try {
      return await this.travelInfoRepo.findOne({
        where: { registrationId, deletedAt: IsNull() } 
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_GET_FAILED, error);
    }
  }
  async updateTravelInfoStatus(
    infoId: number,   
    status: TravelStatusEnum,
    userId: number
  ): Promise<void> {
    try {
      await this.travelInfoRepo.update(
        { id:infoId, deletedAt: IsNull() },
        { travelInfoStatus: status, updatedAt: new Date(), updatedBy: { id: userId } as any }
      );
      this.logger.log(`Travel info status updated for registration ${infoId} to ${status}`);
    } catch (error) {
      this.logger.error('Failed to update travel info status', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_UPDATE_FAILED, error);
    }
  }
  async createTravelInfo(
    registrationId: number,
    travelInfoStatus: TravelStatusEnum,   
    userId: number
  ): Promise<RegistrationTravelInfo> {
    try {
      const travelInfo = this.travelInfoRepo.create({
        registrationId,
        travelInfoStatus: travelInfoStatus,
        createdBy: { id: userId } as any,
        updatedBy: { id: userId } as any,
      });
      return await this.travelInfoRepo.save(travelInfo);
    } catch (error) {
      this.logger.error('Failed to create travel info', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_CREATE_FAILED, error);
    }
  }

  /**
   * Find save-as-draft registrations with search and basic filters
   */
  async findSaveDraftRegistrations(
    limit: number = 10,
    offset: number = 10,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    filterRegistrationsByUserId?: number | null,
    sortKey: string = 'id',
    sortOrder: 'ASC' | 'DESC' = 'ASC',
  ) {
    try {
      const whereClause: any = {
        deletedAt: IsNull(),
        registrationStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
      };

      if (programId) {
        const program = await this.programRepo.findOne({ where: { id: programId } });
        if (program?.registrationLevel === 'program') {
          whereClause.program = { id: programId };
          whereClause.programSession = IsNull();
        } else if (program?.registrationLevel === 'session') {
          if (programSessionId) {
            whereClause.programSession = { id: programSessionId };
          } else {
            whereClause.program = { id: programId };
          }
        } else {
          whereClause.program = { id: programId };
        }
      } else if (programSessionId) {
        whereClause.programSession = { id: programSessionId };
      }

      if (parsedFilters?.gender) {
        if (Array.isArray(parsedFilters.gender)) {
          whereClause.gender = In(parsedFilters.gender);
        } else {
          whereClause.gender = parsedFilters.gender;
        }
      }

      if (filterRegistrationsByUserId) {
        whereClause.rmContactUser = { id: filterRegistrationsByUserId };
      }

      let finalWhereClause: any = whereClause;

      // FIXED: Search functionality with role filtering
      if (searchText != null && searchText.trim() !== '') {
        const searchTerm = searchText.trim();

        // Create search conditions - each must include the role filter if present
        const searchConditions = [
          { ...whereClause, fullName: ILike(`%${searchTerm}%`) },
          { ...whereClause, emailAddress: ILike(`%${searchTerm}%`) },
          { ...whereClause, mobileNumber: ILike(`%${searchTerm}%`) },
        ];

        finalWhereClause = searchConditions;
      }

      const relations = ['program', 'program.type', 'programSession', 'rmContactUser'];

      const data = await this.commonDataService.get(
        this.registrationRepo,
        undefined,
        finalWhereClause,
        limit,
        offset,
        { [sortKey]: sortOrder },
        undefined,
        relations,
        { rmContactUser: ['id', 'fullName'] },
      );

      const total = await this.registrationRepo.count({ where: whereClause });

      const paginationInfo =
      {
        totalPages: Math.ceil(total / (limit as number)),
        pageNumber: Math.floor((offset as number) / (limit as number)) + 1,
        pageSize: limit as number,
        totalRecords: total,
        numberOfRecords: data.length,
      };

      return { data, pagination: paginationInfo };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }


/**
 * 
 * @param programId - The ID of the program for which to fetch registration metrics
 * @returns - An object containing various registration metrics for the specified program
 */
async getAllRegistrationMetrics(programId: number): Promise<any> {
    try {
      const approvalStatusesForAllKpis = [
        ApprovalStatusEnum.REJECTED,
        ApprovalStatusEnum.PENDING,
        ApprovalStatusEnum.ON_HOLD,
      ];
      let allKpis = {
        totalRegistrations: 0,
        femaleRegistrations: 0,
        maleRegistrations: 0,
        mahatriaChoicesRegistrations: 0,
      };
      const totalAllCountQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .select(`COUNT(registration.id)`, 'totalRegistrations')
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.registration_status != :saveAsDraftStatus')
        .andWhere('registration.program_id = :programId', { programId })
        .setParameters({
          saveAsDraftStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
        });

      const totalAll = await totalAllCountQuery.getRawOne();
      allKpis.totalRegistrations = parseInt(totalAll.totalRegistrations) || 0;

      const allGenderCountQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .select(
          `COUNT(CASE WHEN registration.gender = 'female' THEN registration.id END)`,
          'femaleRegistrations',
        )
        .addSelect(
          `COUNT(CASE WHEN registration.gender = 'male' THEN registration.id END)`,
          'maleRegistrations',
        )
        .leftJoin('registration.approvals', 'approval')
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status != :saveAsDraftStatus')
        // .andWhere('approval.approval_status IN (:...statuses)')
        .setParameters({
          saveAsDraftStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
          // statuses: approvalStatusesForAllKpis,
        });
      const allGenderCount = await allGenderCountQuery.getRawOne();
      allKpis.femaleRegistrations = parseInt(allGenderCount.femaleRegistrations) || 0;
      allKpis.maleRegistrations = parseInt(allGenderCount.maleRegistrations) || 0;

      const allMahatriaChoicesCountQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .select('COUNT(registration.id)', 'mahatriaChoicesRegistrations')
        .leftJoin('registration.approvals', 'approval')
        .leftJoin('registration.preferences', 'preference')
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status != :saveAsDraftStatus')
        .andWhere('approval.approval_status IN (:...statuses)')
        .andWhere('preference.id IS NULL')
        .setParameters({
          saveAsDraftStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
          statuses: approvalStatusesForAllKpis,
        });

      const allMahatriaChoicesCount = await allMahatriaChoicesCountQuery.getRawOne();
      allKpis.mahatriaChoicesRegistrations =
        parseInt(allMahatriaChoicesCount.mahatriaChoicesRegistrations) || 0;

      const subProgramPreferredRegistrationsQuery = await this.programRepo
        .createQueryBuilder('subProgram')
        .select('subProgram.id', 'subProgramId')
        .addSelect('subProgram.name', 'subProgramName')
        .addSelect('ARRAY_AGG(DISTINCT registration.id)', 'registrationIds')
        .leftJoin('hdb_preference', 'preference', 'preference.preferred_program_id = subProgram.id ')
        .leftJoin(
          'hdb_program_registration',
          'registration',
          'registration.id = preference.registration_id',
        )
        .leftJoin('registration.approvals', 'approval')
        .where('subProgram.primary_program_id = :programId', { programId })
        .andWhere('registration.deletedAt IS NULL')
        .andWhere('registration.registration_status != :saveAsDraftStatus')
        .andWhere('approval.approval_status IN (:...statuses)')
        .andWhere('preference.priority_order = 1')
        .setParameters({
          statuses: approvalStatusesForAllKpis,
          saveAsDraftStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
        })
        .groupBy('subProgram.id')
        .addGroupBy('subProgram.name');

      const subProgramPreferredRegistrations =
        await subProgramPreferredRegistrationsQuery.getRawMany();
      // const subProgramSwapRequestedRegistrationsQuery = this.programRepo
      //   .createQueryBuilder('subProgram')
      //   .select('subProgram.id', 'subProgramId')
      //   .addSelect('subProgram.name', 'subProgramName')
      //   .addSelect('ARRAY_AGG(DISTINCT registration.id)', 'registrationIds')
      //   .leftJoin('subProgram.allocatedProgramRegistrations', 'registration')
      //   .leftJoin('registration.approvals', 'approval')
      //   .leftJoin('registration.swapsRequests', 'swap')
      //   .where('subProgram.primary_program_id = :programId')
      //   .andWhere('"registration"."deleted_at" IS NULL')
      //   .andWhere('"approval"."approval_status" = :approvalStatus')
      //   .andWhere('swap.id IS NOT NULL')
      //   .andWhere('swap.status = :swapStatus')
      //   .setParameters({
      //     approvalStatus: ApprovalStatusEnum.APPROVED,
      //     swapStatus: SwapRequestStatus.ACTIVE,
      //     programId: programId,
      //   })
      //   .groupBy('subProgram.id')
      //   .addGroupBy('subProgram.name');

      // const subProgramSwapRequestedRegistrations =
      //   await subProgramSwapRequestedRegistrationsQuery.getRawMany();
      const subProgramMetrics: {
        subProgramId: number;
        subProgramName: string;
        count: number;
      }[] = [];

      const allSubPrograms = await this.getSubPrograms(programId);

      for (const subProgram of allSubPrograms) {
        const preferred = subProgramPreferredRegistrations.filter(
          (item) => item.subProgramId === subProgram.id,
        )[0];
        // const swap = subProgramSwapRequestedRegistrations.filter(
        //   (item) => item.subProgramId === subProgram.id,
        // )[0];

        const allIds = new Set<string>();
        if (preferred?.registrationIds) {
          for (const id of preferred.registrationIds) allIds.add(id);
        }
        // if (swap?.registrationIds) {
        //   for (const id of swap.registrationIds) allIds.add(id);
        // }
        subProgramMetrics.push({
          subProgramId: subProgram.id,
          subProgramName: subProgram.name,
          count: allIds.size,
        });
      }
      return {
        ...allKpis,
        subProgramMetrics,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  // update registration status
  async updateRegistrationStatus(
    queryRunner: any,
    registrationId: number,
    status: RegistrationStatusEnum,
    userId: number
  ): Promise<void> {
    const registration = await queryRunner.manager.findOne(ProgramRegistration, {
      where: { id: registrationId, deletedAt: IsNull() },
      relations: ['program', 'programSession'],
    });

    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }

    console.log(`Updating registration ${registrationId} status to ${status} ${queryRunner}`);
    // Update the registration status
    await queryRunner.manager.update(ProgramRegistration, registrationId, {
      registrationStatus: status,
      updatedBy: { id: userId } as any,
    });

    this.logger.log(`Registration ${registrationId} status updated to ${status}`);
  }

  async updateApprovalStatus(
    registrationId: number,
    status: ApprovalStatusEnum,
    userId: number,
  ) {
    try {
      const approval = await this.approvalRepo.findOne({
        where: { registrationId, deletedAt: IsNull() },
        order: { createdAt: 'DESC' },
      });

      if (!approval) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND,
          null,
          null,
          registrationId.toString(),
        );
      }

      await this.approvalRepo.update(
        { id: approval.id },
        { approvalStatus: status, updatedBy: { id: userId } as any },
      );
      return await this.approvalRepo.findOne({ where: { id: approval.id } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }
}