import {
  Body,
  Controller,
  HttpCode,
  HttpStatus,
  Post,
  Get,
  Put,
  Res,
  UseGuards,
  ValidationPipe,
  Req,
  Query,
  Param,
  ParseIntPipe,
  Delete,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiResponse, ApiSecurity, ApiTags, ApiQuery, ApiParam } from '@nestjs/swagger';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { registrationMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { RegistrationService } from './registration.service';
import { CreateRegistrationDto } from './dto/create-registration.dto';
import { UpdateRegistrationDto } from './dto/update-registration.dto';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { MainProgramDimensions, RoleKPIs, SubProgramDimensions } from 'src/common/constants/constants';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('registration')
@Controller('registration')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'viewer','mahatria', 'rm', 'finance_manager', 'relational_manager', 'shoba', 'operational_manger')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class RegistrationController {
  constructor(
    private readonly service: RegistrationService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) { }

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: registrationMessages.CREATE_REQUEST_RECEIVED })
  @ApiBody({ type: CreateRegistrationDto })
  @ApiResponse({
    status: HttpStatus.CREATED,
    description: SWAGGER_API_RESPONSE.CREATED,
    schema: {
      properties: {
        registrationId: { type: 'number' },
        pendingApproval: { type: 'boolean' },
        approvalType: { type: 'string' },
        registrationLevel: { type: 'string' },
        registrationStatus: { type: 'string' }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async register(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateRegistrationDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    const userId = user.id;
    const isSelfRegister = dto.isSelfRegister === true;
    this.logger.log(registrationMessages.CREATE_REQUEST_RECEIVED, dto);
    
    try {
      const data = await this.service.register(dto, userId, isSelfRegister);
      
      // Determine message based on registration outcome
      let message = registrationMessages.REGISTRATION_SUCCESS;
      
      if (data.pendingApproval) {
        message = `${registrationMessages.REGISTRATION_PENDING_APPROVAL} (Type: ${data.approvalType})`;
      } else if (data.waitlisted) {
        message = registrationMessages.REGISTRATION_WAITLISTED;
      }
      
      await this.responseService.success(res, message, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id/statuses')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_STATUSES })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getStatuses(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    try {
      const data = await this.service.getRegistrationStatuses(id);
      await this.responseService.success(res, registrationMessages.STATUSES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.UPDATE_REQUEST_RECEIVED })
  @ApiBody({ type: UpdateRegistrationDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async updateRegistration(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateRegistrationDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    this.logger.log(registrationMessages.UPDATE_REQUEST_RECEIVED, dto);
    
    try {
      const data = await this.service.update(dto, user);
      
      // Determine message based on update outcome
      let message = registrationMessages.REGISTRATION_UPDATED;
      
      if (data.cancelled) {
        message = 'Registration cancelled successfully';
      } else if (data.registrationStatus === RegistrationStatusEnum.PENDING) {
        message = registrationMessages.REGISTRATION_PENDING_APPROVAL;
      } else if (data.waitlisted === true) {
        message = registrationMessages.REGISTRATION_WAITLISTED;
      }
      
      await this.responseService.success(res, message, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }


  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_ALL })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];
      
      const data = await this.service.findAll(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null
      );
      const allowedSaveDraftRoles = ['viewer'];
      if (data) {
        // for each registration make registration status as pending_approval if it is in pendinG state and approovals status is pending
        data.data.forEach(registration => {
          if (registration.registrationStatus === RegistrationStatusEnum.PENDING && registration.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
            registration.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
          }
          if (!userRoles.some(role => allowedSaveDraftRoles.includes(role.name))) {
             // if travelPlanStatus map contains save_as_draft, then set travelPlanStatus to [] remove that same in travelInfo also
            if (registration.travelPlans && registration.travelPlans.length> 0 && registration.travelPlans.some(tp => tp.travelPlanStatus === 'save_as_draft')) {
              registration.travelPlans = registration.travelPlans?.filter(tp => tp.travelPlanStatus !== 'save_as_draft');
            }
            if (registration.travelInfo && registration.travelInfo.length > 0 && registration.travelInfo.some(ti => ti.travelInfoStatus === 'save_as_draft')) {
              registration.travelInfo = registration.travelInfo?.filter(ti => ti.travelInfoStatus !== 'save_as_draft');
            }
         }
        }
      );
      }
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('list-view')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get registrations list with role-based KPIs and data filtering' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiQuery({ name: 'downloadType', type: String, required: false, description: 'Download type: "all" or "filtered"' })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Registrations list with role-based KPIs retrieved successfully',
    schema: {
      properties: {
        data: { 
          type: 'array',
          description: 'Flat array of registration records',
          items: {
            type: 'object',
            properties: {
              id: { type: 'number' },
              seekerName: { type: 'string' },
              age: { type: 'number' },
              gender: { type: 'string' },
              location: { type: 'string' },
              blessedWith: { type: 'string' },
              numberOfHDBs: { type: 'number' },
              rmComments: { type: 'string' },
              paymentStatus: { type: 'string' },
              travelPlanStatus: { type: 'string' },
              mobileNumber: { type: 'string' },
              registrationStatus: { type: 'string' }
            }
          }
        },
        tableHeaders: {
          type: 'array',
          description: 'Table column headers configuration',
          items: {
            type: 'object',
            properties: {
              key: { type: 'string', description: 'Data field key' },
              label: { type: 'string', description: 'Display label' },
              sortable: { type: 'boolean', description: 'Whether column is sortable' },
              filterable: { type: 'boolean', description: 'Whether column is filterable' },
              type: { type: 'string', description: 'Data type for formatting' }
            }
          }
        },
        pagination: {
          type: 'object',
          description: 'Pagination information',
          properties: {
            totalPages: { type: 'number' },
            pageNumber: { type: 'number' },
            pageSize: { type: 'number' },
            totalRecords: { type: 'number' },
            numberOfRecords: { type: 'number' }
          }
        },
        kpis: {
          type: 'array',
          description: 'Flat KPI array for easy rendering',
          items: {
            type: 'object',
            properties: {
              label: { type: 'string' },
              value: { type: 'number' },
              kpiCategory: { type: 'string' },
              kpiFilter: { type: 'string' },
              category: { type: 'string' },
              categoryLabel: { type: 'string' }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getRegistrationsListView(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Req() req: any,
    @Res() res: Response,
    @Query('downloadType') downloadType?: string,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];
      
      const result = await this.service.getRegistrationsListView(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null,
        downloadType 
      );
  
      // If it's a download request, generate Excel and return S3 URL
      if (downloadType) {
        const excelUrl = await this.service.generateExcelDownload(result.data, userRoles, downloadType);
        
        await this.responseService.success(res, 'Excel file generated successfully', {
          downloadUrl: excelUrl,
          totalRecords: result.pagination.totalRecords,
          generatedAt: new Date().toISOString()
        });
        return;
      }

      // Regular response for non-download requests
      const transformedData = await this.service.transformRegistrationListData(result, userRoles);
      
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
  
  @Get('mahatria-kpis')
  @HttpCode(HttpStatus.OK)
  @Roles('mahatria', 'admin', 'shoba') // Restrict to mahatria and admin roles only
  @ApiOperation({ summary: 'Get Mahatria-specific KPIs for program allocation management with registration data' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: true, description: 'Program ID for KPI generation' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text for filtering' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters for KPI calculation' })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Mahatria KPIs with registration data retrieved successfully',
    schema: {
      properties: {
        data: { 
          type: 'array',
          description: 'Array of registration records (filtered for mahatria role)'
        },
        pagination: {
          type: 'object',
          description: 'Pagination information',
          properties: {
            totalPages: { type: 'number' },
            pageNumber: { type: 'number' },
            pageSize: { type: 'number' },
            totalRecords: { type: 'number' },
            numberOfRecords: { type: 'number' }
          }
        },
        kpis: {
          type: 'object',
          properties: {
            mahatria: {
              type: 'object',
              properties: {
                groupedProgramMetrics: {
                  type: 'object',
                  properties: {
                    isGroupedProgram: { type: 'boolean' },
                    primaryProgramId: { type: 'number' },
                    primaryProgramName: { type: 'string' },
                    allocated: {
                      type: 'object',
                      properties: {
                        categoryLabel: { type: 'string' },
                        programs: { type: 'array' },
                        totals: {
                          type: 'object',
                          properties: {
                            holdCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            rejectCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            }
                          }
                        }
                      }
                    },
                    unallocated: {
                      type: 'object',
                      properties: {
                        categoryLabel: { type: 'string' },
                        programs: { type: 'array' },
                        totals: {
                          type: 'object',
                          properties: {
                            totalMaleCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalFemaleCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalUnallocatedCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalMahatriaChoiceCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalOrganisationUserCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.FORBIDDEN, description: 'Access denied - Mahatria role required' })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getMahatriaKPIs(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId', ParseIntPipe) programId: number,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      
      const data = await this.service.getMahatriaKPIs(
        +limit,
        +offset,
        programId,
        searchText,
        parsedFilters,
        user
      );

      // Apply the same registration status transformation as list-view
      if (data && data.data) {
        data.data.forEach(registration => {
          if (registration.registrationStatus === RegistrationStatusEnum.PENDING && 
              registration.approvals && 
              registration.approvals[0] && 
              registration.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
            registration.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
          }
        });
      }
      
      const transformedData = await this.service.transformMahatriaKPIData(data, user.roles);
      const responseData = {
        data: transformedData,
        pagination: data.pagination,
        kpis: data.kpis,
        filters: data.filters,
      };
      await this.responseService.success(res, 'Mahatria KPIs with registration data retrieved successfully', responseData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('drafts')
  @HttpCode(HttpStatus.OK)
  @Roles('relational_manager', 'shoba', 'admin')
  @ApiOperation({ summary: 'Get save-as-draft registrations with search and filters' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiQuery({
    name: 'sortKey',
    type: String,
    required: false,
    description: 'Field name to sort by (matches table header key)',
  })
  @ApiQuery({
    name: 'sortOrder',
    enum: ['ASC', 'DESC'],
    required: false,
    description: 'Sort direction (ASC or DESC)',
  })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getSaveDraftRegistrations(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Query('sortKey') sortKey = 'id',
    @Query('sortOrder') sortOrder: 'ASC' | 'DESC' = 'ASC',
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];

      const data = await this.service.getSaveDraftRegistrations(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null,
        sortKey,
        sortOrder,
      );

      const transformedDataDraft = await this.service.transformDraftRegistrationListData(data, userRoles);
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedDataDraft);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_BY_ID })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response, @Req() req: any) {
    const user = req.user;
    const userRoles = user.userRoleMaps;
    const isMahatria = userRoles.some(urm => urm.role.name === 'mahatria');
    const viewer = userRoles.some(urm => urm.role.name === 'viewer');
    try {
      const data = await this.service.findOne(id);
      let finalData: any = data;
      if (finalData) {
        // if registration is in pending state and approval status is pending, then set registration status to pending_approval
        if (finalData.registrationStatus === RegistrationStatusEnum.PENDING && finalData.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
          finalData.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
        }
      }
      if (data && isMahatria) {
        // fetch user filled questions for Mahatria's Questions section
        const questionResponses = await this.service.getQuestionResponses(id);
        const filteredResponses =
          questionResponses.filter(qr => qr.sectionKey === 'FS_MAHATRIAQUESTIONS');
        finalData = {
          ...data,
          questionResponses: filteredResponses,
        };
      }

      const allowedSaveDraftRoles = ['viewer'];
      if (data && !userRoles.some(role => allowedSaveDraftRoles.includes(role.role.name))) {
        let hasSaveAsDraft =  false
        // if travelPlanStatus map contains save_as_draft, then set travelPlanStatus to [] remove that same in travelInfo also
        if (finalData.travelPlans && finalData.travelPlans.length> 0 && finalData.travelPlans.some(tp => tp.travelPlanStatus === 'save_as_draft')) {
          finalData.travelPlans = finalData.travelPlans?.filter(tp => tp.travelPlanStatus !== 'save_as_draft');
          hasSaveAsDraft = true;
        }
        if (finalData.travelInfo && finalData.travelInfo.length > 0 && finalData.travelInfo.some(ti => ti.travelInfoStatus === 'save_as_draft')) {
          finalData.travelInfo = finalData.travelInfo?.filter(ti => ti.travelInfoStatus !== 'save_as_draft');
           hasSaveAsDraft = true;
        }
        // filter section "FS_TRAVELPLAN" then ignore that questions if any of these status is save_as_draft
        if (hasSaveAsDraft && finalData.questionResponses && finalData.questionResponses.length > 0) {
          finalData.questionResponses = finalData.questionResponses?.filter(qr => qr.sectionKey !== 'FS_TRAVELPLAN');
        }
      }
      if (viewer) {
        delete finalData.swapsRequests;
        delete finalData.ratings
      }
      await this.responseService.success(res, registrationMessages.REGISTRATION_RETRIEVED, finalData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id/questions')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_QA_REQUEST_RECEIVED })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getQuestionResponses(@Param('id', ParseIntPipe) id: number, @Res() res: Response, @Req() req: any) {
    try {
      const registration = await this.service.findOne(id);
      if (!registration) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_NOT_FOUND, null, null, `Registration with ID ${id} not found`);
      }
      let data = await this.service.getQuestionResponses(id);
      // Only allow 'mahatria', 'admin', or 'viewer' to see section 4 questions
      const allowedRoles = ['mahatria', 'admin', 'viewer', 'shoba'];
      if (
        req.user &&
        !req.user.userRoleMaps.some((role) => allowedRoles.includes(role.role.name))
      ) {
        // filter out the questions with section 4
        data = data.filter(question => question.sectionKey != "FS_MAHATRIAQUESTIONS");
      }

      const allowedSaveDraftRoles = ['viewer'];
      if (
        req.user &&
        !req.user.userRoleMaps.some((role) => allowedSaveDraftRoles.includes(role.role.name))
      ) {
        let hasSaveAsDraft =  false
        // if travelPlanStatus map contains save_as_draft, then set travelPlanStatus to [] remove that same in travelInfo also
        if (registration.travelPlans && registration.travelPlans.length> 0 && registration.travelPlans.some(tp => tp.travelPlanStatus === 'save_as_draft')) {
          registration.travelPlans = registration.travelPlans?.filter(tp => tp.travelPlanStatus !== 'save_as_draft');
          hasSaveAsDraft = true;
        }
        if (registration.travelInfo && registration.travelInfo.length > 0 && registration.travelInfo.some(ti => ti.travelInfoStatus === 'save_as_draft')) {
          registration.travelInfo = registration.travelInfo?.filter(ti => ti.travelInfoStatus !== 'save_as_draft');
           hasSaveAsDraft = true;
        }
        if (hasSaveAsDraft && data && data.length > 0) {
         data = data?.filter(question => question.sectionKey != "FS_TRAVELPLAN");
        }
      }
      // filter empty question responses
      data = data.filter(question => question && question.answer.trim() !== '');
      await this.responseService.success(res, registrationMessages.QA_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get dashboard data for a specific program
   * @param programId The ID of the program
   * @param res The response object
   * @returns The dashboard data for the program
   */
  @Get('program/:programId/dashboard')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_PROGRAM_DASHBOARD })
  @ApiParam({ name: 'programId', description: 'Program ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findByProgramId(@Param('programId', ParseIntPipe) programId: number, @Res() res: Response, @Req() req: any) {
    try {
      const userRoles = req.user?.roles
      const mainProgramDimensions = RoleKPIs[userRoles[0]] || MainProgramDimensions;
      const subProgramDimensions = [];
      const data = await this.service.getProgramDashboardData({programId, mainProgramDimensions, subProgramId: undefined, subProgramDimensions, rmId: userRoles.includes('relational_manager') ? req.user.id : undefined, userRole: userRoles[0]});
      await this.responseService.success(res, registrationMessages.DASHBOARD_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get dashboard data for a specific program
   * @param programId The ID of the program
   * @param subProgramId The ID of the sub-program
   * @param res The response object
   */
  @Get('program/:programId/dashboard/sub-program/:subProgramId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_SUB_PROGRAM_DASHBOARD })
  @ApiParam({ name: 'programId', description: 'Program ID', type: Number })
  @ApiParam({ name: 'subProgramId', description: 'Sub Program ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findBySubProgramId(@Param('programId', ParseIntPipe) programId: number, @Param('subProgramId', ParseIntPipe) subProgramId: number, @Res() res: Response, @Req() req: any) {
    try {
      const userRoles = req.user?.roles
      const mainProgramDimensions = RoleKPIs[userRoles[0]] || MainProgramDimensions;
      const subProgramDimensions = SubProgramDimensions;
      const data = await this.service.getProgramDashboardData({programId, mainProgramDimensions, subProgramId, subProgramDimensions, rmId: userRoles.includes('relational_manager') ? req.user.id : undefined, userRole: userRoles[0]});
      await this.responseService.success(res, registrationMessages.DASHBOARD_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.DELETE_BY_ID })
  @ApiParam({ name: 'id', description: registrationMessages.REGISTRATION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    try {
      const user = req.user;
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
      }
      await this.service.remove(id, user);
      await this.responseService.success(res, registrationMessages.REGISTRATION_DELETED, { id });
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
