import { Injectable } from '@nestjs/common';
import { CreateRegistrationTravelInfoDto } from './dto/create-registration-travel-info.dto';
import { UpdateRegistrationTravelInfoDto } from './dto/update-registration-travel-info.dto';
import { RegistrationTravelInfoRepository } from './registration-travel-info.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationTravelInfoMessages } from 'src/common/constants/strings-constants';

@Injectable()
export class RegistrationTravelInfoService {
  constructor(
    private readonly repo: RegistrationTravelInfoRepository,
    private readonly logger: AppLoggerService,
  ) {}

  async create(dto: CreateRegistrationTravelInfoDto) {
    this.logger.log(registrationTravelInfoMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.repo.createEntity(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(registrationTravelInfoMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationTravelInfoMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationTravelInfoDto) {
    this.logger.log(registrationTravelInfoMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.repo.updateEntity(id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(registrationTravelInfoMessages.DELETE_REQUEST_RECEIVED);
    try {
      await this.repo.remove(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_DELETE_FAILED, error);
    }
  }
}
