import { Injectable } from '@nestjs/common';
import { CreateRegistrationCustomResponseDto } from './dto/create-registration-custom-response.dto';
import { UpdateRegistrationCustomResponseDto } from './dto/update-registration-custom-response.dto';
import { RegistrationCustomResponseRepository } from './registration-custom-response.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationCustomResponseMessages } from 'src/common/constants/strings-constants';

@Injectable()
export class RegistrationCustomResponseService {
  constructor(
    private readonly repo: RegistrationCustomResponseRepository,
    private readonly logger: AppLoggerService,
  ) {}

  async create(dto: CreateRegistrationCustomResponseDto) {
    this.logger.log(registrationCustomResponseMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.repo.createEntity(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(registrationCustomResponseMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationCustomResponseMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationCustomResponseDto) {
    this.logger.log(registrationCustomResponseMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.repo.updateEntity(id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(registrationCustomResponseMessages.DELETE_REQUEST_RECEIVED);
    try {
      await this.repo.remove(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_DELETE_FAILED, error);
    }
  }
}
