import { Injectable } from '@nestjs/common';
import { CreateRegistrationApprovalDto } from './dto/create-registration-approval.dto';
import { UpdateRegistrationApprovalDto } from './dto/update-registration-approval.dto';
import { RegistrationApprovalRepository } from './registration-approval.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationApprovalMessages, zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { CommunicationService } from 'src/communication/communication.service';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { amountInWordsIndian, deductDaysFromDate, formatDate, getWeekName } from 'src/common/utils/common.util';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { UserRepository } from 'src/user/user.repository';
import { ProFormaInvoice } from 'src/common/templates/dtos/templates.dto';
import { UPDATED_PRO_FORMA_INVOICE } from 'src/common/templates/templates';
import axios from 'axios';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { v4 as uuidv4 } from 'uuid';
import { RegistrationRepository } from 'src/registration/registration.repository';
import { CommonDataService } from 'src/common/services/commonData.service';
import { ProgramRegistration } from 'src/common/entities';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import { InjectDataSource, InjectRepository } from '@nestjs/typeorm';
import { DataSource, QueryRunner, Repository } from 'typeorm';
import { SEEKER_FE_REG_PATH } from 'src/common/constants/constants';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import chromium from "@sparticuz/chromium-min";

const puppeteer = require('puppeteer-core');
// const chromium = require('@sparticuz/chromium');

@Injectable()
export class RegistrationApprovalService {
  constructor(
    private readonly repo: RegistrationApprovalRepository,
    private readonly logger: AppLoggerService,
    private readonly communicationService: CommunicationService,
    private readonly userRepo: UserRepository,
    private readonly commonService: CommonDataService,
    private readonly awsS3Service: AwsS3Service,
    @InjectRepository(ProgramRegistration) 
    private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectDataSource()
    private readonly dataSource: DataSource,
  ) {}

  async create(dto: CreateRegistrationApprovalDto) {
    this.logger.log(registrationApprovalMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.repo.createEntity(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number, programId?: number, search?: string, filters: Record<string, any> = {}) {
    this.logger.log(registrationApprovalMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset, programId, search, filters);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationApprovalMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationApprovalDto) {
    this.logger.log(registrationApprovalMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.repo.updateEntity(id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async updateByRegistrationId(registrationId: number, dto: UpdateRegistrationApprovalDto) {
    this.logger.log('Update approval by registrationId', { registrationId, dto });
    try {
      return await this.repo.updateByRegistrationId(registrationId, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(registrationApprovalMessages.DELETE_REQUEST_RECEIVED);
    try {
      await this.repo.remove(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_DELETE_FAILED, error);
    }
  }

  async findOneByRegistrationId(registrationId: number) {
    this.logger.log('Find approval by registrationId', { registrationId });
    try {
      return await this.repo.findByRegistrationId(registrationId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, error);
    }
  }
  
  async sendApprovalEmailNotification(approvalId: number, existingApprovalStatus?: ApprovalStatusEnum) {
    try {
      const approval = await this.findOne(approvalId);
      const programRegistration = approval.registration;
      this.logger.log('Approval Details:', approval);
      let bccEmails: { emailAddress: string; name: string }[] = [];
      if (process.env.HDB_BCC_EMAIL) {
        bccEmails.push({
          emailAddress: process.env.HDB_BCC_EMAIL,
          name: process.env.HDB_BCC_EMAIL_NAME || 'HDB Team',
        });
      }
      try {
        const financeUsers = await this.userRepo.getUsersByRoleKey('ROLE_FINANCE_MANAGER');
        if (financeUsers && financeUsers.length > 0) {
          bccEmails = bccEmails.concat(
            financeUsers.map((u) => ({
              emailAddress: u.email,
              name: u.fullName || 'Finance Manager',
            })),
          );
        }
      } catch (err) {
        this.logger.error('Failed to fetch finance managers for BCC', err);
      }
      // console.log('Sending approval email notification', { approval, programRegistration });
      if (approval.approvalStatus === ApprovalStatusEnum.APPROVED) {
        const queryRunner: QueryRunner = this.dataSource.createQueryRunner();
        const proformaInvoiceSeqNumber = programRegistration.proFormaInvoiceSeqNumber 
          ? programRegistration.proFormaInvoiceSeqNumber
          : await this.repo.generateProFormaInvoiceSequenceNumber(queryRunner, programRegistration);
        // save the proFormaInvoiceSeqNumber to the registration registrationrepo
        await this.registrationRepo.update(programRegistration.id, {
          proFormaInvoiceSeqNumber: proformaInvoiceSeqNumber,
        });
        this.logger.log('Proforma Invoice Sequence Number:', {proformaInvoiceSeqNumber});
        const approvedMergeInfo = {
          venue_name: programRegistration?.allocatedProgram?.venueNameInEmails ?? "",
          s_day: programRegistration?.allocatedProgram?.startsAt ? getWeekName(programRegistration?.allocatedProgram?.startsAt ?? new Date)  : "-",
          hdb_price: programRegistration?.allocatedProgram?.basePrice || "",
          payment_last_date: programRegistration?.allocatedProgram?.startsAt ? formatDate(deductDaysFromDate(programRegistration?.allocatedProgram?.startsAt, 10).toISOString() || "") : "-",
          cash_details: (programRegistration.allocatedProgram?.basePrice ?? 0) > 100 ? "" : "4. Cash: If you are paying by cash, after paying the cash, please click on the following link and fill in the details. Please note that this simple update by you will ensure that there is no omission or reconciliation issue owing to human error.",
          e_day: getWeekName(programRegistration?.allocatedProgram?.endsAt ?? new Date ) || "",
          reg_name: programRegistration.fullName ?? "Infinitheist",
          hdb_or_msd: programRegistration?.allocatedProgram?.name ?? "HDB",
          hdb_dates: `${ formatDate(programRegistration?.allocatedProgram?.startsAt.toISOString() ?? "")} to ${formatDate(programRegistration?.allocatedProgram?.endsAt.toISOString() ?? "")}`,
          hdb_no: programRegistration?.allocatedProgram?.name ?? "HDB",
          payment_online_link: this.generatePaymentLink(programRegistration.program.id, programRegistration.userId, PaymentModeEnum.ONLINE_RAZORPAY), //`${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programRegistration.program.id}&paymentMode=${encodeURIComponent(PaymentModeEnum.ONLINE_RAZORPAY)}`,
          payment_back_transfer_link: this.generatePaymentLink(programRegistration.program.id, programRegistration.userId, PaymentModeEnum.DIRECT_BANK_TRANSFER), // `${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programRegistration.program.id}&paymentMode=${encodeURIComponent(PaymentModeEnum.DIRECT_BANK_TRANSFER)}`,
          payment_cheque_link: this.generatePaymentLink(programRegistration.program.id, programRegistration.userId, PaymentModeEnum.CHEQUE), //`${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programRegistration.program.id}&paymentMode=${encodeURIComponent(PaymentModeEnum.CHEQUE)}`,
        }
        const templateKey = process.env.ZEPTO_BLESSED_HDB_EMAIL_TEMPLATE_ID;

        const pdfBuffer = await this.generateProFormaInvoicePDF(programRegistration, proformaInvoiceSeqNumber);
        const response = await axios.get(pdfBuffer.url, { responseType: 'arraybuffer' });
        const fileData = Buffer.from(response.data, 'binary').toString('base64');

        const attachments = {
          filename: `pro-forma-invoice.pdf`, //`invoice-${invoiceData.invoice.invoiceSequenceNumber || invoiceData.invoice.id}.pdf`,
          content: fileData,
          contentType: 'application/pdf',
        }
        this.logger.log('Sending approval email with merge info', { approvedMergeInfo, templateKey} );
        const emailData: SendSingleEmailDto = {
          templateKey: templateKey,
          from: { address: zeptoEmailCreadentials.ZEPTO_EMAIL, name: programRegistration.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME },
          to: { emailAddress: programRegistration.emailAddress, name: programRegistration.fullName },
          bcc: bccEmails,
          mergeInfo: approvedMergeInfo,
          attachments: [
            {
              name: attachments.filename,
              content: attachments.content,
              mime_type: attachments.contentType,
            }
          ],
          subject: '',
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: programRegistration.createdBy.id,
            // updatedBy: programRegistration.updatedBy.id,
          },
        };
    
        await this.communicationService.sendSingleEmail(emailData);
        this.logger.log('Approval email sent successfully');

        // Send WhatsApp message to registred user
        const regMessageData: SendTemplateMessageDto = {
          whatsappNumber: programRegistration.mobileNumber.slice(0), // Remove + prefix
          templateName: process.env.WATI_BLESSED_HDB_TEMPLATE_ID || "",
          broadcastName: process.env.WATI_BLESSED_HDB_TEMPLATE_ID || "",
          parameters: [
            { name: 'reg_name', value: programRegistration?.fullName },
            { name: 'hdb_msd', value: programRegistration?.allocatedProgram?.name || "HDB" },
            { name: 'payment_link', value: this.generatePaymentLink(programRegistration.program.id) }, // `${SEEKER_FE_REG_PATH}${programRegistration.program.id}` },
          ],
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: programRegistration.createdBy.id,
            // updatedBy: programRegistration.updatedBy.id,
          },
        };
        await this.communicationService.sendTemplateMessage(regMessageData);

        this.logger.log('WhatsApp message sent successfully to registered user', { regMessageData });

        // Send WhatsApp message to RM if available
        const rmDetails = await this.userRepo.getUserByField(
          'id',
          programRegistration.rmContact)

        this.logger.log('RM Details:', rmDetails);

        const rmMessageData: SendTemplateMessageDto = {
          whatsappNumber: `${rmDetails.countryCode.slice(0)}${rmDetails.phoneNumber}` ,
          templateName: process.env.WATI_BLESSED_HDB_RM_TEMPLATE_ID || "",
          broadcastName: process.env.WATI_BLESSED_HDB_RM_TEMPLATE_ID || "",
          parameters: [
            { name: 'reg_name', value: programRegistration.fullName },
            { name: 'rm_name', value: rmDetails.firstName + " " + rmDetails.lastName || "-" },
            { name: 'reg_mobile', value: programRegistration.mobileNumber },
            { name: 'hdb_msd_variable', value: programRegistration?.allocatedProgram?.name || "-" },
          ],
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: programRegistration.createdBy.id,
            // updatedBy: programRegistration.updatedBy.id,
          },
        }
        console.log('RM Message Data:', rmMessageData);
        await this.communicationService.sendTemplateMessage(rmMessageData);

        this.logger.log('WhatsApp messages sent successfully to rm', { rmDetails, rmMessageData });
         // Update the registration table with the proFormaInvoicePdfUrl
        await this.registrationRepo.update(programRegistration.id, {
          proFormaInvoicePdfUrl: pdfBuffer.url,
        });

        this.logger.log('Updated registration table with proFormaInvoicePdfUrl', { registrationId: programRegistration.id, proFormaInvoicePdfUrl: pdfBuffer.url });


      } else if (approval.approvalStatus === ApprovalStatusEnum.REJECTED && existingApprovalStatus == ApprovalStatusEnum.APPROVED) {
        const holdMergeInfo = {
          reg_name: programRegistration.fullName || "Infinitheist",
          hdb_msd: "HDB"
        }
        const templateKey = process.env.ZEPTO_HOLD_EMAIL_TEMPLATE_ID;
        this.logger.log('Sending hold email with merge info', { holdMergeInfo, templateKey });
        const emailData: SendSingleEmailDto = {
          templateKey: templateKey,
          from: { address: zeptoEmailCreadentials.ZEPTO_EMAIL, name: programRegistration.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME },
          to: { emailAddress: programRegistration.emailAddress, name: programRegistration.fullName },
          bcc: bccEmails,
          mergeInfo: holdMergeInfo,
          attachments: [],
          subject: '',
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: programRegistration.createdBy.id,
            // updatedBy: programRegistration.updatedBy.id,
          },
        };
    
        this.communicationService.sendSingleEmail(emailData);
        this.logger.log('Hold email sent successfully', { emailData });
      }
      
    } catch (error) {
      this.logger.error('Error sending approval email notification', error);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
    }
  }
  async generateProFormaInvoicePDF(invoiceData: any, sequenceNumber: string):Promise<{ url: string; buffer: Buffer }> {
    try {
      // Launch Puppeteer
      const browser = await puppeteer.launch({
        args: [
          ...(Array.isArray(chromium.args) ? chromium.args : []),
          '--no-sandbox',
          '--disable-setuid-sandbox',
          '--disable-features=AudioServiceOutOfProcess',
          '--disable-gpu',
          '--disable-software-rasterize',
          '--disable-features=AudioServiceOutOfProcessKillAtHang',
          '--single-process',
          '--disable-software-rasterizer',
        ],
        defaultViewport: null,
        executablePath: await chromium.executablePath('https://github.com/Sparticuz/chromium/releases/download/v119.0.2/chromium-v119.0.2-pack.tar'),
        headless: true,
      });
      const page = await browser.newPage();
      // Set the HTML content
      console.log('Generating pro-forma invoice template data', { invoiceData });
      const gstPercentage = (Number(invoiceData.program?.cgst ?? 0) + Number(invoiceData.program?.sgst ?? 0)) || 0;
      const gstAmount = invoiceData.program?.gstNumber 
        ? parseFloat((Number(invoiceData.allocatedProgram?.basePrice) * (gstPercentage / 100)).toFixed(2)) 
        : 0;
      
      const invoiceTemplateData: ProFormaInvoice = {
        companyAddress: invoiceData?.program?.invoiceSenderAddress,
        companyGST: invoiceData.program?.gstNumber || '',
        companyCIN: invoiceData.program?.invoiceSenderCin || '',
        invoiceNumber: sequenceNumber,
        currentDate: formatDate(new Date().toISOString()),
        seekerName: invoiceData?.proFormaInvoiceName || '',
        seekerAddress: invoiceData?.proFormaInvoiceAddress || '',
        amount: Number(invoiceData.allocatedProgram?.basePrice) || 0,
        totalAmount: (Number(invoiceData.allocatedProgram?.basePrice) || 0) + gstAmount,
        amountInWords: amountInWordsIndian(Number((Number(invoiceData.allocatedProgram?.basePrice) || 0) + gstAmount)),
        startDate: formatDate(invoiceData.allocatedProgram?.startsAt) || '',
        endDate: formatDate(invoiceData.allocatedProgram?.endsAt) || '',
        gstAmount: gstAmount,
        programName: invoiceData.allocatedProgram?.name || '',
        gstPercentage: gstPercentage,
        companyPAN: invoiceData?.program?.invoiceSenderPan ?? '',
        checkOutTime: formatDate(String(invoiceData?.allocatedProgram?.checkoutAt)),
        reportingTime: formatDate(String(invoiceData?.allocatedProgram?.checkinAt))
      };

      await page.setContent(UPDATED_PRO_FORMA_INVOICE(invoiceTemplateData));


      
      // For now, return a placeholder buffer
      const pdfBuffer = await page.pdf({
        format: 'A4',
        printBackground: true,
      });      
      await browser.close();
      // const bucketName = process.env.AWS_S3_BUCKET_NAME || '';
      const key = `proforma-invoices/${invoiceData.program.code}/${invoiceData?.registrationSeqNumber || uuidv4()}.pdf`;
      const uploadedInvoice = await this.awsS3Service.uploadToS3(key, pdfBuffer, 'application/pdf');
      // const s3Url = await this.awsS3Service.getSignedUrl(key);
      // const s3Url = this.awsS3Service.getS3Url(key);
      return {url: uploadedInvoice, buffer: pdfBuffer};
      

    } catch (error) {
      this.logger.error('PDF generation failed', '', { invoiceId: invoiceData.id, error });
      throw error;
    }
  }

  /**
   * Generates a payment link for a specific program and user with optional payment mode.
   *
   * @param programId - The unique identifier of the program for which the payment link is generated.
   * @param userId - (Optional) The unique identifier of the user. Defaults to `undefined`.
   * @param paymentMode - (Optional) The payment mode to be included in the link. Should be of type `PaymentModeEnum`.
   * @returns The generated payment link as a string.
   */
  generatePaymentLink(programId: number, userId: number | undefined = undefined, paymentMode?: PaymentModeEnum | undefined): string {
    const baseUrl = process.env.SEEKER_FE_BASE_URL || '';
    var url = `${baseUrl}${SEEKER_FE_REG_PATH}${programId}`;
    if (paymentMode) {
      url += `&paymentMode=${encodeURIComponent(paymentMode)}`;
    }
    if (userId) {
      url += `&userId=${userId}`;
    }
    return url;
  }
}
