import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { ILike, Repository } from 'typeorm';
import { Program, ProgramRegistration, ProgramSession, RegistrationApproval, RegistrationApprovalTrack } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CreateRegistrationApprovalDto } from './dto/create-registration-approval.dto';
import { UpdateRegistrationApprovalDto } from './dto/update-registration-approval.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { decrementSeatCounts, incrementSeatCounts } from '../common/utils/seat-count.util';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { generateNextSequence } from 'src/common/utils/common.util';

@Injectable()
export class RegistrationApprovalRepository {
  constructor(
    @InjectRepository(RegistrationApproval)
    private readonly repo: Repository<RegistrationApproval>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
    @InjectRepository(ProgramRegistration)
    private readonly programRegistrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(RegistrationApprovalTrack)
    private readonly approvalTrackRepo: Repository<RegistrationApprovalTrack>,

  ) {}

  async createEntity(dto: CreateRegistrationApprovalDto) {
    this.logger.log('Creating Registration Approval', dto);
    try {
      return await this.createByRegistrationId(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async createByRegistrationId(dto: CreateRegistrationApprovalDto) {
    try {
      const entity = this.repo.create({
        ...dto,
        registration: { id: dto.registrationId } as any,
        approvedByUser: dto.approvedBy ? ({ id: dto.approvedBy } as any) : undefined,
        createdBy: { id: dto.createdBy } as any,
        updatedBy: { id: dto.updatedBy } as any,
        approvalStatus: dto.approvalStatus || ApprovalStatusEnum.PENDING,
      });

      // if registrattionId is already exists, throw an error
      const existingApproval = await this.repo.findOne({
        where: { registrationId: dto.registrationId },
      });
      if (existingApproval) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_APPROVAL_ALREADY_EXISTS,
          null,
          null,
          `Registration approval already exists for registrationId: ${dto.registrationId}`
        );
      }
      const saved = await this.commonDataService.save(this.repo, entity);
      await this.createApprovalTrack(saved, dto.createdBy);
      const status = saved.approvalStatus;
      // Update related ProgramRegistration status if approvalStatus is set
      if (dto.approvalStatus && dto.registrationId) {
        // Fetch registration with program.type
        const reg = await this.programRegistrationRepo.findOne({
          where: { id: dto.registrationId },
          relations: ['program', 'programSession', 'program.type'],
        });
        let newStatus: RegistrationStatusEnum | undefined;

        if (dto.approvalStatus === ApprovalStatusEnum.APPROVED) {
          if (reg?.program?.type?.requiresPayment === true) {
            newStatus = RegistrationStatusEnum.PENDING;
          } else {
            newStatus = RegistrationStatusEnum.COMPLETED;
          }
        } else if (dto.approvalStatus === ApprovalStatusEnum.REJECTED) {
          newStatus = RegistrationStatusEnum.REJECTED;
        } else if (dto.approvalStatus === ApprovalStatusEnum.PENDING) {
          newStatus = RegistrationStatusEnum.PENDING;
        } else if (dto.approvalStatus === ApprovalStatusEnum.UNDER_REVIEW) {
          newStatus = RegistrationStatusEnum.PENDING;
        }

        if (newStatus) {
          const updateResult = await this.programRegistrationRepo.update(
            dto.registrationId,
            { registrationStatus: newStatus }
          );
          this.logger.log('ProgramRegistration update result', {
            registrationId: dto.registrationId,
            affected: updateResult.affected,
            newStatus
          });
          if (updateResult.affected === 0) {
            this.logger.warn('ProgramRegistration update affected 0 rows', {
              registrationId: dto.registrationId
            });
          }
          // Only increment seat counts if not payment required and status is APPROVED
          if (reg?.program?.id && reg.program.type?.requiresPayment !== true) {
            if (dto.approvalStatus === ApprovalStatusEnum.APPROVED ) {
              await incrementSeatCounts(
                this.repo.manager,
                reg.program.id,
                reg.programSession?.id,
              );
            }
          }
        }
      }
      return saved;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async findAll(
    limit: number,
    offset: number,
    programId?: number,
    search?: string,
    filters: Record<string, any> = {},
  ) {
    try {
      const whereClause: any = { deletedAt: null };

      if (programId) {
      whereClause.registration = {
        program: { id: programId }
      };
    }
    // Add status filter if provided
    if (filters && typeof filters === 'object' && filters.status) {
      whereClause.approvalStatus = filters.status;
    }

    // Add gender filter if provided
    if (filters && typeof filters === 'object' && filters.gender) {
      if (!whereClause.registration) {
        whereClause.registration = {};
      }
      whereClause.registration.gender = filters.gender;
    }

     let finalWhereClause: any = whereClause;
      
      if (search != null && search.trim() !== '') {
        finalWhereClause = [
          { ...whereClause, registration: { ...(whereClause.registration || {}), fullName: ILike(`%${search.trim()}%`) } },
          { ...whereClause, registration: { ...(whereClause.registration || {}), emailAddress: ILike(`%${search.trim()}%`) } },
          { ...whereClause, registration: { ...(whereClause.registration || {}), mobileNumber: ILike(`%${search.trim()}%`) } },
        ];
      }

      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        finalWhereClause,
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        ['registration', 'registration.program'],
      );
      const total = await this.repo.count({ where: finalWhereClause });
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false, ['registration', 'registration.program', 'registration.programSession', 'registration.user', 'registration.allocatedProgram', 'registration.allocatedSession']);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_FIND_BY_ID_FAILED, error);
    }
  }

  async updateEntity(id: number, dto: UpdateRegistrationApprovalDto) {
    try {
      // Find the entity to get the registrationId
      const entity = await this.repo.findOne({ 
        where: { id },
        relations: ['registration'] 
      });

      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, null, null, id.toString());
      }

      // Delegate to updateByRegistrationId for consistent update logic
      return await this.updateByRegistrationId(entity.registration?.id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    try {
      // FIXED: Load the entity with registration relation to access registrationId
      const entity = await this.repo.findOne({ 
        where: { id },
        relations: ['registration'] 
      });
      
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, null, null, id.toString());
      }
      
      // Soft delete the entity
      entity.deletedAt = new Date();
      entity.approvalStatus = ApprovalStatusEnum.REJECTED;
      const saved = await this.commonDataService.save(this.repo, entity);

      // Also update related ProgramRegistration status to REJECTED
      if (entity.registration?.id) {
        const updateResult = await this.programRegistrationRepo.update(
          entity.registration.id, 
          { 
            registrationStatus: RegistrationStatusEnum.REJECTED
          }
        );
        
        // Log the update result for debugging
        this.logger.log('ProgramRegistration update result for removal', {
          registrationId: entity.registration.id,
          affected: updateResult.affected
        });
        
        // Check if the update was successful
        if (updateResult.affected === 0) {
          this.logger.warn('ProgramRegistration update affected 0 rows during removal', {
            registrationId: entity.registration.id
          });
        }
      }

      return saved;

    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_DELETE_FAILED, error);
    }
  }
  async generateProFormaInvoiceSequenceNumber(
      queryRunner: any,
      registration: ProgramRegistration,
    ): Promise<string> {
      const startsAt = registration.program?.startsAt;
      const startYear = startsAt ? startsAt.getFullYear().toString().slice(-2) : '';
      const nextYear = startsAt ? (startsAt.getFullYear() + 1).toString().slice(-2) : '';
      const programCode = registration.allocatedProgram?.code || registration.program.code;
    
      const pattern = `${programCode}/${startYear}${nextYear}PRO`;
    
      // Fetch the latest invoice matching the pattern
      const latestInvoice = await queryRunner.manager
        .createQueryBuilder(ProgramRegistration, 'registration')
        .andWhere('registration.proFormaInvoiceSeqNumber IS NOT NULL')
        .andWhere('registration.proFormaInvoiceSeqNumber LIKE :pattern', { pattern: `${pattern}%` })
        .orderBy('registration.proFormaInvoiceSeqNumber', 'DESC')
        .getOne();

    
      if (latestInvoice) {
        // Generate the next sequence number based on the latest invoice
        return generateNextSequence(pattern, latestInvoice.proFormaInvoiceSeqNumber);
      } else {
        // If no previous invoice, start with 001
        return `${pattern}001`;
      }
    }

  async updateByRegistrationId(registrationId: number, dto: UpdateRegistrationApprovalDto) {
    try {
      // Find the approval record by registrationId
      const entity = await this.repo.findOne({ 
        where: { registrationId },
        relations: ['registration'] 
      });

      const status = entity?.approvalStatus;

      if (!entity) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, 
          null, 
          null, 
          `No approval found for registrationId: ${registrationId}`
        );
      }

      let allocatedProgram: Program | null = null;
      let allocatedSession: ProgramSession | null = null;
      
      if (dto.allocatedProgramId) {
        allocatedProgram = await this.repo.manager.findOne(Program, {
          where: { id: dto.allocatedProgramId }
        });

        if (!allocatedProgram) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            `Program with ID ${dto.allocatedProgramId} not found`
          );
        }
      }
      if (dto.allocatedSessionId) {
        allocatedSession = await this.repo.manager.findOne(ProgramSession, {
          where: { id: dto.allocatedSessionId }
        });

        if (!allocatedSession) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
            null,
            null,
            `Program Session with ID ${dto.allocatedSessionId} not found`
          );
        }

        // Validate if session belongs to the allocated program
        if (allocatedProgram && allocatedSession.programId !== allocatedProgram.id) {
          throw new InifniBadRequestException(
            ERROR_CODES.INVALID_PROGRAM_SESSION,
            null,
            null,
            'Program Session does not belong to the allocated Program'
          );
        }
      }
      // ====== Waitlist logic before approving ======
      if (dto.approvalStatus === ApprovalStatusEnum.APPROVED) {
        // Fetch registration with program.type
        const reg = await this.programRegistrationRepo.findOne({
          where: { id: registrationId },
          relations: ['program', 'programSession', 'program.type'],
        });

        if (!reg) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
            null,
            null,
            registrationId.toString(),
          );
        }


        // if program or session has requiresApproval, we don't need to check seats
        if (reg.program?.requiresApproval != true && reg.programSession?.requiresApproval != true) {
          this.logger.log('Program or Session requires approval, skipping seat checks');
    

        // Determine registration level and seat info
        const registrationLevel = reg.program?.type?.registrationLevel;
        let seatsInfo: any = registrationLevel === 'session' ? reg.programSession : reg.program;
        const isLimitedSeats = seatsInfo?.limitedSeats === true;
        const isWaitlistEnabled = seatsInfo?.waitlistApplicable === true;
        const totalSeats = seatsInfo?.totalSeats || 0;
        const filledSeats = registrationLevel === 'session' ? reg.programSession?.reservedSeats || 0 : reg.program?.filledSeats || 0;
        

        // Waitlist logic: if limited seats and not waitlist enabled, or if waitlist enabled but all seats filled
        if (isLimitedSeats) {
          if (!isWaitlistEnabled && filledSeats >= totalSeats) {
            throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_NO_SPACE, null, null, 'No seats available for approval');
          }
          if (isWaitlistEnabled && filledSeats >= totalSeats) {
            throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_NO_SPACE, null, null, 'No seats available for approval');
          }
        }

        // if allocatedProgram or allocatedSession is provided, check for same waitlist logic
        if (allocatedProgram || allocatedSession) {
          const allocatedSeatsInfo = allocatedSession
            ? allocatedSession
            : allocatedProgram;

          const allocatedIsLimitedSeats = allocatedSeatsInfo?.limitedSeats === true;
          const allocatedIsWaitlistEnabled = allocatedSeatsInfo?.waitlistApplicable === true;
          const allocatedTotalSeats = allocatedSeatsInfo?.totalSeats || 0;
          const allocatedFilledSeats =
            allocatedSession?.reservedSeats || allocatedProgram?.filledSeats || 0;

          // Waitlist logic for allocated program/session
          if (allocatedIsLimitedSeats) {
            if (!allocatedIsWaitlistEnabled && allocatedFilledSeats >= allocatedTotalSeats) {
              throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_NO_SPACE, null, null, 'No seats available for approval in the allocated program/session');
            }
            if (allocatedIsWaitlistEnabled && allocatedFilledSeats >= allocatedTotalSeats) {
              throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_NO_SPACE, null, null, 'No seats available for approval in the allocated program/session');
            }
          }
        }
      }
      }
      // ====== End waitlist logic ======

      // Only update approvalStatus, approvedByUser, and updatedBy
      if (dto.approvalStatus !== undefined) entity.approvalStatus = dto.approvalStatus;
      if (dto.approvedBy !== undefined) entity.approvedByUser = dto.approvedBy ? ({ id: dto.approvedBy } as any) : undefined;
      if (dto.updatedBy !== undefined) entity.updatedBy = { id: dto.updatedBy } as any;
      if (dto.approvalDate !== undefined) entity.approvalDate = dto.approvalDate;
      const saved = await this.commonDataService.save(this.repo, entity);
      await this.createApprovalTrack(saved, dto.updatedBy);

      // Update related ProgramRegistration status if approvalStatus is updated
      if (dto.approvalStatus !== undefined) {
        // Fetch registration with program.type
        const reg = await this.programRegistrationRepo.findOne({
          where: { id: registrationId },
          relations: ['program', 'programSession', 'program.type', 'allocatedProgram', 'allocatedSession'],
        });
        let newStatus: RegistrationStatusEnum | undefined;

        if (dto.approvalStatus === ApprovalStatusEnum.APPROVED) {
          if (reg?.program?.type?.requiresPayment === true) {
            newStatus = RegistrationStatusEnum.PENDING;
          } else {
            newStatus = RegistrationStatusEnum.COMPLETED;
          }
        } else if (dto.approvalStatus === ApprovalStatusEnum.REJECTED) {
          newStatus = RegistrationStatusEnum.REJECTED;
        } else if (dto.approvalStatus === ApprovalStatusEnum.PENDING) {
          newStatus = RegistrationStatusEnum.PENDING;
        } else if (dto.approvalStatus === ApprovalStatusEnum.UNDER_REVIEW) {
          newStatus = RegistrationStatusEnum.PENDING;
        } else if (dto.approvalStatus === ApprovalStatusEnum.ON_HOLD) {
          newStatus = RegistrationStatusEnum.PENDING;
        }

        if (newStatus) {
          const updateData: any = {
            registrationStatus: newStatus,
          };
          if (dto.approvalStatus === ApprovalStatusEnum.APPROVED) {
            if (allocatedProgram) {
              updateData.allocatedProgram = allocatedProgram;
            }
            if (allocatedSession) {
              updateData.allocatedSession = allocatedSession;
            }
          }

          const updateResult = await this.programRegistrationRepo.update(
            registrationId,
            updateData,
          );
          // const updateResult = await this.programRegistrationRepo.update(
          //   registrationId,
          //   {
          //     registrationStatus: newStatus,
          //     allocatedProgram: {},
          //     allocatedSession: {},
          //   }
          // );
          
          // Log the update result for debugging
          this.logger.log('ProgramRegistration update result by registrationId', {
            registrationId,
            affected: updateResult.affected,
            newStatus
          });
          
          // Check if the update was successful
          if (updateResult.affected === 0) {
            this.logger.warn('ProgramRegistration update affected 0 rows by registrationId', {
              registrationId
            });
          }

          // Only update seat counts if not payment required and status is APPROVED
          // if (reg?.program?.id && reg.program.type?.requiresPayment !== true) {
            // If the approval status is APPROVED, increment seat counts if it was previously PENDING or UNDER_REVIEW or rejected
            if (
              reg &&
              dto.approvalStatus === ApprovalStatusEnum.APPROVED &&
              status !== ApprovalStatusEnum.APPROVED
            ) {
              await incrementSeatCounts(
                this.repo.manager,
                reg.program.id,
                reg.programSession?.id,
              );
              if (allocatedProgram) {
                await incrementSeatCounts(
                  this.repo.manager,
                  allocatedProgram.id,
                );
              } else if (allocatedSession) {
                await incrementSeatCounts(
                  this.repo.manager,
                  reg.program.id,
                  allocatedSession.id,
                );
              }
            }
            // Handle seat changes if already approved and allocatedProgram is updated
            else if (
              reg &&
              dto.approvalStatus === ApprovalStatusEnum.APPROVED &&
              status === ApprovalStatusEnum.APPROVED
            ) {
              const existingAllocatedProgramId = reg.allocatedProgram?.id;
              const existingAllocatedSessionId = reg.allocatedSession?.id;
              if (dto.allocatedProgramId) {
                if (existingAllocatedProgramId && existingAllocatedProgramId !== dto.allocatedProgramId) {
                  await decrementSeatCounts(
                    this.repo.manager,
                    existingAllocatedProgramId,
                    existingAllocatedSessionId,
                  );
                  await incrementSeatCounts(
                    this.repo.manager,
                    dto.allocatedProgramId,
                    dto.allocatedSessionId,
                  );
                } else if (!existingAllocatedProgramId) {
                  await incrementSeatCounts(
                    this.repo.manager,
                    dto.allocatedProgramId,
                    dto.allocatedSessionId,
                  );
                }
              }
            }
            // If the approval status is REJECTED and was previously APPROVED, decrement seat counts
            else if (
              reg &&
              (dto.approvalStatus === ApprovalStatusEnum.REJECTED || dto.approvalStatus === ApprovalStatusEnum.PENDING  || dto.approvalStatus === ApprovalStatusEnum.ON_HOLD) &&
              status === ApprovalStatusEnum.APPROVED
            ) {
              await decrementSeatCounts(
                this.repo.manager,
                reg.program.id,
                reg.programSession?.id,
              );
              if (reg.allocatedProgram) {
                await decrementSeatCounts(
                  this.repo.manager,
                  reg.allocatedProgram?.id,
                  reg.allocatedSession?.id,
                );
              }
              // unallocate the program and session
              await this.programRegistrationRepo.update(
                registrationId,
                {
                  allocatedProgram: null,
                  allocatedSession: null, 
                }
              );
            }
          // }
        }
      }

      return saved;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async removeByRegistrationId(registrationId: number) {
    try {
      // Find the approval record by registrationId
      const entity = await this.repo.findOne({ 
        where: { registrationId },
        relations: ['registration'] 
      });
      
      if (!entity) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, 
          null, 
          null, 
          `No approval found for registrationId: ${registrationId}`
        );
      }
      
      // Soft delete the entity
      entity.deletedAt = new Date();
      entity.approvalStatus = ApprovalStatusEnum.REJECTED;
      const saved = await this.commonDataService.save(this.repo, entity);

      // Also update related ProgramRegistration status to REJECTED
      const updateResult = await this.programRegistrationRepo.update(
        registrationId, 
        { 
          registrationStatus: RegistrationStatusEnum.REJECTED
        }
      );
      
      // Log the update result for debugging
      this.logger.log('ProgramRegistration update result for removal by registrationId', {
        registrationId,
        affected: updateResult.affected
      });
      
      // Check if the update was successful
      if (updateResult.affected === 0) {
        this.logger.warn('ProgramRegistration update affected 0 rows during removal by registrationId', {
          registrationId
        });
      }

      return saved;

    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_DELETE_FAILED, error);
    }
  }

  async findByRegistrationId(registrationId: number) {
    try {
      const data = await this.repo.findOne({
        where: { registrationId },
        relations: ['registration', 'approvedByUser', 'createdBy', 'updatedBy']
      });
      
      if (!data) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, 
          null, 
          null, 
          `No approval found for registrationId: ${registrationId}`
        );
      }
      
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Create a track record for approval changes
   */
  private async createApprovalTrack(
    approval: RegistrationApproval,
    userId: number | null,
  ) {
    try {
      const track = this.approvalTrackRepo.create({
        approvalId: approval.id,
        registrationId: approval.registrationId,
        approvalStatus: approval.approvalStatus,
        approvalDate: approval.approvalDate,
        approvedBy: approval.approvedBy,
        rejectionReason: approval.rejectionReason,
        reviewerComments: approval.reviewerComments,
        autoApproved: approval.autoApproved,
        createdBy: userId,
        updatedBy: userId,
      });
      await this.approvalTrackRepo.save(track);
      this.logger.log('Approval track record created', {
        approvalId: approval.id,
        trackId: track.id,
      });
    } catch (error) {
      this.logger.error('Failed to create approval track', error);
    }
  }

}