import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  ParseIntPipe,
  Post,
  Put,
  Query,
  Req,
  Res,
  UseGuards,
  ValidationPipe,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { registrationApprovalMessages, paginationConstants, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { RegistrationApprovalService } from './registration-approval.service';
import { CreateRegistrationApprovalDto } from './dto/create-registration-approval.dto';
import { UpdateRegistrationApprovalDto } from './dto/update-registration-approval.dto';
import { extractAndValidateUser } from 'src/common/utils/program-question.util';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('registration-approval')
@Controller('registration-approval')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'mahatria','finance_manager', 'relational_manager', 'shoba', 'rm', 'operational_manger')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class RegistrationApprovalController {
  constructor(
    private readonly service: RegistrationApprovalService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) { }

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: registrationApprovalMessages.CREATE_NEW })
  @ApiBody({ type: CreateRegistrationApprovalDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateRegistrationApprovalDto,
    @Res() res: Response,
  ) {
    this.logger.log(registrationApprovalMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      const data = await this.service.create(dto);
      await this.responseService.success(res, registrationApprovalMessages.APPROVAL_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationApprovalMessages.GET_ALL })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by programId' })
  @ApiQuery({ name: 'search', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Encoded URI JSON for filters (e.g. status)' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Res() res: Response,
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId?: number,
    @Query('search') search?: string,
    @Query('filters') filters?: string,
  ) {
    this.logger.log(registrationApprovalMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      let parsedFilters: Record<string, any> = {};
      if (filters) {
        try {
          parsedFilters = JSON.parse(decodeURIComponent(filters));
        } catch (e) {
          parsedFilters = {};
        }
      }
      const data = await this.service.findAll(+limit, +offset, programId ? +programId : undefined, search, parsedFilters);
      await this.responseService.success(res, registrationApprovalMessages.APPROVALS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationApprovalMessages.GET_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: registrationApprovalMessages.APPROVAL_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(registrationApprovalMessages.FIND_ONE_REQUEST_RECEIVED, { id });
    try {
      const data = await this.service.findOne(id);
      await this.responseService.success(res, registrationApprovalMessages.APPROVAL_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationApprovalMessages.UPDATE_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: registrationApprovalMessages.APPROVAL_ID })
  @ApiBody({ type: UpdateRegistrationApprovalDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateRegistrationApprovalDto,
    @Res() res: Response,
    @Req() req: any,
  ) {
    this.logger.log(registrationApprovalMessages.UPDATE_REQUEST_RECEIVED, { id, dto });
    try {
      const user = extractAndValidateUser(req);
      dto.updatedBy = user.id;
      const data = await this.service.update(id, dto);
      await this.responseService.success(res, registrationApprovalMessages.APPROVAL_UPDATED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationApprovalMessages.DELETE_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: registrationApprovalMessages.APPROVAL_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(registrationApprovalMessages.DELETE_REQUEST_RECEIVED, { id });
    try {
      await this.service.remove(id);
      await this.responseService.success(res, registrationApprovalMessages.APPROVAL_DELETED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put('/by-registration/:registrationId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Update approval status by registrationId' })
  @ApiParam({ name: 'registrationId', type: Number, description: 'Registration ID' })
  @ApiBody({ type: UpdateRegistrationApprovalDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async updateByRegistrationId(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateRegistrationApprovalDto,
    @Res() res: Response,
    @Req() req: any,
  ) {
    this.logger.log('Update approval by registrationId', { registrationId, dto });
    try {
      const user = extractAndValidateUser(req);
      dto.updatedBy = user.id;
      const existingApproval = await this.service.findOneByRegistrationId(registrationId);
      if (!existingApproval) {
        return handleControllerError(res, { status: HttpStatus.NOT_FOUND, message: 'Approval not found for the given registrationId' });
      }
      const status = existingApproval.approvalStatus;
      const data = await this.service.updateByRegistrationId(registrationId, dto);
      this.service.sendApprovalEmailNotification(data.id, status);
      await this.responseService.success(res, 'Approval updated by registrationId', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
