import { Injectable } from '@nestjs/common';
import { CreateQuestionOptionDto } from './dto/create-question-option.dto';
import { UpdateQuestionOptionDto } from './dto/update-question-option.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { User } from 'src/common/entities';
import { QuestionOptionRepository } from './question-option.repository';
import { questionOptionConstMessages } from 'src/common/constants/strings-constants';

@Injectable()
export class QuestionOptionService {
  constructor(
    private readonly questionOptionRepository: QuestionOptionRepository,
    private readonly logger: AppLoggerService,
  ) {}

  async create(dto: CreateQuestionOptionDto) {
    this.logger.log(questionOptionConstMessages.CREATE_MAPPING_REQUEST_RECEIVED, dto);
    try {
      return await this.questionOptionRepository.createMappings(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_CREATION_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number, searchText: string) {
    this.logger.log(questionOptionConstMessages.FINDING_ALL_MAPPINGS, { limit, offset, searchText });
    try {
      return await this.questionOptionRepository.findAllMappings(limit, offset, searchText);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_FIND_ALL_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(questionOptionConstMessages.FINDING_MAPPING_BY_ID(id));
    try {
      return await this.questionOptionRepository.findMappingById(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateQuestionOptionDto) {
    this.logger.log(questionOptionConstMessages.UPDATING_MAPPING(id), dto);
    try {
      return await this.questionOptionRepository.updateMapping(id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_UPDATE_FAILED, error);
    }
  }

  async remove(id: number, user: User) {
    this.logger.log(questionOptionConstMessages.REMOVING_MAPPING(id));
    try {
      return await this.questionOptionRepository.softDeleteMapping(id, user);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_DELETE_FAILED, error);
    }
  }
}