import { Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository, ILike } from "typeorm";
import { QuestionOptionMap, Question, Option, User } from "src/common/entities";
import { CreateQuestionOptionDto } from "./dto/create-question-option.dto";
import { UpdateQuestionOptionDto } from "./dto/update-question-option.dto";
import { CommonDataService } from "src/common/services/commonData.service";
import { AppLoggerService } from "src/common/services/logger.service";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import InifniBadRequestException from "src/common/exceptions/infini-badrequest-exception";
import { optionConstMessages, questionConstMessages, questionOptionConstMessages, userConstMessages } from "src/common/constants/strings-constants";

@Injectable()
export class QuestionOptionRepository {
  constructor(
    @InjectRepository(QuestionOptionMap)
    private readonly questionOptionRepo: Repository<QuestionOptionMap>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
    @InjectRepository(Question)
    private readonly questionRepo: Repository<Question>,
    @InjectRepository(Option)
    private readonly optionRepo: Repository<Option>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>
  ) {}

  async createMappings(dto: CreateQuestionOptionDto) {
    try {
      const { questionId, optionIds, createdBy, updatedBy } = dto;

      // Validate question existence
      const question = await this.commonDataService.findOneById(
        this.questionRepo,
        questionId,
        true
      );
      if (!question) {
        this.logger.error(questionConstMessages.QUESTION_NOT_FOUND_ID(questionId));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_NOTFOUND,
          null,
          null,
          questionId.toString()
        );
      }

      // Validate user existence
      const [creator, updater] = await Promise.all([
        this.commonDataService.findOneById(this.userRepo, createdBy, false),
        this.commonDataService.findOneById(this.userRepo, updatedBy, false),
      ]);

      if (!creator) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          createdBy.toString()
        );
      }
      if (!updater) {
        this.logger.error(`Updater not found with ID: ${updatedBy}`);
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      // Validate and map each option ID
      const mappings: QuestionOptionMap[] = [];
      for (const optionId of optionIds) {
        const option = await this.commonDataService.findOneById(
          this.optionRepo,
          optionId,
          true
        );
        if (!option) {
          this.logger.error(optionConstMessages.OPTION_NOT_FOUND_ID(optionId));
          throw new InifniNotFoundException(
            ERROR_CODES.OPTION_NOTFOUND,
            null,
            null,
            optionId.toString()
          );
        }

        // Check for duplicate mapping
        const existingMapping = await this.questionOptionRepo.findOne({
          where: { question: { id: questionId }, option: { id: optionId } },
        });
        if (existingMapping) {
          this.logger.error(
            questionOptionConstMessages.MAPPING_QUES_OPT_ALREADY_EXISTS(questionId, optionId)
          );
          throw new InifniBadRequestException(
            ERROR_CODES.QUESTION_OPTION_DUPLICATE_BADREQUEST,
            null,
            null,
            questionId.toString(),optionId.toString()
          );
        }

        // Create the mapping
        const mapping = new QuestionOptionMap({
          question,
          option, // Fully populated Option entity
          createdBy: creator,
          updatedBy: updater,
        });
        mappings.push(mapping);
      }

      // Save all mappings in bulk
      return await this.questionOptionRepo.save(mappings);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_CREATION_FAILED, error);
    }
  }

  async findAllMappings(limit: number, offset: number, searchText: string) {
    try {
      const relations = ["question", "option", "createdBy", "updatedBy"];
      const whereClause: any = { deletedAt: null };
      if (searchText) {
        whereClause.label = ILike(`%${searchText}%`);
      }

      const data = await this.commonDataService.get(
        this.questionOptionRepo,
        undefined,
        whereClause,
        limit,
        offset,
        { id: "ASC" },
        undefined,
        relations
      );

      const totalCount = await this.questionOptionRepo.count({
        where: whereClause,
      });

      return {
        data,
        pagination: {
          totalPages: Math.ceil(totalCount / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: totalCount,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_FIND_ALL_FAILED, error);
    }
  }
  async findMappingById(id: number) {
    try {
      const mapping = await this.commonDataService.findOneById(
        this.questionOptionRepo,
        id,
        true,
        ["question", "option", "createdBy", "updatedBy"]
      );

      if (!mapping) {
        this.logger.error(questionOptionConstMessages.MAPPING_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_OPTION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      return mapping;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_FIND_BY_ID_FAILED, error);
    }
  }

  async updateMapping(id: number, dto: UpdateQuestionOptionDto) {
    try {
      // Fetch the existing mapping
      const mapping = await this.commonDataService.findOneById(
        this.questionOptionRepo,
        id,
        true,
        ['question', 'option', 'createdBy', 'updatedBy']
      );
  
      if (!mapping) {
        this.logger.error(questionOptionConstMessages.MAPPING_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_OPTION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      // Fetch the question entity
      const question = await this.commonDataService.findOneById(
        this.questionRepo,
        dto.questionId || mapping.question.id,
        true
      );
      if (!question) {
        this.logger.error(questionConstMessages.QUESTION_NOT_FOUND_ID(dto.questionId || mapping.question.id));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_NOTFOUND,
          null,
          null,
          (dto.questionId || mapping.question.id).toString()
        );
      }
  
      // Fetch the updated option entity
      const option = await this.commonDataService.findOneById(
        this.optionRepo,
        dto.optionId,
        true
      );
  
      if (!option) {
        this.logger.error(optionConstMessages.OPTION_NOT_FOUND_ID(dto.optionId));
        throw new InifniNotFoundException(
          ERROR_CODES.OPTION_NOTFOUND,
          null,
          null,
          dto.optionId.toString()
        );
      }
  
      // Fetch the updater user entity
      const updater = await this.commonDataService.findOneById(
        this.userRepo,
        dto.updatedBy,
        false
      );
  
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(dto.updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          dto.updatedBy.toString()
        );
      }
  
      // Update the mapping
      mapping.option = option; // Assign the fully populated Option entity
      mapping.updatedBy = updater; // Assign the fully populated User entity
      mapping.updatedAt = new Date(); // Update the timestamp
  
      // Save the updated mapping
      return await this.commonDataService.save(this.questionOptionRepo, mapping);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_UPDATE_FAILED, error);
    }
  }

  async softDeleteMapping(id: number, user: User) {
    try {
      const mapping = await this.commonDataService.findOneById(
        this.questionOptionRepo,
        id,
        true
      );

      if (!mapping) {
        this.logger.error(questionOptionConstMessages.MAPPING_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_OPTION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      mapping.updatedBy = user;
      mapping.deletedAt = new Date();
      return await this.commonDataService.save(
        this.questionOptionRepo,
        mapping
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_DELETE_FAILED, error);
    }
  }
}
