import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Put,
  Delete,
  HttpStatus,
  ParseIntPipe,
  ValidationPipe,
  UsePipes,
  HttpCode,
  UseGuards,
  Res,
  Req,
  Query,
} from '@nestjs/common';
import { Response } from 'express';
import { QuestionOptionService } from './question-option.service';
import { CreateQuestionOptionDto } from './dto/create-question-option.dto';
import { UpdateQuestionOptionDto } from './dto/update-question-option.dto';
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiBearerAuth, ApiSecurity, ApiQuery } from '@nestjs/swagger';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { paginationConstants, questionOptionConstMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('question-option')
@Controller('question-option')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class QuestionOptionController {
  constructor(
    private readonly service: QuestionOptionService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: questionOptionConstMessages.CREATE_NEW_MAPPING })
  @ApiOperation({ summary: questionOptionConstMessages.CREATE_NEW_MAPPING })
  @ApiBody({ type: CreateQuestionOptionDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() dto: CreateQuestionOptionDto, @Req() req: any, @Res() res: Response) {
    this.logger.log(questionOptionConstMessages.CREATE_MAPPING_REQUEST_RECEIVED, dto);
    try {
      const userId = req.user?.id || dto.createdBy;
      dto.createdBy = userId;
      dto.updatedBy = userId;
      const data = await this.service.create(dto);
      await this.responseService.success(res, questionOptionConstMessages.MAPPING_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: questionOptionConstMessages.GET_ALL_MAPPINGS })
  @ApiOperation({ summary: questionOptionConstMessages.GET_ALL_MAPPINGS })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Res() res: Response) {
    this.logger.log(questionOptionConstMessages.FIND_ALL_MAPPINGS_REQUEST_RECEIVED, { limit, offset, searchText });
    try {
      const data = await this.service.findAll(limit, offset, searchText);
      await this.responseService.success(res, questionOptionConstMessages.MAPPINGS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: questionOptionConstMessages.GET_MAPPING_BY_ID })
  @ApiParam({ name: 'id', description: questionOptionConstMessages.MAPPING_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(questionOptionConstMessages.FIND_ONE_MAPPING_REQUEST_RECEIVED, { id });
    try {
      const data = await this.service.findOne(id);
      await this.responseService.success(res, questionOptionConstMessages.MAPPING_RETRIEVED, data);
      await this.responseService.success(res, questionOptionConstMessages.MAPPING_RETRIEVED, data);
    } catch (error) {
      if (error.status === HttpStatus.NOT_FOUND) {
        return this.errorHandler.notFound(res, error.response?.message || questionOptionConstMessages.MAPPING_NOT_FOUND);
      }
      handleControllerError(res,error);
    }
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: questionOptionConstMessages.UPDATE_QUESTION_OPTION_BY_ID })
  @ApiParam({ name: 'id', description: questionOptionConstMessages.MAPPING_ID, type: Number })
  @ApiOperation({ summary: questionOptionConstMessages.UPDATE_QUESTION_OPTION_BY_ID })
  @ApiParam({ name: 'id', description: questionOptionConstMessages.MAPPING_ID, type: Number })
  @ApiBody({ type: UpdateQuestionOptionDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateQuestionOptionDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(questionOptionConstMessages.UPDATE_MAPPING_REQUEST_RECEIVED, { id, dto });
    try {
      const userId = req.user?.id || dto.updatedBy;
      dto.updatedBy = userId;
      const data = await this.service.update(id, dto);
      await this.responseService.success(res, questionOptionConstMessages.MAPPING_UPDATED, data);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: questionOptionConstMessages.DELETE_QUESTION_OPTION_BY_ID })
  @ApiParam({ name: 'id', description: questionOptionConstMessages.MAPPING_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    this.logger.log(questionOptionConstMessages.DELETE_MAPPING_REQUEST_RECEIVED, { id });
    try {
      const user = req.user
      if (!user) {
        return this.errorHandler.unauthorized(res);
      }
      await this.service.remove(id, user);
      await this.responseService.success(res, questionOptionConstMessages.MAPPING_DELETED);
    } catch (error) {
      handleControllerError(res,error);
    }
  }
}