import { Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository, In, ILike, EntityManager, IsNull } from "typeorm";
import { Address, Program, ProgramType, User } from "src/common/entities";
import { CreateProgramDto } from "./dto/create-program.dto";
import { UpdateProgramDto } from "./dto/update-program.dto";
import { CommonDataService } from "src/common/services/commonData.service";
import { AppLoggerService } from "src/common/services/logger.service";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import InifniBadRequestException from "src/common/exceptions/infini-badrequest-exception";
import { programConstMessages, userConstMessages } from "src/common/constants/strings-constants";
import { RegistrationWindowStatusEnum } from "src/common/enum/registration-window-status.enum";

@Injectable()
export class ProgramRepository {
  constructor(
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,
    @InjectRepository(ProgramType)
    private readonly programTypeRepo: Repository<ProgramType>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService
  ) {}

  async createProgram(createDto: CreateProgramDto, manager?: EntityManager): Promise<Program> {
    try {
      const { typeId, name, code, createdBy, updatedBy } = createDto;

      // Get the appropriate repositories (transaction-aware or regular)
      const programTypeRepo = manager ? manager.getRepository(ProgramType) : this.programTypeRepo;
      const userRepo = manager ? manager.getRepository(User) : this.userRepo;
      const programRepo = manager ? manager.getRepository(Program) : this.programRepo;

        const programType = await programTypeRepo.findOne({
          where: { id: typeId },
        });
        if (!programType) {
          this.logger.error(programConstMessages.PROGRAM_TYPE_NOT_FOUND_ID(typeId));
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
            null,
            null,
            typeId.toString()
          );
        }


      // Validate creator and updater
      const [creator, updater] = await Promise.all([
        userRepo.findOne({ where: { id: createdBy } }),
        userRepo.findOne({ where: { id: updatedBy } }),
      ]);

      if (!creator) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_CREATOR_NOTFOUND,
          null,
          null,
          createdBy.toString()
        );
      }
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }
      
      // Ensure venueAddress is an Address entity, not a DTO
      const { venueAddress, groupedPrograms, ...restCreateDto } = createDto;

      // Determine if this program should be marked as grouped based on the
      // payload. If groupedPrograms is provided and has length > 0, or if a
      // groupId is present (child programs), isGroupedProgram will be true.
      const isGroupedProgram =
        (Array.isArray(groupedPrograms) && groupedPrograms.length > 0) ||
        !!(restCreateDto as any).groupId;
        

      const program = new Program({
        ...restCreateDto,
        creator,
        updater,
        isGroupedProgram,
        venueAddress: venueAddress as Address,
      });

      if (manager) {
        return await manager.save(Program, program);
      } else {
        return this.commonDataService.save(this.programRepo, program);
      }
    } catch (error) {
      this.logger.error(programConstMessages.ERROR_OCCURRED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
    }
  }

  async findAllPrograms(
    limit: number,
    offset: number,
    searchText: string,
    typeId: number[],
    parsedFilters: Record<string, any>,
    manager?: EntityManager
  ) {
    try {
      const programRepo = manager ? manager.getRepository(Program) : this.programRepo;

      // 1. Query only primary programs with limit/offset
      const primaryQueryBuilder = programRepo.createQueryBuilder('program')
        .leftJoinAndSelect('program.type', 'type')
        .leftJoinAndSelect('program.sessions', 'sessions')
        .leftJoinAndSelect('program.programQuestionMaps', 'programQuestionMaps')
        .where('program.deletedAt IS NULL')
        .andWhere('(program.isPrimaryProgram = :isPrimary OR program.groupId IS NULL)', { isPrimary: true });


      if (typeId?.length) {
        primaryQueryBuilder.andWhere('program.type IN (:...typeId)', { typeId });
      }
      if (
        parsedFilters?.status !== undefined &&
        parsedFilters?.status !== null &&
        parsedFilters?.status !== ''
      ) {
        if (Array.isArray(parsedFilters.status)) {
          primaryQueryBuilder.andWhere('program.status IN (:...status)', {
            status: parsedFilters.status,
          });
        } else {
          primaryQueryBuilder.andWhere('program.status = :status', {
            status: parsedFilters.status,
          });
        }
      }
      if (parsedFilters?.programKey) {
        if (Array.isArray(parsedFilters.programKey)) {
          primaryQueryBuilder.andWhere('type.key IN (:...programKeys)', {
            programKeys: parsedFilters.programKey,
          });
        } else {
          primaryQueryBuilder.andWhere('type.key = :programKey', {
            programKey: parsedFilters.programKey,
          });
        }
      }
      if (parsedFilters?.mode_of_operation) {
        primaryQueryBuilder.andWhere('program.modeOfOperation = :modeOfOperation',
          { modeOfOperation: parsedFilters.mode_of_operation });
      }
      if (searchText) {
        primaryQueryBuilder.andWhere('LOWER(program.name) LIKE LOWER(:searchText)',
          { searchText: `%${searchText}%` });
      }

      const total = await primaryQueryBuilder.getCount();

      const primaryPrograms = await primaryQueryBuilder
        .orderBy('program.id', 'ASC')
        .take(limit)
        .skip(offset)
        .getMany();

      // 2. For each primary program, fetch its grouped programs (children)
      let allPrograms: Program[] = [];
      for (const primary of primaryPrograms) {
        allPrograms.push(primary);
        if (primary.groupId) {
          // Fetch grouped programs (children) for this primary program
          const children = await this.findGroupedPrograms(primary.groupId, manager);
          // Only add children (not the primary itself)
          allPrograms.push(...children.filter(c => !c.isPrimaryProgram));
        }
      }

      const programsWithStatus = allPrograms.map((p) => ({
        ...p,
        registrationWindowStatus: this.computeRegistrationWindowStatus(p),
      }));

      return {
        data: programsWithStatus,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: limit,
          totalRecords: total,
          numberOfRecords: primaryPrograms.length,
        },
      };
    } catch (error) {
      this.logger.error(programConstMessages.ERROR_OCCURRED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_GET_FAILED, error);
    }
  }

  async findOneById(id: number, manager?: EntityManager): Promise<Program | null> {
  try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      const qb = repo.createQueryBuilder('program')
        .leftJoinAndSelect('program.type', 'type')
        .leftJoinAndSelect('program.sessions', 'sessions')
        .leftJoinAndSelect('program.programQuestionMaps', 'programQuestionMaps')
        .leftJoinAndSelect('programQuestionMaps.question', 'question')
        .leftJoinAndSelect('programQuestionMaps.programQuestionFormSection', 'programQuestionFormSection')
        .leftJoinAndSelect('question.questionOptionMaps', 'questionOptionMaps')
        .leftJoinAndSelect('questionOptionMaps.option', 'option')
        .leftJoinAndSelect('question.formSection', 'formSection')
        .where('program.id = :id', { id })
        .andWhere('program.deletedAt IS NULL')
        .orderBy('option.id', 'ASC');

      const program = await qb.getOne();

      if (!program) {
        this.logger.error(programConstMessages.PROGRAM_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      const programWithStatus = {
        ...program,
        registrationWindowStatus: this.computeRegistrationWindowStatus(program),
      };
      
      return programWithStatus;
    } catch (error) {
      this.logger.error(programConstMessages.ERROR_OCCURRED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds all programs in a group by group ID (only child programs)
   */
  async findGroupedPrograms(groupId: string, manager?: EntityManager): Promise<Program[]> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        return await repo.find({
          where: { groupId, deletedAt: IsNull() },
          relations: [
            "type",
            "sessions",
            "programQuestionMaps",
            "programQuestionMaps.question",
            "programQuestionMaps.question.questionOptionMaps",
            "programQuestionMaps.question.questionOptionMaps.option",
            "programQuestionMaps.question.formSection"
          ],
          order: {
            groupDisplayOrder: "ASC"
          }
        });
      } else {
        return await this.commonDataService.findByData(
          repo,
          { groupId, deletedAt: IsNull() },
          ["type","sessions", "programQuestionMaps", "programQuestionMaps.question", "programQuestionMaps.question.questionOptionMaps", "programQuestionMaps.question.questionOptionMaps.option", "programQuestionMaps.question.formSection"],
          { groupDisplayOrder: "ASC" }
        );
      }
    } catch (error) {
      this.logger.error(programConstMessages.ERROR_OCCURRED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds all programs in a group (both primary and children)
   */
  async findAllGroupPrograms(groupId: string, manager?: EntityManager): Promise<Program[]> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        return await repo.find({
          where: { groupId, deletedAt: IsNull() },
          relations: [
            "type",
            "sessions",
            "programQuestionMaps",
            "programQuestionMaps.question",
            "programQuestionMaps.question.questionOptionMaps",
            "programQuestionMaps.question.questionOptionMaps.option",
            "programQuestionMaps.question.formSection"
          ],
          order: {
            isPrimaryProgram: "DESC", // Primary program first
            groupDisplayOrder: "ASC"
          }
        });
      } else {
        return await this.commonDataService.findByData(
          repo,
          { groupId, deletedAt: IsNull() },
          ["type","sessions", "programQuestionMaps", "programQuestionMaps.question", "programQuestionMaps.question.questionOptionMaps", "programQuestionMaps.question.questionOptionMaps.option", "programQuestionMaps.question.formSection"],
          { 
            isPrimaryProgram: "DESC", // Primary program first
            groupDisplayOrder: "ASC" 
          }
        );
      }
    } catch (error) {
      this.logger.error(programConstMessages.ERROR_OCCURRED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  async updateProgram(id: number, updateDto: UpdateProgramDto, manager?: EntityManager): Promise<Program> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      const programTypeRepo = manager ? manager.getRepository(ProgramType) : this.programTypeRepo;
      const userRepo = manager ? manager.getRepository(User) : this.userRepo;
      
      const program = await this.findOneById(id, manager);
      if (!program) {
        this.logger.error(programConstMessages.PROGRAM_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      const { typeId, name, code, updatedBy } = updateDto;

      // Validate program type
      if (typeId) {
        const programType = await programTypeRepo.findOne({
          where: { id: typeId },
        });
        if (!programType) {
          this.logger.error(programConstMessages.PROGRAM_TYPE_NOT_FOUND_ID(typeId));
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
            null,
            null,
            typeId.toString()
          );
        }
        program.type = programType;
      }

      // Validate updater
      const updater = await userRepo.findOne({ where: { id: updatedBy } });
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      // Check for duplicate program by name
      // if (name) {
      //   const existingProgramByName = await repo.findOne({
      //     where: { name, deletedAt: IsNull() },
      //   });
      //   if (existingProgramByName && existingProgramByName.id !== id) {
      //     this.logger.error(programConstMessages.PROGRAM_BAD_REQUEST, name);
      //     throw new InifniBadRequestException(
      //       ERROR_CODES.PROGRAM_DUPLICATE_BADREQUEST,
      //       null,
      //       null,
      //       name
      //     );
      //   }
      // }

      // Check for duplicate program by code
      // if (code) {
      //   const existingProgramByCode = await repo.findOne({
      //     where: { code, deletedAt: IsNull() },
      //   });
      //   if (existingProgramByCode && existingProgramByCode.id !== id) {
      //     this.logger.error(`Duplicate program found with code: ${code}`);
      //     throw new InifniBadRequestException(
      //       ERROR_CODES.PROGRAM_DUPLICATE_BADREQUEST,
      //       null,
      //       null,
      //       code
      //     );
      //   }
      // }

      Object.assign(program, updateDto);
      program.updater = updater;
      
      if (manager) {
        return await manager.save(Program, program);
      } else {
        return this.commonDataService.save(this.programRepo, program);
      }
    } catch (error) {
      this.logger.error('Error updating program:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
    }
  }

  async updateById(id: number, updateDto: Partial<Program>, manager?: EntityManager): Promise<Program> {
    try {
      const program = await this.findOneById(id, manager);
      if (!program) {
        this.logger.error(programConstMessages.PROGRAM_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      Object.assign(program, updateDto);
      return await this.commonDataService.save(this.programRepo, program);
    } catch (error) {
      this.logger.error('Error updating program:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
    }
  }

  async softDeleteProgram(id: number, user: User, manager?: EntityManager): Promise<void> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      const program = await this.findOneById(id, manager);
      
      if (!program) {
        this.logger.error(programConstMessages.PROGRAM_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      program.isActive = false;
      program.updatedBy = user.id;
      program.deletedAt = new Date();
      program.updater = user;
      
      if (manager) {
        await manager.save(Program, program);
      } else {
        await this.commonDataService.save(this.programRepo, program);
      }
      
      this.logger.log(`Successfully soft deleted program ${id}`);
    } catch (error) {
      this.logger.error('Error soft deleting program:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_DELETE_FAILED, error);
    }
  }

  /**
   * Soft deletes only child programs in a group (preserves primary program)
   */
  async softDeleteChildPrograms(groupId: string, manager?: EntityManager): Promise<void> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        // Update only child programs (isPrimaryProgram = false)
        await repo.update(
          { 
            groupId, 
            isPrimaryProgram: false,
            deletedAt: IsNull() 
          },
          { 
            deletedAt: new Date(),
            isActive: false 
          }
        );
      } else {
        // For non-transaction context
        const childPrograms = await this.commonDataService.findByData(
          this.programRepo,
          { groupId, isPrimaryProgram: false, deletedAt: IsNull() }
        );

        for (const program of childPrograms) {
          program.deletedAt = new Date();
          program.isActive = false;
          await this.commonDataService.save(this.programRepo, program);
        }
      }
      
      this.logger.log(`Successfully soft deleted child programs for group ${groupId}`);
    } catch (error) {
      this.logger.error('Error soft deleting child programs:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_DELETE_FAILED, error);
    }
  }

  /**
   * Soft deletes all programs in a group (including primary)
   */
  async softDeleteAllGroupPrograms(groupId: string, user: User, manager?: EntityManager): Promise<void> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        // Update all programs in the group
        await repo.update(
          { 
            groupId,
            deletedAt: IsNull() 
          },
          { 
            deletedAt: new Date(),
            isActive: false,
            updatedBy: user.id
          }
        );
      } else {
        // For non-transaction context
        const groupPrograms = await this.commonDataService.findByData(
          this.programRepo,
          { groupId, deletedAt: IsNull() }
        );

        for (const program of groupPrograms) {
          program.deletedAt = new Date();
          program.isActive = false;
          program.updatedBy = user.id;
          program.updater = user;
          await this.commonDataService.save(this.programRepo, program);
        }
      }
      
      this.logger.log(`Successfully soft deleted all programs for group ${groupId}`);
    } catch (error) {
      this.logger.error('Error soft deleting group programs:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_DELETE_FAILED, error);
    }
  }

  /**
   * Finds programs by primary program ID
   */
  async findProgramsByPrimaryId(primaryProgramId: number, manager?: EntityManager): Promise<Program[]> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        return await repo.find({
          where: { primaryProgramId, deletedAt: IsNull() },
          relations: [
            "type",
            "sessions",
            "programQuestionMaps",
            "programQuestionMaps.question",
            "programQuestionMaps.question.questionOptionMaps",
            "programQuestionMaps.question.questionOptionMaps.option",
            "programQuestionMaps.question.formSection"
          ],
          order: {
            groupDisplayOrder: "ASC"
          }
        });
      } else {
        return await this.commonDataService.findByData(
          repo,
          { primaryProgramId, deletedAt: IsNull() },
          ["type","sessions", "programQuestionMaps", "programQuestionMaps.question", "programQuestionMaps.question.questionOptionMaps", "programQuestionMaps.question.questionOptionMaps.option", "programQuestionMaps.question.formSection"],
          { groupDisplayOrder: "ASC" }
        );
      }
    } catch (error) {
      this.logger.error(programConstMessages.ERROR_OCCURRED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds primary program by group ID
   */
  async findPrimaryProgramByGroupId(groupId: string, manager?: EntityManager): Promise<Program | null> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        return await repo.findOne({
          where: { 
            groupId, 
            isPrimaryProgram: true, 
            deletedAt: IsNull() 
          },
          relations: [
            "type",
            "sessions",
            "programQuestionMaps",
            "programQuestionMaps.question",
            "programQuestionMaps.question.questionOptionMaps",
            "programQuestionMaps.question.questionOptionMaps.option",
            "programQuestionMaps.question.formSection"
          ]
        });
      } else {
        const results = await this.commonDataService.findByData(
          repo,
          { groupId, isPrimaryProgram: true, deletedAt: IsNull() },
          ["type","sessions", "programQuestionMaps", "programQuestionMaps.question", "programQuestionMaps.question.questionOptionMaps", "programQuestionMaps.question.questionOptionMaps.option", "programQuestionMaps.question.formSection"]
        );
        return results.length > 0 ? results[0] : null;
      }
    } catch (error) {
      this.logger.error('Error finding primary program by group ID:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Checks if a program type supports grouped programs
   */
  async isProgramTypeGrouped(typeId: number, manager?: EntityManager): Promise<boolean> {
    try {
      const programTypeRepo = manager ? manager.getRepository(ProgramType) : this.programTypeRepo;
      
      const programType = await programTypeRepo.findOne({
        where: { id: typeId },
        select: ['isGroupedProgram']
      });
      
      return programType?.isGroupedProgram || false;
    } catch (error) {
      this.logger.error('Error checking if program type is grouped:', error);
      return false;
    }
  }

  /**
   * Gets count of programs in a group
   */
  async getGroupedProgramsCount(groupId: string, manager?: EntityManager): Promise<number> {
    try {
      const repo = manager ? manager.getRepository(Program) : this.programRepo;
      
      if (manager) {
        return await repo.count({
          where: { groupId, deletedAt: IsNull() }
        });
      } else {
        const programs = await this.commonDataService.findByData(
          repo,
          { groupId, deletedAt: IsNull() }
        );
        return programs.length;
      }
    } catch (error) {
      this.logger.error('Error getting grouped programs count:', error);
      return 0;
    }
  }

  /**
   * Validates if a program type supports grouped programs
   */
  async validateGroupedProgramSupport(typeId: number, manager?: EntityManager): Promise<boolean> {
    try {
      const programTypeRepo = manager ? manager.getRepository(ProgramType) : this.programTypeRepo;
      
      const programType = await programTypeRepo.findOne({
        where: { id: typeId, deletedAt: IsNull() },
        select: ['isGroupedProgram']
      });
      
      if (!programType) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
          null,
          null,
          typeId.toString()
        );
      }
      
      return programType.isGroupedProgram || false;
    } catch (error) {
      this.logger.error('Error validating grouped program support:', error);
      return false;
    }
  }

  /**
   * Updates the softDeleteGroupedPrograms method to only delete child programs
   */
  async softDeleteGroupedPrograms(groupId: string, manager?: EntityManager): Promise<void> {
    // This method now delegates to softDeleteChildPrograms for clarity
    await this.softDeleteChildPrograms(groupId, manager);
  }

  /**
   * Checks if a program is part of a group and returns group information
   */
  async getGroupInfo(programId: number, manager?: EntityManager): Promise<{
    isGrouped: boolean;
    isPrimary: boolean;
    groupId: string | null;
    primaryProgramId: number | null;
  }> {
    try {
      const program = await this.findOneById(programId, manager);
      
      if (!program) {
        return {
          isGrouped: false,
          isPrimary: false,
          groupId: null,
          primaryProgramId: null
        };
      }

      return {
        isGrouped: !!program.groupId,
        isPrimary: program.isPrimaryProgram || false,
        groupId: program.groupId,
        primaryProgramId: program.primaryProgramId
      };
    } catch (error) {
      this.logger.error('Error getting group info:', error);
      return {
        isGrouped: false,
        isPrimary: false,
        groupId: null,
        primaryProgramId: null
      };
    }
  }
  private computeRegistrationWindowStatus(
    program: { registrationStartsAt?: Date | null; registrationEndsAt?: Date | null },
  ): RegistrationWindowStatusEnum {
    // Always use UTC for all date comparisons
    const nowUtc = new Date();
    if (program.registrationStartsAt && nowUtc < new Date(program.registrationStartsAt)) {
      return RegistrationWindowStatusEnum.NOT_OPENED;
    }
    if (program.registrationEndsAt && nowUtc > new Date(program.registrationEndsAt)) {
      return RegistrationWindowStatusEnum.CLOSED;
    }
    return RegistrationWindowStatusEnum.OPENED;
  }
}