import { 
  Controller, 
  Get, 
  Post, 
  Body, 
  Param, 
  Put, 
  Delete, 
  HttpStatus,
  ParseIntPipe,
  ValidationPipe,
  UsePipes,
  HttpCode,
  Query,
  Res,
  UseGuards,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { ProgramService } from './program.service';
import { CreateProgramDto } from './dto/create-program.dto';
import { UpdateProgramDto } from './dto/update-program.dto';
import { UpdateProgramStatusDto } from './dto/update-program-status.dto';
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { programConstMessages, paginationConstants, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('program')
@Controller('program')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin','mahatria', 'viewer')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class ProgramController {
  constructor(
    private readonly programService: ProgramService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Creates a new program (single or grouped).
   * Validates the input using a validation pipe and ensures the program is created successfully.
   * For grouped programs, creates primary program and all grouped programs in a transaction.
   * @param createProgramDto - Data transfer object containing program details and optional grouped programs.
   * @returns A success message and the created program data (with grouped programs if applicable).
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programConstMessages.CREATE_NEW_PROGRAM })
  @ApiBody({ type: CreateProgramDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() createProgramDto: CreateProgramDto, @Req() req: any, @Res() res: Response) {
    this.logger.log(programConstMessages.CREATE_PROGRAM_REQUEST_RECEIVED, createProgramDto);
    try {
      const userId = req.user?.id || createProgramDto.createdBy;
      createProgramDto.createdBy = userId;
      createProgramDto.updatedBy = userId;
      
      const data = await this.programService.create(createProgramDto);
      
      // Format response based on whether it's grouped or single program
      let respData;
      if ('groupedPrograms' in data && data.primaryProgram) {
        // Grouped program response
        respData = {
          primaryProgram: {
            id: data.primaryProgram.id,
            name: data.primaryProgram.name,
            code: data.primaryProgram.code,
            typeId: data.primaryProgram.typeId,
            status: data.primaryProgram.status,
            groupId: data.primaryProgram.groupId,
            isPrimaryProgram: data.primaryProgram.isPrimaryProgram,
          },
          groupedPrograms: data.groupedPrograms.map(program => ({
            id: program.id,
            name: program.name,
            code: program.code,
            typeId: program.typeId,
            status: program.status,
            groupId: program.groupId,
            groupDisplayOrder: program.groupDisplayOrder,
            primaryProgramId: program.primaryProgramId,
          })),
          totalCreated: data.totalCreated,
        };
      } else {
        // Single program response
        respData = {
          id: 'id' in data ? data.id : undefined,
          name: 'name' in data ? data.name : undefined,
          code: 'code' in data ? data.code : undefined,
          typeId: 'typeId' in data ? data.typeId : undefined,
          status: 'status' in data ? data.status : undefined,
        };
      }
      
      this.logger.log(programConstMessages.PROGRAM_CREATED, respData);
      await this.responseService.success(res, programConstMessages.PROGRAM_CREATED, respData, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves all programs with optional pagination and search functionality.
   * For grouped programs, returns primary programs with their grouped programs nested.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter programs by name.
   * @param typeId - Program type ID array to filter programs.
   * @param filters - Additional filters as a JSON string.
   * @returns A paginated list of programs and metadata.
   */
  @Get()
  @Roles('admin', 'viewer', 'mahatria', 'rm', 'finance_manager', 'relational_manager', 'shoba', 'operational_manger')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programConstMessages.GET_ALL_PROGRAMS })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiQuery({ name: 'typeId', type: String, required: false, description: 'Program type ID to filter programs' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: paginationConstants.FILTERS })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Query('typeId') typeId: number[] = [],
    @Query('filters') filters: string = '',
    @Res() res: Response,
  ) {
    this.logger.log(programConstMessages.FIND_ALL_PROGRAMS_REQUEST_RECEIVED, {
      limit,
      offset,
      searchText,
      typeId,
      filters,
    });
    try {
      // Parse the filters string into an object
      let parsedFilters;
      if (filters) {
        parsedFilters = decodeURIComponent(filters);
        parsedFilters = JSON.parse(parsedFilters);
      }
      let parsedTypeId;
      if (typeId && typeId.length > 0) {  
        parsedTypeId = JSON.parse(typeId.toString());
        if (!Array.isArray(parsedTypeId)) {
          return this.errorHandler.badRequest(res, 'Program type ID must be an array');
        }
      }
      const result = await this.programService.findAll(limit, offset, searchText, parsedTypeId, parsedFilters);
      this.logger.log(programConstMessages.PROGRAMS_RETRIEVED, { count: result.data.length });
      await this.responseService.success(res, programConstMessages.PROGRAMS_RETRIEVED, result);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves a single program by its ID.
   * For grouped programs, includes all grouped programs in the response.
   * @param id - ID of the program to retrieve.
   * @returns The program data if found (with grouped programs if applicable).
   */
  @Get(':id')
  @Roles('admin', 'viewer','mahatria', 'rm', 'finance_manager', 'relational_manager', 'shoba', 'operational_manger')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programConstMessages.GET_PROGRAM_BY_ID })
  @ApiParam({ name: 'id', description: programConstMessages.PROGRAM_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response, @Req() req: any) {
    this.logger.log(programConstMessages.FIND_ONE_PROGRAM_REQUEST_RECEIVED, { id });
    try {
      let data = await this.programService.findOne(id);
      this.logger.log(programConstMessages.PROGRAM_RETRIEVED, { 
        id: data.id, 
        name: 'name' in data ? data.name : undefined,
        isGrouped: !!data.isGroupedProgram 
      });
      // from data.programQuestionMaps programQuestionFormSection if id is 4 then remove it from the response if role is not in ['admin', 'mahatria','viewer']
      const userRoles = req.user?.userRoleMaps?.map(role => role.role?.name) || [];
      if (data.programQuestionMaps && data.programQuestionMaps.length > 0) {
        data.programQuestionMaps = data.programQuestionMaps.filter(map => {
          if (map.programQuestionFormSection?.key === 'FS_MAHATRIAQUESTIONS') {
            return (
              userRoles.includes('admin') ||
              userRoles.includes('mahatria') ||
              userRoles.includes('viewer')
            );
          }
          return true;
        });
        // In result data if type of the question is select then we need to get the options for that question and sort them but if option is other(to lowercase) then we need to add that option at the last in the list
        data.programQuestionMaps.forEach((map) => {
          if (map.question.type.toLowerCase() === 'select') {
            const options = map.question.questionOptionMaps;
            // Check if all option names (except 'other') are numbers
            const filtered = options.filter(opt => opt.option.name.toLowerCase() !== 'other');
            const allNumbers = filtered.length > 0 && filtered.every(opt => !isNaN(Number(opt.option.name)));
            map.question.questionOptionMaps = options.sort((a, b) => {
              const aName = a.option.name.toLowerCase();
              const bName = b.option.name.toLowerCase();
              if (aName === 'other') return 1;
              if (bName === 'other') return -1;
              if (allNumbers) {
                return Number(a.option.name) - Number(b.option.name);
              }
              return a.option.name.localeCompare(b.option.name);
            });
          }
        });
      }
      await this.responseService.success(res, programConstMessages.PROGRAM_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Updates an existing program by its ID.
   * For grouped programs, can update the primary program and replace all grouped programs.
   * @param id - ID of the program to update.
   * @param updateProgramDto - Data transfer object containing updated program details.
   * @returns A success message and the updated program data.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programConstMessages.UPDATE_PROGRAM_BY_ID })
  @ApiParam({ name: 'id', description: programConstMessages.PROGRAM_ID, type: Number })
  @ApiBody({ type: UpdateProgramDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body() updateProgramDto: UpdateProgramDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programConstMessages.UPDATE_PROGRAM_REQUEST_RECEIVED, { id, updateProgramDto });
    try {
      const userId = req.user?.id || updateProgramDto.updatedBy;
      updateProgramDto.updatedBy = userId;
      
      const data = await this.programService.update(id, updateProgramDto);
      
      // Format response based on whether it's grouped or single program
      let respData;
      if ('groupedPrograms' in data && data.primaryProgram) {
        // Grouped program response
        respData = {
          primaryProgram: {
            id: data.primaryProgram.id,
            name: data.primaryProgram.name,
            code: data.primaryProgram.code,
            typeId: data.primaryProgram.typeId,
            status: data.primaryProgram.status,
            type: {
              id: data.primaryProgram.type?.id,
              name: data.primaryProgram.type?.name,
            },
          },
          groupedPrograms: data.groupedPrograms.map(program => ({
            id: program.id,
            name: program.name,
            code: program.code,
            typeId: program.typeId,
            status: program.status,
            groupDisplayOrder: program.groupDisplayOrder,
          })),
          totalUpdated: data.totalUpdated,
          totalDeleted: data.totalDeleted || 0,
          deletedProgramIds: data.deletedProgramIds || [],
          summary: {
            operation: 'grouped_program_update',
            programsUpdated: data.totalUpdated,
            programsDeleted: data.totalDeleted || 0,
            explicitlyDeleted: data.deletedProgramIds || [],
          }
        };
      } else {
        // Single program response
        respData = {
          id: 'id' in data ? data.id : undefined,
          name: 'name' in data ? data.name : undefined,
          code: 'code' in data ? data.code : undefined,
          typeId: 'typeId' in data ? data.typeId : undefined,
          status: 'status' in data ? data.status : undefined,
          type: {
            id: 'type' in data && data.type ? data.type.id : undefined,
            name: 'type' in data && data.type ? data.type.name : undefined,
          },
          summary: {
            operation: 'single_program_update',
            programsUpdated: 1,
            programsDeleted: 0,
          }
        };
      }
      
      this.logger.log(programConstMessages.PROGRAM_UPDATED, respData);
      await this.responseService.success(res, programConstMessages.PROGRAM_UPDATED, respData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Updates only the status of a program
   */
  @Put(':id/status')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: 'Update program status' })
  @ApiParam({ name: 'id', description: programConstMessages.PROGRAM_ID, type: Number })
  @ApiBody({ type: UpdateProgramStatusDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async updateStatus(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateProgramStatusDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const userId = req.user?.id || dto.updatedBy;
      dto.updatedBy = userId;

      const data = await this.programService.updateStatus(id, dto);
      await this.responseService.success(res, programConstMessages.PROGRAM_UPDATED, data ? {id: data.id, status: data.status} : undefined);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Deletes a program by its ID.
   * For grouped programs, deletes the primary program and all associated grouped programs.
   * @param id - ID of the program to delete.
   * @returns A success message indicating the program was deleted.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programConstMessages.DELETE_PROGRAM_BY_ID })
  @ApiParam({ name: 'id', description: programConstMessages.PROGRAM_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    this.logger.log(programConstMessages.DELETE_PROGRAM_REQUEST_RECEIVED, { id });
    try {
      const user = req.user;
      if (!user) {
        return this.errorHandler.unauthorized(res);
      }
      await this.programService.remove(id, user);
      this.logger.log(programConstMessages.PROGRAM_DELETED, { id });
      await this.responseService.success(res, programConstMessages.PROGRAM_DELETED, undefined, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}