import {
  Injectable,
} from "@nestjs/common";
import { ProgramType, User } from "src/common/entities";
import { CreateProgramTypeDto } from "./dto/create-program-type.dto";
import { UpdateProgramTypeDto } from "./dto/update-program-type.dto";
import { CommonDataService } from "src/common/services/commonData.service";
import { ProgramTypeRepository } from "./program-type.repository";
import { ILike, IsNull, Repository } from "typeorm";
import {
  userConstMessages,
  programTypeConstMessages,
} from "src/common/constants/strings-constants";
import { InjectRepository } from "@nestjs/typeorm";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import InifniBadRequestException from "src/common/exceptions/infini-badrequest-exception";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { AppLoggerService } from "src/common/services/logger.service";

@Injectable()
export class ProgramTypeService {
  constructor(
    private readonly programTypeRepository: ProgramTypeRepository,
    @InjectRepository(User) private readonly userRepository: Repository<User>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Creates a new program type.
   * Validates the existence of the creator and updater users.
   * Ensures the program type is unique by its name.
   * @param createDto - Data transfer object containing program type details.
   * @returns The created program type.
   * @throws NotFoundException if the creator or updater user does not exist.
   * @throws BadRequestException if the program type already exists.
   */
  async create(createDto: CreateProgramTypeDto) {
    this.logger.log(programTypeConstMessages.CREATING_PROGRAM_TYPE(createDto));
    const { name, description, modeOfOperation, onlineType, maxSessionDurationDays, hasMultipleSessions, frequency, defaultStartTime, defaultEndTime, defaultDuration, requiresResidence, involvesTravel, hasCheckinCheckout, requiresPayment, requiresAttendanceAllSessions, allowsMinors, allowsProxyRegistration, requiresApproval, registrationLevel, isActive, noOfSession, waitlistApplicable, maxCapacity, meta, venue, limitedSeats, createdBy, updatedBy } = createDto;

    // Validate creator and updater users
    const [creator, updater] = await Promise.all([
      this.userRepository.findOne({ where: { id: createdBy } }),
      this.userRepository.findOne({ where: { id: updatedBy } }),
    ]);

    if (!creator) {
      this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_TYPE_CREATOR_NOTFOUND,
        null,
        null,
        createdBy.toString()
      );
    }
    if (!updater) {
      this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_TYPE_UPDATOR_NOTFOUND,
        null,
        null,
        updatedBy.toString()
      );
    }

    // Check if a program type with the same name already exists
    const existingProgramType = await this.programTypeRepository.findOneByName(name);
    if (existingProgramType) {
      this.logger.error(programTypeConstMessages.DUPLICATE_PROGRAM_TYPE_FOUND(name));
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_TYPE_DUPLICATE_BADREQUEST,
        null,
        null,
        name
      );
    }

    const programType = new ProgramType({
      name,
      description,
      modeOfOperation,
      onlineType,
      maxSessionDurationDays,
      hasMultipleSessions,
      frequency,
      defaultStartTime,
      defaultEndTime,
      defaultDuration,
      requiresResidence,
      involvesTravel,
      hasCheckinCheckout,
      requiresPayment,
      requiresAttendanceAllSessions,
      allowsMinors,
      allowsProxyRegistration,
      requiresApproval,
      registrationLevel,
      isActive,
      noOfSession,
      waitlistApplicable,
      maxCapacity,
      meta,
      venue,
      limitedSeats,
      createdBy: creator,
      updatedBy: updater,
    });

    this.logger.log(programTypeConstMessages.CREATING_PROGRAM_TYPE(programType));
    try {
      return await this.programTypeRepository.save(programType);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_SAVE_FAILED, error);
    }
  }

  /**
   * Retrieves all program types with optional pagination and search functionality.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text to filter program types by name.
   * @param parsedFilters - Additional filters for the query.
   * @returns A paginated list of program types and metadata.
   */
  async findAll(
    limit: number,
    offset: number,
    searchText: string,
    parsedFilters: Record<string, any>
  ) {
    this.logger.log(
      programTypeConstMessages.FINDING_ALL_PROGRAM_TYPES(limit, offset, searchText)
    );
    try {
      const whereClause: any = {};
      whereClause.deletedAt = IsNull();
      if (parsedFilters?.modeOfOperation) {
        whereClause.modeOfOperation = parsedFilters.modeOfOperation;
      }
      if (parsedFilters?.createdBy) {
        whereClause.createdBy = { id: parsedFilters.createdBy };
      }
      if (parsedFilters?.isActive !== undefined) {
        whereClause.isActive = parsedFilters.isActive;
      }
      if (searchText) {
        whereClause.name = ILike(`%${searchText}%`);
      }

      this.logger.log(
        programTypeConstMessages.RETRIEVING_PROGRAM_TYPES_WHERE_CLAUSE(whereClause)
      );
      const data = await this.commonDataService.get(
        this.programTypeRepository["programTypeRepo"],
        undefined,
        whereClause,
        limit,
        offset,
        { id: "ASC" },
        searchText ? { name: searchText } : undefined,
        [
          "programs",
          "programs.sessions",
          "programs.programQuestionMaps",
        ]
      );

      const total = await this.programTypeRepository.count(whereClause);
      const totalPages = Math.ceil(total / limit);

      return {
        data,
        pagination: {
          totalPages,
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_GET_FAILED, error);
    }
  }

  /**
   * Retrieves a single program type by its ID.
   * @param id - ID of the program type to retrieve.
   * @returns The program type data if found.
   * @throws NotFoundException if the program type does not exist.
   */
  async findOne(id: number) {
    this.logger.log(programTypeConstMessages.FINDING_PROGRAM_TYPE_BY_ID(id));
    try {
      const programType = await this.programTypeRepository.findOneById(id);
      if (!programType) {
        this.logger.error(programTypeConstMessages.PROGRAM_TYPE_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      return programType;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Updates an existing program type by its ID.
   * Validates the existence of the updater user.
   * Ensures the updated program type is unique by its name.
   * @param id - ID of the program type to update.
   * @param updateDto - Data transfer object containing updated program type details.
   * @returns The updated program type.
   * @throws NotFoundException if the program type or updater user does not exist.
   * @throws BadRequestException if a duplicate program type exists.
   */
  async update(id: number, updateDto: UpdateProgramTypeDto) {
    this.logger.log(programTypeConstMessages.UPDATING_PROGRAM_TYPE(id));
    try {
      const programType = await this.programTypeRepository.findOneById(id);
      if (!programType) {
        this.logger.error(programTypeConstMessages.PROGRAM_TYPE_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      const { name, updatedBy } = updateDto;

      const updater = await this.userRepository.findOne({
        where: { id: updatedBy },
      });
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_TYPE_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      if (name) {
        const existingProgramType = await this.programTypeRepository.findOneByName(name);
        if (existingProgramType && existingProgramType.id !== id) {
          this.logger.error(
            programTypeConstMessages.DUPLICATE_PROGRAM_TYPE_FOUND(name)
          );
          throw new InifniBadRequestException(
            ERROR_CODES.PROGRAM_TYPE_DUPLICATE_BADREQUEST,
            null,
            null,
            name
          );
        }
      }

      Object.assign(programType, updateDto, { updatedBy: updater });
      return await this.programTypeRepository.save(programType);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_SAVE_FAILED, error);
    }
  }

  /**
   * Deletes a program type by its ID.
   * @param id - ID of the program type to delete.
   * @returns The deleted program type.
   * @throws NotFoundException if the program type does not exist.
   * @throws BadRequestException if the deletion fails.
   */
  async remove(id: number, userId: User) {
    this.logger.log(programTypeConstMessages.REMOVING_PROGRAM_TYPE(id));
    try {
      const programType = await this.programTypeRepository.findOneById(id);
      if (!programType) {
        this.logger.error(programTypeConstMessages.PROGRAM_TYPE_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      programType.updatedBy = userId;
      return await this.programTypeRepository.softDelete(programType.id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_DELETE_FAILED, error);
    }
  }
}