import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Put,
  Delete,
  HttpStatus,
  ParseIntPipe,
  ValidationPipe,
  UsePipes,
  HttpCode,
  Query,
  Res,
  UseGuards,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { ProgramTypeService } from './program-type.service';
import { CreateProgramTypeDto } from './dto/create-program-type.dto';
import { UpdateProgramTypeDto } from './dto/update-program-type.dto';
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { paginationConstants, programTypeConstMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('program-type')
@Controller('program-type')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class ProgramTypeController {
  constructor(
    private readonly programTypeService: ProgramTypeService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Creates a new program type.
   * Validates the input using a validation pipe and ensures the program type is created successfully.
   * @param createProgramTypeDto - Data transfer object containing program type details.
   * @param res - Express Response object.
   * @returns A success message and the created program type data.
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programTypeConstMessages.CREATE_NEW_PROGRAM_TYPE })
  @ApiBody({ type: CreateProgramTypeDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() createProgramTypeDto: CreateProgramTypeDto, @Req() req: any, @Res() res: Response) {
    this.logger.log(programTypeConstMessages.CREATE_PROGRAM_TYPE_REQUEST_RECEIVED, createProgramTypeDto);
    try {
      const userId = req.user?.id || createProgramTypeDto.createdBy;
      createProgramTypeDto.createdBy = userId;
      createProgramTypeDto.updatedBy = userId;
      const data = await this.programTypeService.create(createProgramTypeDto);
      await this.responseService.success(res, programTypeConstMessages.PROGRAM_TYPE_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      console.log('create controller error:', error);
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves all program types with optional pagination and search functionality.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter program types by name.
   * @param filters - Additional filters as a JSON string.
   * @param res - Express Response object.
   * @returns A paginated list of program types and metadata.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programTypeConstMessages.GET_ALL_PROGRAM_TYPES })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiQuery({ name: 'filters', type: String, required: false, description: paginationConstants.FILTERS })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Query('filters') filters: string = '',
    @Res() res: Response,
  ) {
    this.logger.log(programTypeConstMessages.FIND_ALL_PROGRAM_TYPES_REQUEST_RECEIVED, { limit, offset, searchText, filters });
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const result = await this.programTypeService.findAll(limit, offset, searchText, parsedFilters);
      await this.responseService.success(res, programTypeConstMessages.PROGRAM_TYPES_RETRIEVED, result);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves a single program type by its ID.
   * @param id - ID of the program type to retrieve.
   * @param res - Express Response object.
   * @returns The program type data if found.
   */
  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programTypeConstMessages.GET_PROGRAM_TYPE_BY_ID })
  @ApiParam({ name: 'id', description: programTypeConstMessages.PROGRAM_TYPE_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(programTypeConstMessages.FIND_ONE_PROGRAM_TYPE_REQUEST_RECEIVED, { id });
    try {
      const data = await this.programTypeService.findOne(id);
      await this.responseService.success(res, programTypeConstMessages.PROGRAM_TYPE_RETRIEVED, data);
    } catch (error) {
      console.log('get by id controller error:', error);
      handleControllerError(res, error);
    }
  }

  /**
   * Updates an existing program type by its ID.
   * Validates the input using a validation pipe and ensures the program type is updated successfully.
   * @param id - ID of the program type to update.
   * @param updateProgramTypeDto - Data transfer object containing updated program type details.
   * @param res - Express Response object.
   * @returns A success message and the updated program type data.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programTypeConstMessages.UPDATE_PROGRAM_TYPE_BY_ID })
  @ApiParam({ name: 'id', description: programTypeConstMessages.PROGRAM_TYPE_ID, type: Number })
  @ApiBody({ type: UpdateProgramTypeDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body() updateProgramTypeDto: UpdateProgramTypeDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programTypeConstMessages.UPDATE_PROGRAM_TYPE_REQUEST_RECEIVED, { id, updateProgramTypeDto });
    try {
      const userId = req.user?.id || updateProgramTypeDto.updatedBy;
      updateProgramTypeDto.updatedBy = userId;
      const data = await this.programTypeService.update(id, updateProgramTypeDto);
      await this.responseService.success(res, programTypeConstMessages.PROGRAM_TYPE_UPDATED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Deletes a program type by its ID.
   * @param id - ID of the program type to delete.
   * @param res - Express Response object.
   * @returns A success message indicating the program type was deleted.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programTypeConstMessages.DELETE_PROGRAM_TYPE_BY_ID })
  @ApiParam({ name: 'id', description: programTypeConstMessages.PROGRAM_TYPE_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    this.logger.log(programTypeConstMessages.DELETE_PROGRAM_TYPE_REQUEST_RECEIVED, { id });
    try {
      const userId = req.user;
      if (!userId) {
        return this.errorHandler.unauthorized(res);
      }
      await this.programTypeService.remove(id, userId);
      await this.responseService.success(res, programTypeConstMessages.PROGRAM_TYPE_DELETED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}