import { IsNotEmpty, IsString, MaxLength, IsOptional, IsEnum, IsBoolean, IsNumber, IsInt, IsObject } from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';
import { ModeOfOperationEnum, OnlineTypeEnum, FrequencyEnum, RegistrationLevelEnum } from 'src/common/enum/program.enums';

/**
 * DTO for creating a new ProgramType.
 */
export class CreateProgramTypeDto {
  @ApiProperty({ description: 'The name of the program type', maxLength: 255 })
  @IsNotEmpty({ message: 'The name is required.' })
  @IsString({ message: 'The name must be a valid string.' })
  @MaxLength(255, { message: 'The name must not exceed 255 characters.' })
  name: string;

  @ApiProperty({ description: 'A brief description of the program type', required: false })
  @IsOptional()
  @IsString({ message: 'The description must be a valid string.' })
  description?: string;

  @ApiProperty({ description: 'Mode of operation', enum: ModeOfOperationEnum })
  @IsNotEmpty({ message: 'The mode of operation is required.' })
  @IsEnum(ModeOfOperationEnum, { message: 'The mode of operation must be a valid enum value.' })
  modeOfOperation: ModeOfOperationEnum;

  @ApiProperty({ description: 'Online type', enum: OnlineTypeEnum, required: false, default: OnlineTypeEnum.NA })
  @IsOptional()
  @IsEnum(OnlineTypeEnum, { message: 'The online type must be a valid enum value.' })
  onlineType?: OnlineTypeEnum;

  @ApiProperty({ description: 'Max session duration days', required: false, default: 8 })
  @IsOptional()
  @IsInt({ message: 'The max session duration days must be a valid integer.' })
  maxSessionDurationDays?: number;

  @ApiProperty({ description: 'Has multiple sessions', required: false, default: true })
  @IsOptional()
  @IsBoolean({ message: 'The has multiple sessions must be a boolean value.' })
  hasMultipleSessions?: boolean;

  @ApiProperty({ description: 'Frequency', enum: FrequencyEnum, required: false, default: FrequencyEnum.YEARLY })
  @IsOptional()
  @IsEnum(FrequencyEnum, { message: 'The frequency must be a valid enum value.' })
  frequency?: FrequencyEnum;

  @ApiProperty({ description: 'Default start time', required: false })
  @IsOptional()
  @IsString({ message: 'The default start time must be a valid string.' })
  defaultStartTime?: string;

  @ApiProperty({ description: 'Default end time', required: false })
  @IsOptional()
  @IsString({ message: 'The default end time must be a valid string.' })
  defaultEndTime?: string;

  @ApiProperty({ description: 'Default duration', required: false })
  @IsOptional()
  @IsString({ message: 'The default duration must be a valid string.' })
  defaultDuration?: string;

  @ApiProperty({ description: 'Requires residence', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The requires residence must be a boolean value.' })
  requiresResidence?: boolean;

  @ApiProperty({ description: 'Involves travel', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The involves travel must be a boolean value.' })
  involvesTravel?: boolean;

  @ApiProperty({ description: 'Has checkin/checkout', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The has checkin checkout must be a boolean value.' })
  hasCheckinCheckout?: boolean;

  @ApiProperty({ description: 'Requires payment', required: false, default: true })
  @IsOptional()
  @IsBoolean({ message: 'The requires payment must be a boolean value.' })
  requiresPayment?: boolean;

  @ApiProperty({ description: 'Requires attendance all sessions', required: false, default: true })
  @IsOptional()
  @IsBoolean({ message: 'The requires attendance all sessions must be a boolean value.' })
  requiresAttendanceAllSessions?: boolean;

  @ApiProperty({ description: 'Allows minors', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The allows minors must be a boolean value.' })
  allowsMinors?: boolean;

  @ApiProperty({ description: 'Allows proxy registration', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The allows proxy registration must be a boolean value.' })
  allowsProxyRegistration?: boolean;

  @ApiProperty({ description: 'Requires approval', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The requires approval must be a boolean value.' })
  requiresApproval?: boolean;

  @ApiProperty({ description: 'Registration level', enum: RegistrationLevelEnum, required: false, default: RegistrationLevelEnum.SESSION })
  @IsOptional()
  @IsEnum(RegistrationLevelEnum, { message: 'The registration level must be a valid enum value.' })
  registrationLevel?: RegistrationLevelEnum;

  @ApiProperty({ description: 'Is active', required: false, default: true })
  @IsOptional()
  @IsBoolean({ message: 'The is active must be a boolean value.' })
  isActive?: boolean;

  @ApiProperty({ description: 'Number of sessions', required: false, default: 0 })
  @IsOptional()
  @IsInt({ message: 'The number of sessions must be a valid integer.' })
  noOfSession?: number;

  @ApiProperty({ description: 'Waitlist applicable', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The waitlist applicable must be a boolean value.' })
  waitlistApplicable?: boolean;

  @ApiProperty({ description: 'Max capacity', required: false, default: 0 })
  @IsOptional()
  @IsInt({ message: 'The max capacity must be a valid integer.' })
  maxCapacity?: number;

  @ApiProperty({ description: 'Meta information (JSON)', required: false, type: Object, default: {} })
  @IsOptional()
  @IsObject({ message: 'The meta must be a valid JSON object if provided.' })
  meta?: Record<string, any>;

  @ApiProperty({ description: 'Venue', required: false })
  @IsOptional()
  @IsString({ message: 'The venue must be a valid string.' })
  venue?: string;

  @ApiProperty({ description: 'Limited seats', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The limited seats must be a boolean value.' })
  limitedSeats?: boolean;

  @ApiProperty({ description: 'Is grouped program', required: false, default: false })
  @IsOptional()
  @IsBoolean({ message: 'The is grouped program must be a boolean value.' })
  isGroupedProgram?: boolean;

  //logo_url
  @ApiProperty({ description: 'Logo URL', required: false })
  @IsOptional()
  @IsString({ message: 'The logo URL must be a valid string.' })
  logoUrl?: string;


  @ApiProperty({ description: 'Created by user ID' })
  @IsInt({ message: 'The createdBy field must be a valid integer.' })
  createdBy: number;

  @ApiProperty({ description: 'Updated by user ID' })
  @IsInt({ message: 'The updatedBy field must be a valid integer.' })
  updatedBy: number;
}