import { ApiProperty } from '@nestjs/swagger';
import {
  IsString,
  IsOptional,
  IsNumber,
  IsEnum,
  IsBoolean,
  IsDateString,
  Min,
  Max,
  ValidateNested,
} from 'class-validator';
import { Type } from 'class-transformer';
import { CreateAddressDto } from 'src/address/dto/create-address.dto';
import { TdsApplicabilityEnum } from 'src/common/enum/tds-applicability.enum';

export class CreateProgramSessionDto {
  @ApiProperty({ description: 'Session ID (for updates)', example: 123, required: false })
  @IsOptional()
  @IsNumber()
  id?: number;

  @ApiProperty({ description: 'Program ID', example: 1 })
  @IsNumber()
  programId: number;

  @ApiProperty({ description: 'Session name', example: 'Opening Session' })
  @IsString()
  name: string;

  @ApiProperty({ description: 'Session code', example: 'OS2024', required: false })
  @IsOptional()
  @IsString()
  code?: string;

  @ApiProperty({ description: 'Session number', example: 1, required: false })
  @IsOptional()
  @IsNumber()
  @Min(1)
  displayOrder?: number;

  @ApiProperty({ description: 'Start at', required: false })
  @IsOptional()
  @IsDateString()
  startsAt?: Date;

  @ApiProperty({ description: 'End at', required: false })
  @IsOptional()
  @IsDateString()
  endsAt?: Date;

  @ApiProperty({ description: 'Venue', example: 'Conference Hall A', required: false })
  @IsOptional()
  @IsString()
  venue?: string;

  @ApiProperty({
    description: 'Mode of operation',
    enum: ['online', 'offline', 'hybrid'],
    required: false,
  })
  @IsOptional()
  @IsEnum(['online', 'offline', 'hybrid'])
  modeOfOperation?: 'online' | 'offline' | 'hybrid';

  @ApiProperty({
    description: 'Meeting link',
    example: 'https://zoom.us/j/123456789',
    required: false,
  })
  @IsOptional()
  @IsString()
  meetingLink?: string;

  @ApiProperty({ description: 'Meeting ID', example: '123456789', required: false })
  @IsOptional()
  @IsString()
  meetingId?: string;

  @ApiProperty({ description: 'Meeting password', example: 'password123', required: false })
  @IsOptional()
  @IsString()
  meetingPassword?: string;

  @ApiProperty({ description: 'Total seats', example: 100, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  totalSeats?: number;

  @ApiProperty({ description: 'Waitlist trigger count', example: 10, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  waitlistTriggerCount?: number;

  @ApiProperty({ description: 'Waitlist applicable', required: false, default: null })
  @IsOptional()
  @IsBoolean()
  waitlistApplicable?: boolean;

  @ApiProperty({ description: 'Available seats', example: 90, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  availableSeats?: number;

  @ApiProperty({ description: 'Reserved seats', example: 5, required: false, default: null })
  @IsOptional()
  @IsNumber()
  @Min(0)
  reservedSeats?: number;
  @ApiProperty({ description: 'CGST rate', example: 9, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  cgst?: number;

  @ApiProperty({ description: 'SGST rate', example: 9, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  sgst?: number;

  @ApiProperty({ description: 'IGST rate', example: 18, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  igst?: number;


  @ApiProperty({ description: 'TDS percentage', example: 10, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  tdsPercent?: number;

  @ApiProperty({ enum: TdsApplicabilityEnum, required: false })
  @IsOptional()
  @IsEnum(TdsApplicabilityEnum)
  tdsApplicability?: TdsApplicabilityEnum;

  @ApiProperty({ description: 'Invoice sender name', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderName?: string;

  @ApiProperty({ description: 'Invoice sender PAN', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderPan?: string;

  @ApiProperty({ description: 'Invoice sender CIN', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderCin?: string;

  @ApiProperty({ description: 'Venue address', required: false, type: CreateAddressDto })
  @IsOptional()
  @ValidateNested()
  @Type(() => CreateAddressDto)
  venueAddress?: CreateAddressDto;

  @ApiProperty({ description: 'Invoice sender address', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderAddress?: string;


  @ApiProperty({ description: 'GST number', example: '27AAACG1234R1Z5', required: false })
  @IsOptional()
  @IsString()
  gstNumber?: string;

  @ApiProperty({
    description: 'Session status',
    enum: ['scheduled', 'active', 'completed', 'cancelled', 'postponed'],
    required: false,
    default: null,
  })
  @IsOptional()
  @IsEnum(['scheduled', 'active', 'completed', 'cancelled', 'postponed'])
  status?: 'scheduled' | 'active' | 'completed' | 'cancelled' | 'postponed';

  @ApiProperty({ description: 'Is active', required: false, default: null })
  @IsOptional()
  @IsBoolean()
  isActive?: boolean;

  @ApiProperty({ description: 'Registration starts at', required: false })
  @IsOptional()
  @IsDateString()
  registrationStartsAt?: Date;

  @ApiProperty({ description: 'Registration ends at', required: false })
  @IsOptional()
  @IsDateString()
  registrationEndsAt?: Date;

  @ApiProperty({ description: 'Check-in at', required: false })
  @IsOptional()
  @IsDateString()
  checkinAt?: Date;

  @ApiProperty({ description: 'Check-out at', required: false })
  @IsOptional()
  @IsDateString()
  checkoutAt?: Date;

  @ApiProperty({ description: 'Banner image URL', required: false })
  @IsOptional()
  @IsString()
  bannerImageUrl?: string;

  @ApiProperty({ description: 'Created by user ID', example: 1 })
  @IsOptional()
  @IsNumber()
  createdBy: number;

  @ApiProperty({ description: 'Updated by user ID', example: 1 })
  @IsNumber()
  updatedBy: number;

  @ApiProperty({ description: 'Session description', required: false })
  @IsOptional()
  @IsString()
  description?: string;

  @ApiProperty({ description: 'Meta information (JSON)', required: false, type: Object })
  @IsOptional()
  meta?: any;

  @ApiProperty({ description: 'Limited seats availability', required: false })
  @IsOptional()
  @IsBoolean()
  limitedSeats?: boolean;

  // @ApiProperty({ description: 'Is residence required', required: false, default: false })
  // @IsOptional()
  // @IsBoolean()
  // isResidenceRequired?: boolean;

  @ApiProperty({ description: 'Is travel required', required: false, default: false })
  @IsOptional()
  @IsBoolean()
  isTravelRequired?: boolean;

  @ApiProperty({ description: 'Base price', example: 199.99, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  basePrice?: number;

  @ApiProperty({ description: 'Program fee', example: 1000.0, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  programFee?: number;

  @ApiProperty({ description: 'GST percentage', example: 18.0, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  gstPercentage?: number;

  @ApiProperty({ description: 'Currency', example: 'INR', required: false })
  @IsOptional()
  @IsString()
  currency?: string;

  @ApiProperty({ description: 'Helpline number', required: false })
  @IsOptional()
  @IsString()
  helplineNumber?: string;

  @ApiProperty({ description: 'Email sender name', required: false })
  @IsOptional()
  @IsString()
  emailSenderName?: string;

  @ApiProperty({ description: 'Venue name in emails', required: false })
  @IsOptional()
  @IsString()
  venueNameInEmails?: string;

  @ApiProperty({ description: 'Launch date', required: false })
  @IsOptional()
  @IsDateString()
  launchDate?: Date;

  @ApiProperty({ description: 'Check-in-ends-at', required: false })
  @IsOptional()
  @IsDateString()
  checkinEndsAt?: Date;

  @ApiProperty({ description: 'Check-out-ends-at', required: false })
  @IsOptional()
  @IsDateString()
  checkoutEndsAt?: Date;

  //logo url
  @ApiProperty({ description: 'Logo URL', required: false })
  @IsOptional()
  @IsString()
  logoUrl?: string;
}
