import { Injectable } from '@nestjs/common';
import { ProgramRegistrationRmRatingRepository } from '../repositories/program-registration-rm-rating.repository';
import {
  ProgramRegistrationRMRatingDto,
  RmRatingDto,
  UpdateProgramRegistrationRMRatingDto,
} from '../dto/program-registration-rm-ratings.dto';
import { ProgramRegistrationRmRating } from 'src/common/entities/program-registration-rm-rating.entity';
import { ProgramRegistrationRepository } from '../program-registration.repository';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { programRegistrationMessages } from 'src/common/constants/strings-constants';

export interface RatingInput {
  ratingKey: string;
  rating: number;
}

@Injectable()
export class ProgramRegistrationRmRatingService {
  constructor(
    private readonly rmRatingRepo: ProgramRegistrationRmRatingRepository,
    private readonly programRegistrationRepository: ProgramRegistrationRepository,
  ) {}

  /**
   * Posts RM ratings and review for a specific program registration.
   *
   * @param programRegistrationId - The ID of the program registration.
   * @param rmId - The ID of the RM (Relationship Manager).
   * @param data - The ratings and review data.
   * @returns A message indicating success and the posted ratings and review.
   */
  async postRmRatingsAndReview(
    programRegistrationId: number,
    rmId: number,
    data: ProgramRegistrationRMRatingDto,
  ) {
    try {
      const { ratings, review } = data;

      const programRegistration =
        await this.programRegistrationRepository.findOne(programRegistrationId);
      if (!programRegistration) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          new InifniNotFoundException(
            `Program registration with ID ${programRegistrationId} not found.`,
          ),
        );
      }

      const rmRatings: ProgramRegistrationRmRating[] = await Promise.all(
        ratings.map((rating: RmRatingDto) => {
          return this.rmRatingRepo.create({
            programRegistrationId: programRegistrationId,
            rmId: rmId,
            ratingKey: rating.ratingKey,
            rating: rating.rating,
            createdBy: rmId,
            updatedBy: rmId,
          });
        }),
      );

      // Save all RM ratings
      await this.rmRatingRepo.save(rmRatings);

      // Save the review in the program registration
      if (review) {
        programRegistration.rmReview = review;
        await this.programRegistrationRepository.update(programRegistrationId, {
          rmReview: review,
        });
      }

      return {
        message: 'RM ratings and review posted successfully.',
        ratings: rmRatings,
        review: programRegistration.rmReview,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_FAILED, error);
    }
  }
  /**
   * Updates RM ratings and review for a specific program registration.
   *
   * @param programRegistrationId - The ID of the program registration.
   * @param rmId - The ID of the RM (Relationship Manager).
   * @param data - The updated ratings and review data.
   * @returns A message indicating success and the updated ratings and review.
   */
  async updateRmRatingAndReview(
    programRegistrationId: number,
    rmId: number,
    data: UpdateProgramRegistrationRMRatingDto,
  ) {
    try {
      const programRegistration =
        await this.programRegistrationRepository.findOne(programRegistrationId);
      if (!programRegistration) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          new InifniNotFoundException(
            `Program registration with ID ${programRegistrationId} not found.`,
          ),
        );
      }
      const ratings = await this.rmRatingRepo.findAllByProgramRegistrationId(programRegistrationId);
      if (!ratings || ratings.length === 0) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_NOT_FOUND,
          new InifniNotFoundException(
            `RM ratings for program registration ID ${programRegistrationId} not found.`,
          ),
        );
      }

      // Fetch existing RM ratings for the given programRegistrationId
      const existingRatings: ProgramRegistrationRmRating[] =
        await this.rmRatingRepo.findAllByProgramRegistrationId(programRegistrationId);

      // Update each rating by matching ratingKey

      if (!existingRatings || existingRatings.length === 0) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_NOT_FOUND,
          new InifniNotFoundException(
            `RM ratings for program registration ID ${programRegistrationId} not found.`,
          ),
        );
      }

      let updatedRatings: (ProgramRegistrationRmRating | null)[] = await Promise.all(
        data.ratings.map(async (rating: RmRatingDto) => {
          const ratingEntity = existingRatings.find((r) => r.ratingKey === rating.ratingKey);
          if (ratingEntity) {
            ratingEntity.rating = rating.rating;
            ratingEntity.updatedBy = rmId;
            return ratingEntity;
          }
          return null;
        }),
      );
      const ratingsToBeUpdated = (updatedRatings = updatedRatings.filter((r) => r !== null));

      // Update rating value(s)
      if (ratingsToBeUpdated && ratingsToBeUpdated.length > 0) {
        await this.rmRatingRepo.save(ratingsToBeUpdated);
      }

      // Optionally update the review on the associated ProgramRegistration
      if (data.review) {
        programRegistration.rmReview = data.review;
        await this.programRegistrationRepository.update(programRegistrationId, {
          rmReview: data.review,
        });
      }

      return {
        message: programRegistrationMessages.RATINGS_UPDATED,
        ratings: ratingsToBeUpdated,
        review: programRegistration.rmReview,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_FAILED, error);
    }
  }
}
