import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, EntityManager } from 'typeorm';
import { ProgramRegistrationSwap } from 'src/common/entities/program-registration-swap.entity';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';

@Injectable()
export class ProgramRegistrationSwapRepository {
  constructor(
    @InjectRepository(ProgramRegistrationSwap)
    private readonly repo: Repository<ProgramRegistrationSwap>,
  ) { }

  /**
   * Finds a ProgramRegistrationSwap entity by its ID.
   * @param id The ID of the ProgramRegistrationSwap entity to retrieve.
   * @returns The ProgramRegistrationSwap entity if found.
   */
  async findOneById(id: number) {
    try {
      const record = await this.repo.findOneBy({ id });
      if (!record) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
            id.toString()
          )
        );
      }
      return record;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Finds a ProgramRegistrationSwap entity by a specific field and value.
   * @param field The field to search by.
   * @param value The value to search for.
   * @returns The ProgramRegistrationSwap entity if found.
   */ 
  async findByField(field : keyof ProgramRegistrationSwap, value: any) {
    try {
      const record = await this.repo.findOneBy({ [field]: value });
      if (!record) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
          )
        );
      }
      return record;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Finds a ProgramRegistrationSwap entity by a specific condition.
   * @param where The condition to search for.
   * @returns The ProgramRegistrationSwap entity if found.
   */
  async findOneByWhere(
    where: Partial<Omit<ProgramRegistrationSwap, 'requestedPrograms'>>,
  ): Promise<ProgramRegistrationSwap | null> {
    try {
      return await this.repo.findOne({ where });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Finds all ProgramRegistrationSwap entities with pagination.
   * @param limit The maximum number of entities to return.
   * @param offset The number of entities to skip.
   * @returns An array of ProgramRegistrationSwap entities.
   */
  async createAndSave(data: Partial<ProgramRegistrationSwap>) {
    try {
      const entity = this.repo.create(data);
      return await this.repo.save(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Updates a ProgramRegistrationSwap entity and saves it to the database.
   * @param id The ID of the entity to update.
   * @param data The new data to update the entity with.
   * @returns The updated entity or null if not found.
   */
  async updateAndSave(
    id: number,
    data: Partial<ProgramRegistrationSwap>,
    manager?: EntityManager
  ) {
    try {
      const repo = manager ? manager.getRepository(ProgramRegistrationSwap) : this.repo;
      const entity = await repo.findOne({ where: { id } });
      if (!entity) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
            id.toString()
          )
        );
      }
      Object.assign(entity, data);
      return await repo.save(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_UPDATE_FAILED, error);
    }
  }

  /**
   * Retrieves all ProgramRegistrationSwap records.
   * @returns An array of ProgramRegistrationSwap entities.
   */
  async findAll(where?: Partial<Omit<ProgramRegistrationSwap, 'requestedPrograms'>>, limit?: number, offset?: number): Promise<ProgramRegistrationSwap[]> {
    try {
      return await this.repo.find({ where, relations: ['requestedPrograms', 'allocatedProgram','programRegistration'], take: limit, skip: offset });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Gets the count of swap requests grouped by a specific field.
   * @param field The field to group by, currently only supports 'gender'.
   * @returns An array of objects containing the count and the field value.
   */
  async getCountsByGroupOf(field: 'gender' | 'requestedPrograms', programId: number): Promise<{ count: string; field: string | number
   }[]> {
    try {
      const query = this.repo.createQueryBuilder('swap')
      .select('COUNT(swap.id)', 'count')
      .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
      .andWhere('swap.type = :type', { type: SwapType.WantsSwap })

      // To group by any field of ProgramRegistration
      if (field === 'gender') {
        query
          .leftJoin('swap.programRegistration', 'registration')
          .addSelect('registration.gender', 'field')
          .andWhere('registration.program_id = :programId', { programId })
          .groupBy('registration.gender');
      }else if (field === 'requestedPrograms') {
        query
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap } )
          .leftJoin('swap.requestedPrograms', 'requestedProgram')
          .addSelect('requestedProgram.id', 'field')
          .groupBy('requestedProgram.id');
      }
      // For other fields, we can directly group by the field in swap
      else {
        query.addSelect(`swap.${field}`, field).groupBy(`swap.${field}`);
      }
      return await query.getRawMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  async getOrgCountsByGroupOf(field: 'gender' | 'requestedPrograms', programId: number): Promise<{ count: string; field: string | number }[]> {
    try {
      const query = this.repo.createQueryBuilder('swap')
        .select('COUNT(swap.id)', 'count')
        .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
        .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
        .leftJoin('swap.programRegistration', 'registration')
        .leftJoin('registration.user', 'user')
        .andWhere('user.userType = :org', { org: UserTypeEnum.ORG });

      if (field === 'gender') {
        query
          .addSelect('registration.gender', 'field')
          .andWhere('registration.program_id = :programId', { programId })
          .groupBy('registration.gender');
      } else if (field === 'requestedPrograms') {
        query
          .leftJoin('swap.requestedPrograms', 'requestedProgram')
          .addSelect('requestedProgram.id', 'field')
          .groupBy('requestedProgram.id');
      } else {
        query.addSelect(`swap.${field}`, 'field').groupBy(`swap.${field}`);
      }

      return await query.getRawMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }
}
