import { Injectable } from '@nestjs/common';
import { Repository } from 'typeorm';
import { InjectRepository } from '@nestjs/typeorm';
import { ProgramRegistrationRmRating } from 'src/common/entities/program-registration-rm-rating.entity';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Injectable()
export class ProgramRegistrationRmRatingRepository {
  constructor(
    @InjectRepository(ProgramRegistrationRmRating)
    private readonly repository: Repository<ProgramRegistrationRmRating>,
  ) {}
  /**
   * Creates a new ProgramRegistrationRmRating entity instance.
   * @param data - Partial data to create the entity.
   * @returns The created ProgramRegistrationRmRating entity.
   */
  create(data: Partial<ProgramRegistrationRmRating>): ProgramRegistrationRmRating {
    try {
      const entity = this.repository.create({
        ...data,
      });
      return entity;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_SAVE_FAILED, error);
    }
  }

  /**
   * Saves an array of ProgramRegistrationRmRating entities to the database.
   * @param ratings - Array of ProgramRegistrationRmRating entities to save.
   * @returns The saved ProgramRegistrationRmRating entities.
   */
  async save(ratings: ProgramRegistrationRmRating[]): Promise<ProgramRegistrationRmRating[]> {
    try {
      return await this.repository.save(ratings);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Finds a ProgramRegistrationRmRating entity by its ID.
   * @param id - The ID of the ProgramRegistrationRmRating entity to find.
   * @returns The found ProgramRegistrationRmRating entity or undefined if not found.
   */
  async findOne(id: number): Promise<ProgramRegistrationRmRating | null> {
    try {
      return await this.repository.findOne({ where: { id } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_NOT_FOUND, error);
    }
  }

  /**
   * Finds all ProgramRegistrationRmRating entities by program registration ID.
   * @param programRegistrationId - The ID of the program registration to find ratings for.
   * @returns An array of ProgramRegistrationRmRating entities.
   */
  async findAllByProgramRegistrationId(
    programRegistrationId: number,
  ): Promise<ProgramRegistrationRmRating[]> {
    try {
      return await this.repository.find({ where: { programRegistrationId } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_NOT_FOUND, error);
    }
  }

  /**
   * Updates a ProgramRegistrationRmRating entity with the given data.
   * @param id - The ID of the ProgramRegistrationRmRating entity to update.
   * @param data - Partial data to update the entity.
   * @returns The updated ProgramRegistrationRmRating entity.
   */
  async update(
    id: number,
    data: Partial<ProgramRegistrationRmRating>,
  ): Promise<ProgramRegistrationRmRating | null>  {
    try {
      await this.repository.update(id, data);
      return await this.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RM_RATING_UPDATE_FAILED, error);
    }
  }
}
