import { forwardRef, Inject, Injectable } from '@nestjs/common';
import { CreateProgramRegistrationDto } from './dto/create-program-registration.dto';
import { UpdateProgramRegistrationDto } from './dto/update-program-registration.dto';
import { ProgramRegistrationRepository } from './program-registration.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { programRegistrationMessages } from 'src/common/constants/strings-constants';
import { CreateProgramRegistrationSwapDto, MakeSwapRequestDto } from './dto/program-registration-swap.dto';
import { ProgramRegistrationSwapRepository } from './repositories/program-registration-swap.repository';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { ProgramRepository } from 'src/program/program.repository';
import { UserRepository } from 'src/user/user.repository';
import { Program, ProgramRegistration, ProgramRegistrationSwap } from 'src/common/entities';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ILike } from 'typeorm';
import { RegistrationRepository } from 'src/registration/registration.repository';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';


@Injectable()
export class ProgramRegistrationService {
  constructor(
    private readonly repo: ProgramRegistrationRepository,
    private readonly logger: AppLoggerService,
    private readonly swapRepo: ProgramRegistrationSwapRepository,
    private readonly programRepo: ProgramRepository,
    private readonly userRepo: UserRepository,
    @Inject(forwardRef(() => RegistrationRepository))
    private readonly registrationRepo: RegistrationRepository,
  ) {}

  async create(dto: CreateProgramRegistrationDto) {
    this.logger.log(programRegistrationMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.repo.createEntity(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(programRegistrationMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(programRegistrationMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds program registrations by program ID and optional sub-program ID.
   * @param programId - The ID of the program to filter registrations by.
   * @param limit - The maximum number of registrations to return.
   * @param offset - The offset for pagination.
   * @param subProgramId - Optional sub-program ID to filter registrations further.
   * @returns A list of program registrations matching the criteria.
   * @throws {Error} If the retrieval fails.
   */
  async findBy(programId: number, limit: number, offset: number, subProgramId?: number, search?: string) {
  try {
    let where = {};
    if (programId) {
      where = {
        program: {
          id: programId,
        } as Program,
      };
    }
    if (subProgramId) {
      where = { ...where,
        allocatedProgram: {
          id: subProgramId,
        } as Program,
      };
    }
    if (search) {
      where = { 
        ...where,
        fullName: ILike(`%${search}%`)
      };
    }
    this.logger.log(programRegistrationMessages.FIND_BY_REQUEST_RECEIVED);
    return await this.repo.findBy(where, ['swapsRequests'], limit, offset);
  } catch (error) {
    handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Updates a program registration with the given ID.
   * @param id - The ID of the program registration to update.
   * @param dto - The data to update the program registration with.
   * @returns The updated program registration.
   * @throws {Error} If the update fails.
   */
  async update(id: number, dto: UpdateProgramRegistrationDto) {
    this.logger.log(programRegistrationMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.repo.updateEntity(id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(programRegistrationMessages.DELETE_REQUEST_RECEIVED);
    try {
      await this.repo.remove(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    }
  }

  /**
   * Requests a swap for a program registration.
   * @param data - The swap request data.
   * @param userId - The ID of the user making the request.
   * @returns The created swap request.
   */
  async requestSwap(data: CreateProgramRegistrationSwapDto, userId: number) {
    try {
      const registrationRecord = await this.repo.findOne(data.programRegistrationId);
      if (!registrationRecord || !registrationRecord.allocatedProgram) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            null,
            null,
            data.programRegistrationId.toString()
          )
        );
      }

      const existingSwapRequest = await this.swapRepo.findOneByWhere({ programRegistrationId: data.programRegistrationId, status: SwapRequestStatus.ACTIVE });
      if (existingSwapRequest && existingSwapRequest.status === SwapRequestStatus.ACTIVE) {
        this.logger.warn(programRegistrationMessages.SWAP_REQUEST_ALREADY_EXISTS, {
          programRegistrationId: data.programRegistrationId,
        });
        handleKnownErrors(
          ERROR_CODES.SWAP_REQUEST_ALREADY_EXISTS,
          new InifniBadRequestException(
            ERROR_CODES.SWAP_REQUEST_ALREADY_EXISTS,
            null,
            null,
            `${data.programRegistrationId}`
          )
        );
      }

      return await this.swapRepo.createAndSave({
        programRegistrationId: data.programRegistrationId,
        type: data.type,
        status: SwapRequestStatus.ACTIVE,
        comment : data.comment,
        createdBy: userId,
        updatedBy: userId,
        createdAt: new Date(),
        updatedAt: new Date(),
        currentProgramId : registrationRecord.allocatedProgram.id,
        requestedPrograms: data.targetPrograms.map((program) => {
          return { id: program.id } as Program;
        }),
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Updates the status of a swap request.
   * @param id - The ID of the swap request.
   * @param userId - The ID of the user updating the request.
   * @param status - The new status of the swap request.
   * @param comment - Optional comment for the swap request.
   * @returns The updated swap request.
   */
  async updateSwapRequestById(
    id: number,
    userId: number,
    data : MakeSwapRequestDto,
    swappingType: string = 'swap', 
  ) {
    const { status, comment } = data;
    this.logger.log(programRegistrationMessages.UPDATE_SWAP_REQUEST, { userId });
    try {
      const swapRequest = await this.swapRepo.findOneById(id);
      // Get the user who is accepting the swap
      const user = await this.userRepo.getUserByField('id', userId);
      
      const movingSeeker = await this.repo.findOne(swapRequest.programRegistrationId);
      // Check if the moving seeker exists and has an allocated program
      if (!movingSeeker || !movingSeeker.allocatedProgram) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            null,
            null,
            swapRequest.programRegistrationId.toString()
          )
        );
      }
      const currentSubProgram = movingSeeker.allocatedProgram;
      let movingToSubProgram;
      let outGoingSeekerSwapRequest;

      if (swapRequest && swapRequest.status === status) {
        this.logger.warn(programRegistrationMessages.SWAP_REQUEST_ALREADY_UPDATED, { id, status });
        handleKnownErrors(
          ERROR_CODES.SWAP_REQUEST_ALREADY_UPDATED,
          new InifniNotFoundException(
            ERROR_CODES.SWAP_REQUEST_ALREADY_UPDATED,
            null,
            null,
            id.toString()
          )
        );
      }
      if (status === SwapRequestStatus.CLOSED) {
        return await this.swapRepo.updateAndSave(id, { status, comment, updatedBy: userId });
      } else if (status === SwapRequestStatus.ACCEPTED) {
        if (swappingType === 'swap') {
          // Check if the outgoing seeker exists and has an allocated program
          const outGoingSeeker = await this.repo.findOne(data.outgoingSeekerRegistrationId);
          if (!outGoingSeeker || !outGoingSeeker.allocatedProgram) {
            handleKnownErrors(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND, null, null),
            );
          }
          movingToSubProgram = outGoingSeeker.allocatedProgram;
          // Outgoing seeker should be from the Sub program where moving seeker is moving to
          if ( outGoingSeeker.allocatedProgram.id === movingSeeker.allocatedProgram.id) {
            this.logger.warn(programRegistrationMessages.INVALID_REQUEST, {
              id,
              movingToSubProgramId: data.movingToSubProgramId,
              outGoingSeekerId: outGoingSeeker.id,
            });
            handleKnownErrors(
              ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
              new InifniBadRequestException(
                ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
                null,
                null,
                `${data.outgoingSeekerRegistrationId}`,
              ),
            );
          }

          outGoingSeekerSwapRequest = await this.swapRepo.findOneByWhere({
            programRegistrationId: outGoingSeeker.id,
            status: SwapRequestStatus.ACTIVE,
          });
          if (!outGoingSeekerSwapRequest) {
            outGoingSeekerSwapRequest = await this.requestSwap(
              {
                programRegistrationId: outGoingSeeker.id,
                targetPrograms: [{ id: movingSeeker.allocatedProgram.id }],
                type: SwapType.ReturnSwap,
                comment: '',
              },
              userId,
            );
          }
          // Update the outgoing seeker's registration with the moving seeker's current program (currentSubProgram)
          await this.repo.update(outGoingSeeker.id, {
            allocatedProgram: { id: currentSubProgram.id } as Program,
            updatedBy: user,
          });
          // // Increase the filled seats of the current program (currentSubProgram)
          // await this.programRepo.updateById(currentSubProgram.id, {
          //   filledSeats: currentSubProgram.filledSeats + 1,
          //   updatedBy: userId
          // });
          // Update the outgoing seeker's swap request status
          if (outGoingSeekerSwapRequest) {
            await this.swapRepo.updateAndSave(outGoingSeekerSwapRequest.id, {
              status: SwapRequestStatus.ACCEPTED,
              comment: '',
              updatedBy: userId,
              allocatedProgramId: currentSubProgram.id,
            });
          }
        } else if (swappingType === 'move') {
          movingToSubProgram = await this.programRepo.findOneById(data.movingToSubProgramId);
        }
        if (!movingToSubProgram) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_NOTFOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_NOTFOUND,
              null,
              null,
              data.movingToSubProgramId.toString(),
            ),
          );
        }
        // // Decrease the filled seats of the moving seeker's current program (movingSeeker.allocatedProgram)
        // await this.programRepo.updateById(movingSeeker.allocatedProgram.id, {
        //   filledSeats: movingSeeker.allocatedProgram.filledSeats - 1,
        //   updatedBy: userId
        // });

        // Update the moving seeker's registration with the new program (movingToSubProgram)
        await this.repo.update(movingSeeker.id, {
          allocatedProgram: { id: movingToSubProgram.id } as Program,
          updatedBy: user
        });

        // // Increase the filled seats of the target program (movingToSubProgram)
        // await this.programRepo.updateById(movingToSubProgram.id, {
        //   filledSeats: movingToSubProgram.filledSeats + 1,
        //   updatedBy: userId
        // });

        // // Decrease the filled seats of the outgoing seeker's current program (outGoingSeeker.allocatedProgram)
        // await this.programRepo.updateById(outGoingSeeker.allocatedProgram.id, {
        //   filledSeats: outGoingSeeker.allocatedProgram.filledSeats - 1,
        //   updatedBy: userId
        // });

        this.logger.log(programRegistrationMessages.SWAP_REQUEST_ACCEPTED, {
          id,
          userId,
          status,
          comment,
        });
        // Update the swap request status of Moving Seeker
        return await this.swapRepo.updateAndSave(id, {
          status: SwapRequestStatus.ACCEPTED,
          comment: '',
          updatedBy: userId,
          allocatedProgramId: movingToSubProgram.id,
        });
      } else if (status === SwapRequestStatus.ON_HOLD || status === SwapRequestStatus.REJECTED) {
        const registrationRecord = await this.repo.findOne(swapRequest.programRegistrationId);
        if (!registrationRecord || !registrationRecord.allocatedProgram) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              null,
              null,
              swapRequest.programRegistrationId.toString(),
            ),
          );
        }

        await this.repo.update(swapRequest.programRegistrationId, {
          allocatedProgram: null,
          updatedBy: user,
        });

        // Increase the filled seats of the moving seeker's current program (movingSeeker.allocatedProgram)
        await this.programRepo.updateById(movingSeeker.allocatedProgram.id, {
          filledSeats: movingSeeker.allocatedProgram.filledSeats - 1,
          updatedBy: userId,
        });

        await this.registrationRepo.updateApprovalStatus(
          swapRequest.programRegistrationId,
          status === SwapRequestStatus.ON_HOLD
            ? ApprovalStatusEnum.ON_HOLD
            : ApprovalStatusEnum.REJECTED,
          userId,
        );

        return await this.swapRepo.updateAndSave(id, {
          status: status,
          comment,
          updatedBy: userId,
        });
      } else {
        this.logger.warn(programRegistrationMessages.INVALID_SWAP_REQUEST_STATUS, { id, status });
        handleKnownErrors(
          ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
          new InifniBadRequestException(
            ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
            null,
            null,
            id.toString(),
          ),
        );
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Finds swap requests by program ID and optional sub-program ID.
   * @param programId - The ID of the program to filter swap requests by.
   * @param subProgramId - Optional sub-program ID to filter swap requests further.
   * @param limit - The maximum number of swap requests to return.
   * @param offset - The offset for pagination.
   * @param filters - Optional filters to apply to the swap requests.
   * @returns A list of swap requests matching the criteria.
   */
  async findSwapRequestsBy(
    programId: number,
    subProgramId?: number,
    limit?: number,
    offset?: number,
    filters?: { [key: string]: any },
  ): Promise<ProgramRegistration[]> {
    try {
      let where = {
         id: programId,
         swapsRequests : {
          status: SwapRequestStatus.ACTIVE,
         }
      } 
      if (subProgramId) {
        where['allocatedProgram'] = {
          id: subProgramId,
        } as Program;
      }
      if (filters && filters.gender) {
        where['gender'] = filters.gender;
      }
      if (filters && filters.requestedProgramId) {
        where.swapsRequests['requestedPrograms'] = {
          id: filters.requestedProgramId,
        } as Program;
      }
        const relations = [
        'program',
        'program.type',
        'programSession',
        'invoiceDetails',
        'paymentDetails', 
        'travelInfo',
        'travelPlans',
        'user',
        'approvals',
        'allocatedProgram',
        'allocatedSession',
        'swapsRequests',
        'swapsRequests.requestedPrograms',
        'ratings',
        'rmContactUser',
        'preferences',
        'preferences.preferredProgram',
        'preferences.preferredSession',
      ];
      const response = await this.repo.getAll(where, relations);
      return response;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Retrieves key performance indicators (KPI) for swap requests in a program.
   * @param data - The data containing program ID, filters, limit, and offset.
   * @returns An object containing swap requests
   */
  async getSwapRequestKPI(data: {
    programId: number;
    filters?: { [key: string]: any };
    limit?: number;
    offset?: number;
  }) {
    const { programId, filters, limit = 10, offset = 0 } = data;
    try {
      const swapRequests = await this.findSwapRequestsBy(
        programId,
        undefined,
        limit,
        offset,
        filters,
      );
      const maleAndFemale = await this.swapRepo.getCountsByGroupOf('gender', programId);
      const maleAndFemaleOrg = await this.swapRepo.getOrgCountsByGroupOf('gender', programId);
      const subProgramsResults = await this.swapRepo.getCountsByGroupOf('requestedPrograms', programId);
      const subProgramsOrgResults = await this.swapRepo.getOrgCountsByGroupOf('requestedPrograms', programId);
      const subPrograms = await this.registrationRepo.getSubPrograms(programId);

      const subProgramKPI = subPrograms.map((subProgram) => {
        return {
          kpiName: subProgram.name,
          count: Number(subProgramsResults.find((result) => result.field === subProgram.id)?.count || 0),
          organisationUserCount: Number(subProgramsOrgResults.find((result) => result.field === subProgram.id)?.count || 0),
          filters: {
            swapPreferredProgramId: subProgram.id,
          },
        };
      });
      const kPIs = [
        ...subProgramKPI,
        {
          kpiName: 'male',
          count: Number(maleAndFemale.find((item) => item.field === 'male')?.count || 0),
          organisationUserCount: Number(maleAndFemaleOrg.find((item) => item.field === 'male')?.count || 0),
          filters: {
            gender: 'male',
          },
        },
        {
          kpiName: 'female',
          count: Number(maleAndFemale.find((item) => item.field === 'female')?.count || 0),
          organisationUserCount:Number(maleAndFemaleOrg.find((item) => item.field === 'female')?.count || 0),
          filters: {
            gender: 'female',
          },
        },
      ];
      return {
        kpis: kPIs,
        data: swapRequests,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async getSwapRequestById(id: number) {
    try {
      const swapRequest = await this.swapRepo.findOneById(id);
      if (!swapRequest) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
            id.toString(),
          ),
        );
      }
      return swapRequest;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   *
   * @param where - Partial object to search for a swap request.
   * This can include fields like `programRegistrationId`, `status`, etc.
   * @returns
   */
  async getSwapRequestByField(where: Partial<Omit<ProgramRegistrationSwap, 'requestedPrograms'>>) {
    try {
      const swapRequest = await this.swapRepo.findOneByWhere(where);
      return swapRequest;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }
}
